﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/svc/svc_Handle.h>
#include <nn/svc/ipc/svc_SessionMessage.h>

#define ENABLE_SHOW_IPC

class NamedPortManager
{
    public:
        NamedPortManager(const char* name, int32_t maxSession)
            : m_PortName(name)
            , m_MaxSession(maxSession)
        {
            nn::Result result;
            result = nn::svc::ManageNamedPort(&m_Handle, m_PortName, 0);
            NN_ASSERT(result.IsSuccess() || result <= nn::svc::ResultNotFound());

            NN_LOG("Register named port: '%s'\n", m_PortName);
            result = nn::svc::ManageNamedPort(&m_Handle, m_PortName, m_MaxSession);
            if (result.IsFailure())
            {
                DebugResult(result);
            }
            NN_ASSERT_RESULT_SUCCESS(result);
        }

        ~NamedPortManager()
        {
            nn::Result result;
            result = nn::svc::CloseHandle(m_Handle);
            NN_ASSERT_RESULT_SUCCESS(result);

            nn::svc::Handle tmpHandle;
            result = nn::svc::ManageNamedPort(&tmpHandle, m_PortName, 0);
            NN_ASSERT_RESULT_SUCCESS(result);
            NN_ASSERT(tmpHandle == nn::svc::INVALID_HANDLE_VALUE);
        }

        const char* GetName() const
        {
            return m_PortName;
        }
        int32_t GetMaxSession() const
        {
            return m_MaxSession;
        }
        nn::svc::Handle GetHandle() const
        {
            return m_Handle;
        }

    private:
        const char* m_PortName;
        int32_t m_MaxSession;
        nn::svc::Handle m_Handle;
};

void SendCopyHandle(nn::svc::Handle handle, nn::svc::Handle session);
void ReceiveCopyHandle(nn::svc::Handle* pOut, nn::svc::Handle session);
void SendMoveHandle(nn::svc::Handle handle, nn::svc::Handle session);
void ReceiveMoveHandle(nn::svc::Handle* pOut, nn::svc::Handle session);
void SendCopyHandleFromServer(nn::svc::Handle handle, nn::svc::Handle session);
void ReceiveCopyHandleFromServer(nn::svc::Handle* pOut, nn::svc::Handle session);
void SendMoveHandleFromServer(nn::svc::Handle handle, nn::svc::Handle session);
void ReceiveMoveHandleFromServer(nn::svc::Handle* pOut, nn::svc::Handle session);
void SendServerProcessHandle(nn::svc::Handle session);
void ReceiveServerProcessHandle(nn::svc::Handle* pOut, nn::svc::Handle session);
void SendClientProcessHandle(nn::svc::Handle session);
void ReceiveClientProcessHandle(nn::svc::Handle* pOut, nn::svc::Handle session);
void ServerSendData(nn::svc::Handle session, void* data, size_t size);
void ClientReceiveData(void* buffer, size_t size, nn::svc::Handle session);
void ServerReceiveData(void* buffer, size_t size, nn::svc::Handle session);
void ClientSendData(nn::svc::Handle session, void* data, size_t size);

void CheckAsyncFailure(nn::Bit32* pMsgBuffer);

nn::Bit16 GetIpcTag(nn::Bit32* pMsgBuffer);

void SetOnlyIpcTag(nn::Bit32* pMsgBuffer, nn::Bit16 ipcTag);

int AddProcessIdData(nn::svc::ipc::MessageBuffer* ipcMsg, int offset);
int AddThreadHandleData(
        nn::svc::Handle* pOutThreadHandle, nn::svc::ipc::MessageBuffer* ipcMsg,
        int offset, uintptr_t sp);
int AddPointerData(
        nn::svc::ipc::MessageBuffer* ipcMsg, int offset,
        const char** data, const size_t* dataSize, int32_t dataNum);
int AddMapData(
        nn::svc::ipc::MessageBuffer* ipcMsg, int offset,
        char* buffer, size_t bufferSize, int32_t data);
int AddRawData(
        nn::svc::ipc::MessageBuffer* ipcMsg, int offset, const char* buffer, size_t bufferSize);
int AddAllData(
        nn::svc::Handle* pOutCopyHandle, nn::svc::Handle* pOutMoveHandle,
        nn::svc::ipc::MessageBuffer* ipcMsg, int offset,
        const char** pointerData, const size_t* pointerDataSizes, int32_t pointerDataNum,
        char* sendDataBuffer, size_t sendDataSize, uint8_t sendDataChar,
        char* receiveDataBuffer, size_t receiveDataSize, uint8_t receiveDataChar,
        char* exchangeDataBuffer, size_t exchangeDataSize, uint8_t exchangeDataChar,
        const char* rawData, size_t rawDataSize);
int AddAllData(
        nn::svc::Handle* pOutCopyHandle, nn::svc::Handle* pOutMoveHandle,
        nn::svc::ipc::MessageBuffer* ipcMsg, int offset,
        const char** pointerData, const size_t* pointerDataSizes, int32_t pointerDataNum,
        char* sendDataBuffer, size_t sendDataSize,
        char* receiveDataBuffer, size_t receiveDataSize, uint8_t receiveDataChar,
        char* exchangeDataBuffer, size_t exchangeDataSize, uint8_t exchangeDataChar,
        const char* rawData, size_t rawDataSize);

int RegisterMapData(
        nn::svc::ipc::MessageBuffer* ipcMsg, int offset,
        char* buffer, size_t bufferSize);

void WriteMapData(char* buffer, size_t bufferSize, uint8_t data);

void MakeOnlyProcessIdData(nn::Bit32* pMsgBuffer);
void MakeOnlyCopyHandleData(nn::svc::Handle* pOutThreadHandle, nn::Bit32* pMsgBuffer);
void MakeOnlyMoveHandleData(nn::svc::Handle* pOutThreadHandle, nn::Bit32* pMsgBuffer);
void MakeOnlyPointerWithIpcBufferData(
        nn::Bit32* pMsgBuffer,
        const char** data, const size_t* dataSize, int32_t dataNum);
void MakeOnlyPointerWithOneUserBuffer(
        nn::Bit32* pMsgBuffer,
        const char** data, const size_t* dataSize, int32_t dataNum,
        char* buffer, size_t bufferSize);
void MakeOnlyPointerWithMultiUserBuffer(
        nn::Bit32* pMsgBuffer,
        const char** data, const size_t* dataSize, int32_t dataNum,
        char** buffer, const size_t* bufferSize);
void MakeOnlyMapSendData(
        nn::Bit32* pMsgBuffer, char* buffer, size_t bufferSize, uint8_t data);
void MakeOnlyMapReceiveData(
        nn::Bit32* pMsgBuffer, char* buffer, size_t bufferSize, uint8_t data);
void MakeOnlyMapExchangeData(
        nn::Bit32* pMsgBuffer, char* buffer, size_t bufferSize, uint8_t data);
void MakeOnlyRawData(nn::Bit32* pMsgBuffer, const char* buffer, size_t bufferSize);
void MakeAllData(
        nn::svc::Handle* pOutCopyHandle, nn::svc::Handle* pOutMoveHandle,
        nn::Bit32* pMsgBuffer,
        const char** pointerData, const size_t* pointerDataSizes, int32_t pointerDataNum,
        char* sendDataBuffer, size_t sendDataSize, uint8_t sendDataChar,
        char* receiveDataBuffer, size_t receiveDataSize, uint8_t receiveDataChar,
        char* exchangeDataBuffer, size_t exchangeDataSize, uint8_t exchangeDataChar,
        const char* rawData, size_t rawDataSize,
        char** receivePointerDataBuffers, const size_t* receivePointerDataSizes, int32_t receiveListNum
        );
void MakeAllData(
        nn::svc::Handle* pOutCopyHandle, nn::svc::Handle* pOutMoveHandle,
        nn::Bit32* pMsgBuffer,
        const char** pointerData, const size_t* pointerDataSizes, int32_t pointerDataNum,
        char* sendDataBuffer, size_t sendDataSize,
        char* receiveDataBuffer, size_t receiveDataSize, uint8_t receiveDataChar,
        char* exchangeDataBuffer, size_t exchangeDataSize, uint8_t exchangeDataChar,
        const char* rawData, size_t rawDataSize,
        char** receivePointerDataBuffers, const size_t* receivePointerDataSizes, int32_t receiveListNum);

void CmpProcessIdData(nn::svc::Handle clientHandle, nn::Bit32* pMsgBuffer, int offset);
void CmpProcessIdData(nn::Bit64 pid, nn::Bit32* pMsgBuffer, int offset);
void CmpHandleData(const nn::svc::ipc::MessageBuffer* ipcMsg, int offset, bool isSuccess);
void CmpPointerData(
        const char** data, const size_t* dataSize, int32_t dataNum,
        const uintptr_t* bufferAddresses, const size_t* bufferSizes,
        nn::svc::ipc::MessageBuffer* ipcMsg, int offset);
void CmpMap(const uint8_t* ptr, size_t bufferSize, uint8_t data);
void CmpMapData(
        uintptr_t* pOutAddr,
        size_t bufferSize, uint8_t data,
        nn::svc::ipc::MessageBuffer* ipcMsg, int offset);
void CmpRawData(const char* buffer, size_t bufferSize, nn::Bit32* pMsgBuffer, int offset);

void CheckOnlyProcessIdData(nn::svc::Handle clientHandle, nn::Bit32* pMsgBuffer);
void CheckOnlyProcessIdData(nn::Bit64 pid, nn::Bit32* pMsgBuffer);
void CheckOnlyCopyHandleData(nn::Bit32* pMsgBuffer, bool isSuccess);
void CheckOnlyMoveHandleData(nn::Bit32* pMsgBuffer, bool isSuccess);
void CheckOnlyPointerWithIpcBufferData(
        const char** data, const size_t* dataSize, int32_t dataNum,
        nn::Bit32* pMsgBuffer, size_t bufferSize);
void CheckOnlyPointerWithOneUserBuffer(
        uintptr_t bufferAddr, size_t bufferSize,
        const char** data, const size_t* dataSize, int32_t dataNum,
        nn::Bit32* pMsgBuffer);
void CheckOnlyPointerWithMultiUserBuffer(
        const uintptr_t* bufferAddr, const size_t* bufferSize,
        const char** data, const size_t* dataSize, int32_t dataNum,
        nn::Bit32* pMsgBuffer);
void CheckOnlyMapSendData(
        uintptr_t* pOutAddr, size_t bufferSize, uint8_t data, nn::Bit32* pMsgBuffer);
void CheckOnlyMapReceiveData(
        uintptr_t* pOutAddr, size_t bufferSize, uint8_t data, nn::Bit32* pMsgBuffer);
void CheckOnlyMapExchangeData(
        uintptr_t* pOutAddr, size_t bufferSize, uint8_t data, nn::Bit32* pMsgBuffer);
void CheckOnlyRawData(const char* buffer, size_t bufferSize, nn::Bit32* pMsgBuffer);
void CheckAllData(
        uintptr_t* pOutSendMapAddr, uintptr_t* pOutReceiveMapAddr, uintptr_t* pOutExchangeMapAddr,
        nn::svc::Handle clientHandle, bool isHandleAlive, bool doCheckMoveHandle,
        const char** pointerData, const size_t* pointerDataSizes, int32_t pointerDataNum,
        const uintptr_t* pointerBuffers, const size_t* pointerBufferSizes,
        size_t sendDataSize, uint8_t sendDataChar,
        size_t receiveDataSize, uint8_t receiveDataChar,
        size_t exchangeDataSize, uint8_t exchangeDataChar,
        const char* rawData, size_t rawDataSize,
        nn::Bit32* pMsgBuffer);
void CheckAllDataFromClient(
        nn::Bit64 pid, bool doCheckMoveHandle,
        const char** pointerData, const size_t* pointerDataSizes, int32_t pointerDataNum,
        const uintptr_t* pointerBuffers, const size_t* pointerBufferSizes,
        const uint8_t* sendMapBuffer, size_t sendDataSize, uint8_t sendDataChar,
        const uint8_t* recvMapBuffer, size_t receiveDataSize, uint8_t receiveDataChar,
        const uint8_t* exchMapBuffer, size_t exchangeDataSize, uint8_t exchangeDataChar,
        const char* rawData, size_t rawDataSize,
        nn::Bit32* pMsgBuffer);

void GetMapAddr(uintptr_t* pOutAddr, nn::svc::ipc::MessageBuffer* ipcMsg, int offset);

#ifdef ENABLE_SHOW_IPC
void ShowIpcHeader(nn::Bit32* pMsgBuffer);
#endif

enum DefaultIpcTag : nn::Bit16
{
    DefaultIpcTag_Send = 0x0001,
    DefaultIpcTag_Receive,
    DefaultIpcTag_Reply,
};

