﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#ifndef NO_GOOGLE_TEST
#include <nnt/gtest/gtest.h>
#endif
#include <nn/svc/svc_Result.h>
#include <nn/nn_Log.h>
#include <nn/util/util_FormatString.h>
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <cstdarg>
#include <cstring>

namespace {
    void OutputToUart(uintptr_t arg, const char* pCharacters, int count)
    {
        nn::svc::OutputDebugString(pCharacters, count);
    }

    void ShowAssertFile(const char* filename, int lineNo)
    {
        TestLog("Asserted at ");
        TestLog(filename);
        TestLog(": %d\n", lineNo);
        TestLog("\n");
    }

    void ShowThreadInfo(nn::svc::Handle handle)
    {
        nn::Bit64 tid;
        auto result = nn::svc::GetThreadId(&tid, handle);
        if (result.IsFailure())
        {
            tid = static_cast<nn::Bit64>(-1);
        }
        TestLog("TID=%lld\n", tid);
    }

    void ShowDumpInfo()
    {
#ifdef ENABLE_ASSERT_DETAIL_LOG
        TestLog("* Thread *\n");
        nn::svc::DumpInfo(nn::svc::DumpInfoType_Thread, 0);

        TestLog("* Thread CallStacks *\n");
        nn::svc::DumpInfo(nn::svc::DumpInfoType_ThreadCallstack, 0);

        TestLog("\n* Kernel Objects*\n");
        nn::svc::DumpInfo(nn::svc::DumpInfoType_KernelObject, 0);

        TestLog("\n* Handle *\n");
        nn::svc::DumpInfo(nn::svc::DumpInfoType_Handle, 0);

        TestLog("\n* Kernel Memory *\n");
        nn::svc::DumpInfo(nn::svc::DumpInfoType_Memory, -2);

        TestLog("\n* Process Memory *\n");
        nn::svc::DumpInfo(nn::svc::DumpInfoType_Memory, -1);
#endif
    }

    void TestAssertDumpCore()
    {
        TestLog("** Assert Thread Info **\n");
        ShowThreadInfo(nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
        TestLog("\n");

        TestLog("** Dump Info **\n");
        ShowDumpInfo();
    }
}

void TestLog(const char* pFormat, ...)
{
    std::va_list args;
    va_start(args, pFormat);

    nn::util::VFormatString(OutputToUart, 0, pFormat, args);
}

nn::Bit32 g_ClientBeforeData[DefaultUserBufferSize / sizeof(nn::Bit32)];
nn::Bit32 g_ClientAfterData[DefaultUserBufferSize / sizeof(nn::Bit32)];
nn::Bit32 g_ServerBeforeData[DefaultUserBufferSize / sizeof(nn::Bit32)];
nn::Bit32 g_ServerAfterData[DefaultUserBufferSize / sizeof(nn::Bit32)];

void TestAssertDump(const char* filename, int lineNo, nn::Result* result)
{
    TestLog("=== TEST ASSERT ===\n");
    ShowAssertFile(filename, lineNo);

    if (result)
    {
        TestLog("** Result **\n");
        TestLog("Module=%d, Description=%d\n", result->GetModule(), result->GetDescription());
        TestLog("\n");
    }

    TestAssertDumpCore();
}

void TestAssertDump(const char* filename, int lineNo, nn::Bit64 val1, nn::Bit64 val2)
{
    TestLog("=== TEST ASSERT ===\n");
    ShowAssertFile(filename, lineNo);

    TestLog("** Compared Values **\n");
    TestLog("Val1=0x%llx, Val2=0x%llx\n", val1, val2);
    TestLog("\n");

    TestAssertDumpCore();
}

void TestIpcAssertObject::OutputLog()
{
    TestLog("handle=0x%x\n", m_Handle.GetPrintableBits());
    TestLog("pMsgBuffer=0x%p\n", m_pMsgBuffer);
    TestLog("size=0x%p\n", m_Size);

    nn::svc::ipc::MessageBuffer ipcMsg(m_pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    TestLog("+++ IPC Header +++\n");
    TestLog("Tag: 0x%x\n", ipcHeader.GetTag());
    TestLog("PointerNum: 0x%x\n", ipcHeader.GetPointerNum());
    TestLog("SendNum: 0x%x\n", ipcHeader.GetSendNum());
    TestLog("ReceiveNum: 0x%x\n", ipcHeader.GetReceiveNum());
    TestLog("ExchangeNum: 0x%x\n", ipcHeader.GetExchangeNum());
    TestLog("RawNum: 0x%x\n", ipcHeader.GetRawNum());
    TestLog("ReceiveListNum: 0x%x\n", ipcHeader.GetReceiveListNum());
    TestLog("SpecialNum: 0x%x\n", ipcHeader.GetSpecialNum());

    if (ipcHeader.GetSpecialNum() > 0)
    {
        TestLog("ProcessIdFlag: 0x%x\n", ipcSpecial.GetProcessIdFlag());
        TestLog("CopyHandleNum: 0x%x\n", ipcSpecial.GetCopyHandleNum());
        TestLog("MoveHandleNum: 0x%x\n", ipcSpecial.GetMoveHandleNum());
    }
    TestLog("--- IPC Header ---\n");
}

void TestMemoryAreaObject::OutputLog()
{
    TestLog("+++ Memory Area +++\n");
    TestLog("address = 0x%llx\n", m_Address);

    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;
    nn::Result result = nn::svc::QueryProcessMemory(&blockInfo, &pageInfo, m_Handle, m_Address);
    if (result.IsFailure())
    {
        TestLog("[warn] Cannot get memory info. (handle=0x%x, result.mod=%d, result.desc=%d)\n",
                m_Handle.GetPrintableBits(), result.GetModule(), result.GetDescription());
    }
    else
    {
        TestLog("state = %d\n", blockInfo.state);
        TestLog("permission = %d\n", blockInfo.permission);
        TestLog("attribute = %d\n", blockInfo.attribute);
        TestLog("ipc = %d\n", blockInfo.ipcCount);
        TestLog("device = %d\n", blockInfo.deviceCount);
    }
    TestLog("--- Memory Area ---\n\n");
}
