﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestMemory.h"
#include <nn/svc/svc_Dd.h>
#include <nn/svc/ipc/svc_SessionMessage.h>

extern "C" void nnMain();

namespace {
char g_Buffer[0x1000] __attribute__((aligned(0x1000)));
const uint64_t SpaceAddr = 0;

void CheckSuccessMemoryArea(TestMemoryInfo** array, int numState, nn::svc::Handle handle)
{
    nn::Result result;

    for (int i = 0; i < numState; i++)
    {
        ASSERT_TRUE(array[i]->GetSize() > 0 && ((array[i]->GetSize() & 0xfff) == 0));

        array[i]->Initialize();
        size_t size = 0x1000;
        size_t align = 0x400000;
        uintptr_t addr = array[i]->GetAddress();
        uintptr_t deviceAddress = addr & (align - 1);
        size_t outSize;

        result = nn::svc::MapDeviceAddressSpace(&outSize,
                    handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                    addr, size, deviceAddress,
                    nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_SUCCESS(result);

        CheckMemory(
                addr,
                array[i]->GetState(),
                array[i]->GetPermission(),
                nn::svc::MemoryAttribute_DeviceShared | array[i]->GetAttribute());

        result = nn::svc::UnmapDeviceAddressSpace(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                addr, outSize, deviceAddress);
        ASSERT_RESULT_SUCCESS(result);

        array[i]->CheckDefaultState();

        result = nn::svc::UnmapDeviceAddressSpace(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                addr, outSize, deviceAddress);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        array[i]->CheckDefaultState();

        array[i]->Close();
    }
}

void CheckInvalidMemoryArea(TestMemoryInfo** array, int numState, nn::svc::Handle handle)
{
    nn::Result result;

    for (int i = 0; i < numState; i++)
    {
        if (!(array[i]->GetSize() > 0 && ((array[i]->GetSize() & 0xfff) == 0)))
        {
            array[i]->Close();
            continue;
        }

        array[i]->Initialize();
        size_t size = 0x1000;
        size_t align = 0x400000;
        uintptr_t addr =  array[i]->GetAddress();
        uintptr_t deviceAddress = addr & (align - 1);

        result = nn::svc::UnmapDeviceAddressSpace(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                addr, size, deviceAddress);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        array[i]->CheckDefaultState();

        array[i]->Close();
    }
}

} // namespace

TEST(UnmapDeviceAddressSpace, AddressSpaceHandleTest)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    uint64_t size = 0x80000000;
    uint64_t deviceAddress = 0;
    size_t align = 0x400000;
    nn::svc::Handle handle;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, size);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    TestHeap heap(align * 2);
    uintptr_t addr = RoundUp<uintptr_t>(heap.GetAddress(), align);
    size_t mapSize = 0x1000;

    // TEST 133-6
    // MapDeviceAddressSpaceAligned でマップされていないハンドルを受け付けない
    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, mapSize, deviceAddress);
    ASSERT_RESULT_FAILURE(result);

    result = nn::svc::MapDeviceAddressSpaceAligned(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, mapSize, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 133-2
    // INVALID_HANDLE_VALUE を受け付けない
    result = nn::svc::UnmapDeviceAddressSpace(
            nn::svc::INVALID_HANDLE_VALUE, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, mapSize, deviceAddress);
    ASSERT_RESULT_FAILURE(result);

    // TEST 133-3
    // スレッドの擬似ハンドルを受け付けない
    result = nn::svc::UnmapDeviceAddressSpace(
            nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, mapSize, deviceAddress);
    ASSERT_RESULT_FAILURE(result);

    // TEST 133-4
    // プロセスの擬似ハンドルを受け付けない
    result = nn::svc::UnmapDeviceAddressSpace(
            nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, mapSize, deviceAddress);
    ASSERT_RESULT_FAILURE(result);

    // TEST 133-1
    // CreateDeviceAddressSpace のハンドルを受け付ける
    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, mapSize, deviceAddress);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 133-5
    // Close したハンドラ を受け付けない
    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, mapSize, deviceAddress);
    ASSERT_RESULT_FAILURE(result);
}

TEST(UnmapDeviceAddressSpace, ProcessHandleTest)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    uint64_t spaceSize = 0x80000000;
    uint64_t deviceAddress= 0;
    uint64_t align = 0x400000;
    nn::svc::Handle handle;

    TestHeap heap(align * 2);
    uintptr_t addr = RoundUp<uintptr_t>(heap.GetAddress(), align);

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    size_t size = 0x1000;

    result = nn::svc::MapDeviceAddressSpaceAligned(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, size, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);


    // TEST 133-7
    // CreateDeviceAddressSpace のハンドルを受け付けない
    // test_MapDeviceAddressSpaceAligned_Arguments.cpp でテスト

    // TEST 133-8
    // プロセスのハンドルを受け付ける
    // test_MapDeviceAddressSpaceAligned_Arguments.cpp でテスト

    // TEST 133-11
    // プロセスの擬似ハンドルを受け付ける
    // test_MapDeviceAddressSpaceAligned_Arguments.cpp でテスト

    // TEST 133-12
    // Close したハンドラを受け付けない
    // test_MapDeviceAddressSpaceAligned_Arguments.cpp でテスト

    // TEST 133-9
    // INVALID_HANDLE_VALUE を受け付けない
    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::INVALID_HANDLE_VALUE,
            addr, size, deviceAddress);
    ASSERT_RESULT_FAILURE(result);

    // TEST 133-10
    // スレッドの擬似ハンドルを受け付けない
    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD,
            addr, size, deviceAddress);
    ASSERT_RESULT_FAILURE(result);

    // 終了処理
    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, size, deviceAddress);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(UnmapDeviceAddressSpace, ProcessAddressTest)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    uint64_t spaceSize = 0x80000000;
    uint64_t deviceAddress = 0;
    nn::svc::Handle handle;


    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    uintptr_t heapAddr;
    size_t align = 0x400000;
    size_t heapSize = align * 2;
    TestHeap heap(align * 2);

    heapAddr = heap.GetAddress();
    ASSERT_TRUE(heapSize == heap.GetSize());
    heapAddr = RoundUp<uintptr_t>(heap.GetAddress(), align);
    heapSize -= heapAddr - heap.GetAddress();

    nn::svc::PhysicalMemoryInfo physInfo = { 0 };
    result = nn::svc::QueryPhysicalAddress(&physInfo, heapAddr);
    ASSERT_RESULT_SUCCESS(result);
    // 4MBアラインされるはず
    uintptr_t physAddr = physInfo.physicalAddress + (heapAddr - physInfo.virtualAddress);
    ASSERT_TRUE((physAddr & (align - 1)) == 0);

    result = nn::svc::MapDeviceAddressSpaceAligned(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            heapAddr, heapSize, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 133-13
    // MapDeviceAddressSpaceAligned と同じアドレスを受け付ける
    // test_MapDeviceAddressSpaceAligned_Arguments.cpp でテスト

    // TEST 133-14
    // MapDeviceAddressSpaceAligned と違うアドレスを受け付けない
    for (uint64_t i = 1; i < 0x1000; i++)
    {
        result = nn::svc::UnmapDeviceAddressSpace(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr + i, 0x1000, deviceAddress);
        ASSERT_RESULT_FAILURE(result);
    }

    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            heapAddr, heapSize, deviceAddress);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(UnmapDeviceAddressSpace, SizeTest)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    uint64_t spaceSize = 0x80000000;
    uint64_t deviceAddress = 0;
    nn::svc::Handle handle;
    size_t align = 0x400000;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    uintptr_t heapAddr;
    size_t heapSize = align * 2;

    TestHeap heap(heapSize);
    heapAddr = heap.GetAddress();
    ASSERT_TRUE(heapSize == heap.GetSize());
    heapAddr = RoundUp<uintptr_t>(heap.GetAddress(), align);
    heapSize -= heapAddr - heap.GetAddress();

    nn::svc::PhysicalMemoryInfo physInfo = { 0 };
    result = nn::svc::QueryPhysicalAddress(&physInfo, heapAddr);
    ASSERT_RESULT_SUCCESS(result);
    // 4MBアラインされるはず
    uintptr_t physAddr = physInfo.physicalAddress + (heapAddr - physInfo.virtualAddress);
    ASSERT_TRUE((physAddr & (align - 1)) == 0);

    result = nn::svc::MapDeviceAddressSpaceAligned(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            heapAddr, heapSize, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 133-15
    // MapDeviceAddressSpaceAligned と同じサイズを受け付ける
    // test_MapDeviceAddressSpaceAligned_Arguments.cpp でテスト

    // TEST 133-16
    // 4KB にアライメントされていないと失敗する
    for (uint64_t i = 1; i < 0x1000; i++)
    {
        result = nn::svc::UnmapDeviceAddressSpace(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, i, deviceAddress);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());
    }

    // TEST 133-17
    // 0 を受け付けない
    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            heapAddr, 0, deviceAddress);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());

    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            heapAddr, heapSize, deviceAddress);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(UnmapDeviceAddressSpace, DeviceAddressTest)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    uint64_t spaceSize = 0x1000000;
    nn::svc::Handle handle;
    size_t align = 0x400000;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    uintptr_t heapAddr;
    size_t heapSize = align * 2;
    uint64_t deviceAddress = 0x1000;

    TestHeap heap(heapSize);
    heapAddr = heap.GetAddress();
    ASSERT_TRUE(heapSize == heap.GetSize());
    heapAddr = RoundUp<uintptr_t>(heap.GetAddress(), align);
    heapSize -= heapAddr - heap.GetAddress();

    nn::svc::PhysicalMemoryInfo physInfo = { 0 };
    result = nn::svc::QueryPhysicalAddress(&physInfo, heapAddr);
    ASSERT_RESULT_SUCCESS(result);
    // 4MBアラインされるはず
    uintptr_t physAddr = physInfo.physicalAddress + (heapAddr - physInfo.virtualAddress);
    ASSERT_TRUE((physAddr & (align - 1)) == 0);

    // deviceAddress に合わせるため、0x1000 を足しておく
    heapAddr += 0x1000;
    heapSize -= 0x1000;

    uint64_t mapSize = 0x2000;
    ASSERT_TRUE(heapSize >= mapSize);

    result = nn::svc::MapDeviceAddressSpaceAligned(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            heapAddr, mapSize, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);


    // TEST 133-18
    // MapDeviceAddressSpaceAligned と同じアドレスを受け付ける
    // test_MapDeviceAddressSpaceAligned_Arguments.cpp でテスト

    // TEST 133-19
    // MapDeviceAddressSpaceAligned と違うアドレスを受け付けない
    for (uint64_t addr = 0; addr < deviceAddress; addr++)
    {
        result = nn::svc::UnmapDeviceAddressSpace(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, 0x1000, addr);
        ASSERT_RESULT_FAILURE(result);
    }

    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            heapAddr, mapSize, deviceAddress);
    ASSERT_RESULT_SUCCESS(result);

    // deviceAddress に合わせるため、0x1000 を引いておく
    heapAddr -= 0x1000;
    heapSize += 0x1000;

    result = nn::svc::MapDeviceAddressSpaceAligned(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            heapAddr, mapSize, 0,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 133-42
    // 0 を受け付ける
    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            heapAddr, mapSize, 0);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(UnmapDeviceAddressSpace, ProcessAreaTest)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;

    uint64_t spaceSize = 0x80000000;
    size_t align = 0x400000;
    uint64_t deviceAddress = 0;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    uintptr_t addr;
    uint64_t size = 0x1000;

    nn::svc::MemoryState successState[] = {
        nn::svc::MemoryState_Normal,
        nn::svc::MemoryState_Stack,
        nn::svc::MemoryState_Alias,
        nn::svc::MemoryState_Code,
        nn::svc::MemoryState_CodeData,
        nn::svc::MemoryState_AliasCode,
#ifdef SUPPORT_ALIAS_CODE_DATA
        nn::svc::MemoryState_AliasCodeData,
#endif // SUPPORT_ALIAS_CODE_DATA
        nn::svc::MemoryState_Ipc,
        nn::svc::MemoryState_NonSecureIpc,
        nn::svc::MemoryState_NonDeviceIpc,
        nn::svc::MemoryState_Transfered,
        nn::svc::MemoryState_SharedTransfered,
    };

    const int NumSuccessState = sizeof(successState) / sizeof(nn::svc::MemoryState);

    TestMemoryInfo* header;
    GenerateMemoryStateList(
            &header, reinterpret_cast<uintptr_t>(g_Buffer), sizeof(g_Buffer));

    // 許可されたメモリ状態の領域を受け付ける
    TestMemoryInfo* successArea[NumSuccessState];
    header->GetTestListWithStates(successArea, successState, NumSuccessState);

    CheckSuccessMemoryArea(successArea, NumSuccessState, handle);

    // 許可されていないメモリ状態の領域を受け付けない
    const int NumInvalid = NumTestMemoryState - NumSuccessState;
    TestMemoryInfo* invalidArea[NumInvalid];
    header->GetTestListExceptStates(invalidArea, NumInvalid, successState, NumSuccessState);

    CheckInvalidMemoryArea(invalidArea, NumInvalid, handle);

    uint32_t successAttribute[] = {
        0,
        nn::svc::MemoryAttribute_Uncached,
        nn::svc::MemoryAttribute_DeviceShared,
        nn::svc::MemoryAttribute_Uncached | nn::svc::MemoryAttribute_DeviceShared,
    };
    const int NumSuccessAttr = sizeof(successAttribute) / sizeof(nn::svc::MemoryAttribute);
    TestMemoryInfo* successAttrArea[NumSuccessAttr];

    GenerateMemoryAttributeList(
            &header, reinterpret_cast<uintptr_t>(g_Buffer), sizeof(g_Buffer));
    header->GetTestListWithAttributes(successAttrArea, successAttribute, NumSuccessAttr);
    CheckSuccessMemoryArea(successAttrArea, NumSuccessAttr, handle);

    const int NumInvalidAttr = NumTestMemoryAttribute - NumSuccessAttr;
    TestMemoryInfo* invalidAttr[NumInvalidAttr];
    header->GetTestListExceptAttributes(
            invalidAttr, NumInvalidAttr, successAttribute, NumSuccessAttr);
    CheckInvalidMemoryArea(invalidAttr, NumInvalidAttr, handle);

    {
        TestNormalMemoryState normal;
        normal.Initialize();

        size = 0x1000;
        addr = normal.GetAddress();
        ASSERT_TRUE(normal.GetSize() >= size * 3);
        deviceAddress = addr & (align - 1);

        // TEST 133-35
        // メモリ状態が一様でないと失敗する
        result = nn::svc::MapDeviceAddressSpaceAligned(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                addr + size, size, deviceAddress + size,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        CheckMemory(
                addr + size,
                normal.GetState(),
                nn::svc::MemoryPermission_ReadWrite,
                nn::svc::MemoryAttribute_DeviceShared);

        // 前方
        result = nn::svc::UnmapDeviceAddressSpace(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                addr, size * 2, deviceAddress);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        CheckMemory(
                addr + size,
                normal.GetState(),
                nn::svc::MemoryPermission_ReadWrite,
                nn::svc::MemoryAttribute_DeviceShared);

        // 後方
        result = nn::svc::UnmapDeviceAddressSpace(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                addr + size, size * 2, deviceAddress + size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        CheckMemory(
                addr + size,
                normal.GetState(),
                nn::svc::MemoryPermission_ReadWrite,
                nn::svc::MemoryAttribute_DeviceShared);

        // TEST 133-36
        // オーバーフローする組み合わせを指定できない
        uint64_t backSize = size;
        size = static_cast<uint64_t>(-size);

        result = nn::svc::UnmapDeviceAddressSpace(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                addr + backSize, size, deviceAddress + backSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        CheckMemory(
                addr + backSize,
                normal.GetState(),
                nn::svc::MemoryPermission_ReadWrite,
                nn::svc::MemoryAttribute_DeviceShared);

        size = backSize;
        result = nn::svc::UnmapDeviceAddressSpace(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                addr + size, size, deviceAddress + size);
        ASSERT_RESULT_SUCCESS(result);

        normal.Close();
    }
} // NOLINT(readability/fn_size)

TEST(UnmapDeviceAddressSpace, DeviceAreaTest)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;

    uint64_t spaceSize = 0x80000000;
    uint64_t deviceAddress = 0;
    size_t align = 0x400000;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    TestHeap heap(align * 2);
    ASSERT_TRUE(align * 2 == heap.GetSize());
    uintptr_t heapAddr = RoundUp<uintptr_t>(heap.GetAddress(), align);
    size_t heapSize = heap.GetSize() - (heapAddr - heap.GetAddress());
    uint64_t size = 0x1000;
    ASSERT_TRUE(heapSize >= size * 3);

    result = nn::svc::MapDeviceAddressSpaceAligned(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            heapAddr, size * 2, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    CheckMemory(
            heapAddr,
            nn::svc::MemoryState_Normal,
            nn::svc::MemoryPermission_ReadWrite,
            nn::svc::MemoryAttribute_DeviceShared);

    // TEST 133-37
    // メモリ状態が一様でないと失敗する
    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            heapAddr, size * 2, deviceAddress + size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    CheckMemory(
            heapAddr,
            nn::svc::MemoryState_Normal,
            nn::svc::MemoryPermission_ReadWrite,
            nn::svc::MemoryAttribute_DeviceShared);

    // TEST 133-38
    // オーバーフローする組み合わせを指定できない
    deviceAddress = static_cast<uint64_t>(-size * 2);
    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            heapAddr, size * 2, deviceAddress);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    CheckMemory(
            heapAddr,
            nn::svc::MemoryState_Normal,
            nn::svc::MemoryPermission_ReadWrite,
            nn::svc::MemoryAttribute_DeviceShared);

    deviceAddress = 0;
    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            heapAddr, size * 2, deviceAddress);
    ASSERT_RESULT_SUCCESS(result);

    CheckMemory(
            heapAddr,
            nn::svc::MemoryState_Normal,
            nn::svc::MemoryPermission_ReadWrite,
            0);
}

