﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/os.h>

namespace {
const int64_t SleepTime = 100 * 1000 * 1000;
const uintptr_t WaitKey = 0xcccccccc;

nn::Bit32 g_Value[2];
char g_Stack[2][DefaultStackSize] __attribute__((aligned(0x1000)));

void WaitProcessWideKeyThread(uintptr_t addr)
{
    AutoThreadExit autoExit;
    nn::Result result;

    result = nn::svc::WaitProcessWideKeyAtomic(addr, WaitKey, 1, -1);
    NN_ASSERT_RESULT_SUCCESS(result);

    /*
    result = nn::svc::ArbitrateUnlock(addr);
    NN_ASSERT_RESULT_SUCCESS(result);
    */
}

} // namespace

TEST(SignalProcessWideKey, key)
{
    nn::Result result;
    uintptr_t ValidKey[] = {
        0, 1, 0x1000, 0xFFFF, 0x10000, 0xFFFFFFFF, static_cast<uintptr_t>(-1)
    };

    // TEST 139-1
    // uintptr_t の値を受け付ける
    for (int32_t i = 0; i < static_cast<int32_t>(sizeof(ValidKey) / sizeof(uintptr_t)); i++)
    {
        nn::svc::SignalProcessWideKey(ValidKey[i], 0);
    }
}

TEST(SignalProcessWideKey, NumTest)
{
    nn::Result result;

    nn::Bit32 value = nn::os::GetCurrentThread()->_handle;
    uintptr_t addr;

    {
        uintptr_t pc = reinterpret_cast<uintptr_t>(WaitProcessWideKeyThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack[0]) + sizeof(g_Stack[0]);
        uintptr_t param = reinterpret_cast<uintptr_t>(&g_Value[0]);
        int32_t priority = TestLowestThreadPriority;
        int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

        TestThread thread1(pc, param, sp, priority, idealCore);
        thread1.Start();

        sp = reinterpret_cast<uintptr_t>(g_Stack[1]) + sizeof(g_Stack[1]);
        param = reinterpret_cast<uintptr_t>(&g_Value[1]);

        TestThread thread2(pc, param, sp, priority, idealCore);
        thread2.Start();

        // 作成したスレッドに遷移
        nn::svc::SleepThread(SleepTime);

        g_Value[0] = value;
        g_Value[1] = value;

        // TEST 139-2
        // 0 を与えるとすべてのスレッドの待機を解除する
        nn::svc::SignalProcessWideKey(WaitKey, 0);

        // 作成したスレッドに遷移
        nn::svc::SleepThread(SleepTime);

        addr = reinterpret_cast<uintptr_t>(&g_Value[0]);
        result = nn::svc::ArbitrateUnlock(addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(g_Value[0] == 1);

        addr = reinterpret_cast<uintptr_t>(&g_Value[1]);
        result = nn::svc::ArbitrateUnlock(addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(g_Value[1] == 1);

        thread1.Wait();
        thread2.Wait();
    }

    // TEST 139-6
    // num 個よりも多いスレッドがいるときは、優先度が高いものからnum個のスレッドを起床させる
    {
        uintptr_t pc = reinterpret_cast<uintptr_t>(WaitProcessWideKeyThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack[0]) + sizeof(g_Stack[0]);
        uintptr_t param = reinterpret_cast<uintptr_t>(&g_Value[0]);
        int32_t priority = TestLowestThreadPriority;
        int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

        TestThread thread1(pc, param, sp, priority, idealCore);
        thread1.Start();

        priority = TestLowestThreadPriority - 1;
        sp = reinterpret_cast<uintptr_t>(g_Stack[1]) + sizeof(g_Stack[1]);
        param = reinterpret_cast<uintptr_t>(&g_Value[1]);

        TestThread thread2(pc, param, sp, priority, idealCore);
        thread2.Start();

        // 作成したスレッドに遷移
        nn::svc::SleepThread(SleepTime);

        g_Value[0] = value;
        g_Value[1] = value;

        // TEST 139-3
        // 正の値を与えると、最大num個のスレッドの待機を解除する
        nn::svc::SignalProcessWideKey(WaitKey, 1);

        // 作成したスレッドに遷移
        nn::svc::SleepThread(SleepTime);

        // 低優先度のスレッドは起きていない
        addr = reinterpret_cast<uintptr_t>(&g_Value[0]);
        result = nn::svc::ArbitrateUnlock(addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(g_Value[0] == 0);

        // 高優先度のスレッドは起きてArbitrateLockが呼ばれている
        addr = reinterpret_cast<uintptr_t>(&g_Value[1]);
        result = nn::svc::ArbitrateUnlock(addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(g_Value[1] == 1);

        nn::svc::SignalProcessWideKey(WaitKey, 1);

        ASSERT_TRUE(g_Value[1] == 1);

        thread1.Wait();
        thread2.Wait();
    }

    // TEST 139-5
    // num 個以内のスレッドしかいないときは、すべてのスレッドを起床させる
    {
        uintptr_t pc = reinterpret_cast<uintptr_t>(WaitProcessWideKeyThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack[0]) + sizeof(g_Stack[0]);
        uintptr_t param = reinterpret_cast<uintptr_t>(&g_Value[0]);
        int32_t priority = TestLowestThreadPriority;
        int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

        TestThread thread1(pc, param, sp, priority, idealCore);
        thread1.Start();

        sp = reinterpret_cast<uintptr_t>(g_Stack[1]) + sizeof(g_Stack[1]);
        param = reinterpret_cast<uintptr_t>(&g_Value[1]);

        TestThread thread2(pc, param, sp, priority, idealCore);
        thread2.Start();

        // 作成したスレッドに遷移
        nn::svc::SleepThread(SleepTime);

        g_Value[0] = value;
        g_Value[1] = value;

        // TEST 139-3
        // 正の値を与えると、最大num個のスレッドの待機を解除する
        nn::svc::SignalProcessWideKey(WaitKey, INT32_MAX);

        // 作成したスレッドに遷移
        nn::svc::SleepThread(SleepTime);

        addr = reinterpret_cast<uintptr_t>(&g_Value[0]);
        result = nn::svc::ArbitrateUnlock(addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(g_Value[0] == 1);

        addr = reinterpret_cast<uintptr_t>(&g_Value[1]);
        result = nn::svc::ArbitrateUnlock(addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(g_Value[1] == 1);

        thread1.Wait();
        thread2.Wait();
    }

    {
        uintptr_t pc = reinterpret_cast<uintptr_t>(WaitProcessWideKeyThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack[0]) + sizeof(g_Stack[0]);
        uintptr_t param = reinterpret_cast<uintptr_t>(&g_Value[0]);
        int32_t priority = TestLowestThreadPriority;
        int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

        TestThread thread1(pc, param, sp, priority, idealCore);
        thread1.Start();

        sp = reinterpret_cast<uintptr_t>(g_Stack[1]) + sizeof(g_Stack[1]);
        param = reinterpret_cast<uintptr_t>(&g_Value[1]);

        TestThread thread2(pc, param, sp, priority, idealCore);
        thread2.Start();

        // 作成したスレッドに遷移
        nn::svc::SleepThread(SleepTime);

        g_Value[0] = value;
        g_Value[1] = value;

        // TEST 139-4
        // 負の値を与えると、すべてのスレッドの待機を解除する
        nn::svc::SignalProcessWideKey(WaitKey, -1);

        // 作成したスレッドに遷移
        nn::svc::SleepThread(SleepTime);

        addr = reinterpret_cast<uintptr_t>(&g_Value[0]);
        result = nn::svc::ArbitrateUnlock(addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(g_Value[0] == 1);

        addr = reinterpret_cast<uintptr_t>(&g_Value[1]);
        result = nn::svc::ArbitrateUnlock(addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(g_Value[1] == 1);

        thread1.Wait();
        thread2.Wait();
    }

    {
        uintptr_t pc = reinterpret_cast<uintptr_t>(WaitProcessWideKeyThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack[0]) + sizeof(g_Stack[0]);
        uintptr_t param = reinterpret_cast<uintptr_t>(&g_Value[0]);
        int32_t priority = TestLowestThreadPriority;
        int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

        TestThread thread1(pc, param, sp, priority, idealCore);
        thread1.Start();

        sp = reinterpret_cast<uintptr_t>(g_Stack[1]) + sizeof(g_Stack[1]);
        param = reinterpret_cast<uintptr_t>(&g_Value[1]);

        TestThread thread2(pc, param, sp, priority, idealCore);
        thread2.Start();

        // 作成したスレッドに遷移
        nn::svc::SleepThread(SleepTime);

        g_Value[0] = value;
        g_Value[1] = value;

        // TEST 139-4
        // 負の値を与えると、すべてのスレッドの待機を解除する
        nn::svc::SignalProcessWideKey(WaitKey, 0x80000000);

        // 作成したスレッドに遷移
        nn::svc::SleepThread(SleepTime);

        addr = reinterpret_cast<uintptr_t>(&g_Value[0]);
        result = nn::svc::ArbitrateUnlock(addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(g_Value[0] == 1);

        addr = reinterpret_cast<uintptr_t>(&g_Value[1]);
        result = nn::svc::ArbitrateUnlock(addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(g_Value[1] == 1);

        thread1.Wait();
        thread2.Wait();
    }
} // NOLINT(readability/fn_size)

