﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Dd.h>
#include <cstring>


/* TEST 80-11 から 80-25 */
/* パーミッションのテスト*/
TEST(SharedMemoryTest, CreateSharedMemory)
{
    TestSharedMemoryLeak leakTest;
    nn::svc::Handle handle;
    nn::Result result;
    nn::svc::MemoryPermission myPermission;
    nn::svc::MemoryPermission otherPermission;
    size_t size;
    const nn::svc::MemoryPermission perm[] =
    {
        nn::svc::MemoryPermission_None,
        nn::svc::MemoryPermission_Read,
        nn::svc::MemoryPermission_Write,
        nn::svc::MemoryPermission_Execute,
        nn::svc::MemoryPermission_ReadWrite,
        nn::svc::MemoryPermission_ReadExecute,
        nn::svc::MemoryPermission_DontCare
    };

    for (int i = 0; i < static_cast<int>(sizeof(perm) / sizeof(*perm)); i++)
    {
        myPermission = perm[i];
        bool myPermissionIsValid = false;
        switch (myPermission)
        {
            case nn::svc::MemoryPermission_Read:
            case nn::svc::MemoryPermission_ReadWrite:
                {
                    myPermissionIsValid = true;
                }
                break;
            default:
                break;
        }

        for (int j = 0; j < static_cast<int>(sizeof(perm) / sizeof(*perm)); j++)
        {
            otherPermission = perm[j];
            bool otherPermissionIsValid = false;
            switch (otherPermission)
            {
                case nn::svc::MemoryPermission_Read:
                case nn::svc::MemoryPermission_ReadWrite:
                case nn::svc::MemoryPermission_DontCare:
                    {
                        otherPermissionIsValid = true;
                    }
                    break;
                default:
                    break;
            }

            size = 4 * 1024 * 1024;
            result = nn::svc::CreateSharedMemory(&handle, size, myPermission, otherPermission);
            if (myPermissionIsValid && otherPermissionIsValid)
            {
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(handle);
                ASSERT_RESULT_SUCCESS(result);
            }
            else
            {
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());
            }
        }
    }

    {
        size = static_cast<size_t>(-1) & ~(0xFFF);
        myPermission = nn::svc::MemoryPermission_ReadWrite;
        otherPermission = nn::svc::MemoryPermission_ReadWrite;

        TestHeap heap(sizeof(nn::svc::Handle) * NN_KERN_SLAB_OBJ_NUM_SHARED_MEMORY);
        nn::svc::Handle* pHandles = reinterpret_cast<nn::svc::Handle*>(heap.GetAddress());
        int count;
        for (count = 0; count < NN_KERN_SLAB_OBJ_NUM_SHARED_MEMORY; count++)
        {
            result = nn::svc::CreateSharedMemory(&pHandles[count], size, myPermission, otherPermission);
            if (result <= nn::svc::ResultOutOfMemory() || result <= nn::svc::ResultInvalidSize())
            {
                break;
            }
            ASSERT_RESULT_SUCCESS(result);
        }
        ASSERT_TRUE(count < NN_KERN_SLAB_OBJ_NUM_SHARED_MEMORY);

        for (int i = 0; i < count; i++)
        {
            result = nn::svc::CloseHandle(pHandles[i]);
            ASSERT_RESULT_SUCCESS(result);
        }
    }
}

TEST(SharedMemoryTest, MapSharedMemoryFailure)
{
    TestSharedMemoryLeak leakTest;
    nn::svc::Handle handle;
    nn::Result result;
    size_t size;

    size = 1024 * 1024;
    result = nn::svc::CreateSharedMemory(&handle, size, nn::svc::MemoryPermission_ReadWrite, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 17-34
    result = nn::svc::MapSharedMemory(handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_None);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    // TEST 17-36
    result = nn::svc::MapSharedMemory(handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_Execute);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    // TEST 17-44
    result = nn::svc::MapSharedMemory(handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_ReadExecute);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    // TEST 17-37
    result = nn::svc::MapSharedMemory(handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_DontCare);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}


TEST(SharedMemoryTest, MapSharedMemoryRw)
{
    TestSharedMemoryLeak leakTest;
    nn::svc::Handle handle;
    nn::Result result;
    size_t size;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;

    size = 1024 * 1024;
    result = nn::svc::CreateSharedMemory(&handle, size, nn::svc::MemoryPermission_ReadWrite, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 17-1
    // Free領域に対して、マップを行うことが出来る
    result = nn::svc::MapSharedMemory(handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, g_FreeAreaBegin);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.baseAddress == g_FreeAreaBegin);
    ASSERT_TRUE(blockInfo.size == size);
    TEST_RW(g_FreeAreaBegin);

    // TEST 17-2
    // マップした後ろのFree領域に対して、同じパーミッションでマップすると、ブロックがマージされる
    result = nn::svc::MapSharedMemory(handle, g_FreeAreaBegin + size, size, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, g_FreeAreaBegin + size);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.baseAddress == g_FreeAreaBegin);
    ASSERT_TRUE(blockInfo.size == size * 2);
    TEST_RW(g_FreeAreaBegin + size);

    // TEST 17-46
    // CreateSharedMemory で指定したパーミッションと違うパーミッションでマップすると失敗する
    result = nn::svc::MapSharedMemory(handle, g_FreeAreaBegin + size * 2, size, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    // TEST 17-4
    // マップした領域で、書き込んだデータが共有されることを確認する
    *reinterpret_cast<uint32_t*>(g_FreeAreaBegin) = 0xdeadbeef;
    asm volatile ("dmb ish":::"memory");
    ASSERT_TRUE(*reinterpret_cast<uint32_t*>(g_FreeAreaBegin + size) == 0xdeadbeef);
    *reinterpret_cast<uint32_t*>(g_FreeAreaBegin + size) = 0xbabeface;
    asm volatile ("dmb ish":::"memory");
    ASSERT_TRUE(*reinterpret_cast<uint32_t*>(g_FreeAreaBegin) == 0xbabeface);

    // TEST 17-5
    // マップした領域全体のデータが、すべて等しく共有されていることを確認する
    ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(g_FreeAreaBegin), reinterpret_cast<void*>(g_FreeAreaBegin + size), size) == 0);

    // TEST 17-6
    // 既にマップされている領域に重複してマップすることはできない
    result = nn::svc::MapSharedMemory(handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 17-7
    // MemoryPermission_ReadExecute を指定することはできない
    result = nn::svc::MapSharedMemory(handle, g_FreeAreaBegin + size * 3, size, nn::svc::MemoryPermission_ReadExecute);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    // TEST 17-8
    // CreateSharedMemory で指定したサイズと違うサイズを指定することはできない
    result = nn::svc::MapSharedMemory(handle, g_FreeAreaBegin + size * 3, size * 2, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());
    result = nn::svc::MapSharedMemory(handle, g_FreeAreaBegin + size * 3, size / 2, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());

    // TEST 18-1
    // SharedMemory のサイズと一致していないと失敗する
    result = nn::svc::UnmapSharedMemory(handle, g_FreeAreaBegin, size / 2);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());

    // TEST 18-2
    // addr/size のペアが指す領域と SharedMemory の物理メモリが一致しないと失敗する
    result = nn::svc::UnmapSharedMemory(handle, g_FreeAreaBegin + size / 2, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());

    // TEST 18-3
    // 対象のSharedMemory が解放できる
    result = nn::svc::UnmapSharedMemory(handle, g_FreeAreaBegin, size);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::UnmapSharedMemory(handle, g_FreeAreaBegin + size, size);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::UnmapSharedMemory(handle, g_FreeAreaBegin + size * 2, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());


    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SharedMemoryTest, MapSharedMemoryRo)
{
    TestSharedMemoryLeak leakTest;
    nn::svc::Handle handle;
    nn::Result result;
    size_t size;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;

    size = 1024 * 1024;
    result = nn::svc::CreateSharedMemory(&handle, size, nn::svc::MemoryPermission_Read, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 17-9
    // Free領域に対して、マップを行うことが出来る
    result = nn::svc::MapSharedMemory(handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, g_FreeAreaBegin);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.baseAddress == g_FreeAreaBegin);
    ASSERT_TRUE(blockInfo.size == size);
    TEST_RO(g_FreeAreaBegin);

    // TEST 17-10
    // マップした後ろのFree領域に対して、同じパーミッションでマップすると、ブロックがマージされる
    result = nn::svc::MapSharedMemory(handle, g_FreeAreaBegin + size, size, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, g_FreeAreaBegin + size);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.baseAddress == g_FreeAreaBegin);
    ASSERT_TRUE(blockInfo.size == size * 2);
    TEST_RO(g_FreeAreaBegin + size);

    // TEST 17-11
    // マップした領域全体のデータが、すべて等しく共有されていることを確認する
    asm volatile ("dmb ish":::"memory");
    ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(g_FreeAreaBegin), reinterpret_cast<void*>(g_FreeAreaBegin + size), size) == 0);

    // TEST 17-12
    // 既にマップされている領域に重複してマップすることはできない
    result = nn::svc::MapSharedMemory(handle, g_FreeAreaBegin + size, size, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 17-13
    // MemoryPermission_ReadWrite を指定することはできない
    result = nn::svc::MapSharedMemory(handle, g_FreeAreaBegin + size * 2, size, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    // TEST 17-14
    // CreateSharedMemory で指定したサイズと違うサイズを指定することはできない
    result = nn::svc::MapSharedMemory(handle, g_FreeAreaBegin + size * 2, size * 2, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());
    result = nn::svc::MapSharedMemory(handle, g_FreeAreaBegin + size * 2, size / 2, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());

    // TEST 18-4
    // SharedMemory のサイズと一致していないと失敗する
    result = nn::svc::UnmapSharedMemory(handle, g_FreeAreaBegin, size / 2);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());

    // TEST 18-5
    // addr/size のペアが指す領域と SharedMemory の物理メモリが一致しないと失敗する
    result = nn::svc::UnmapSharedMemory(handle, g_FreeAreaBegin + size / 2, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());

    // TEST 18-6
    // 対象のSharedMemory が解放できる
    result = nn::svc::UnmapSharedMemory(handle, g_FreeAreaBegin, size);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::UnmapSharedMemory(handle, g_FreeAreaBegin + size, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}


