﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_MemoryState.h"
#include "util_TestMemory.h"
#include <nn/svc/svc_Dd.h>
#include <nn/svc/ipc/svc_SessionMessage.h>

extern "C" void nnMain();

namespace {

char g_Buffer[0x1000] __attribute__((aligned(0x1000)));

void CheckSuccessMemoryArea(TestMemoryInfo** array, int numState)
{
    nn::Result result;

    size_t size = 0x1000;
    nn::Bit32 mask = nn::svc::MemoryAttribute_Uncached;

    for (int i = 0; i < numState; i++)
    {
        nn::Bit32 attribute = nn::svc::MemoryAttribute_Uncached;

        ASSERT_TRUE(array[i]->GetSize() >= size && ((array[i]->GetSize() & 0xfff) == 0));
        array[i]->Initialize();

        uintptr_t addr = array[i]->GetAddress();
        if ((array[i]->GetAttribute() & mask) == attribute)
        {
            attribute = 0;
        }

        // 属性が変わらない
        result = nn::svc::SetMemoryAttribute(addr, size, mask, (array[i]->GetAttribute() & mask));
        ASSERT_RESULT_SUCCESS(result);

        array[i]->CheckDefaultState();

        // 属性が変わる
        result = nn::svc::SetMemoryAttribute(addr, size, mask, attribute);
        ASSERT_RESULT_SUCCESS(result);

        CheckMemory(
                addr,
                array[i]->GetState(),
                array[i]->GetPermission(),
                (attribute == 0) ? (array[i]->GetAttribute() ^ mask) : (array[i]->GetAttribute() | attribute));

        // 属性が元に戻る
        result = nn::svc::SetMemoryAttribute(addr, size, mask, (array[i]->GetAttribute() & mask));
        ASSERT_RESULT_SUCCESS(result);

        array[i]->CheckDefaultState();

        array[i]->Close();
    }
}

void CheckInvalidMemoryArea(TestMemoryInfo** array, int numState)
{
    nn::Result result;

    size_t size = 0x1000;
    nn::Bit32 attribute = nn::svc::MemoryAttribute_Uncached;
    nn::Bit32 mask = nn::svc::MemoryAttribute_Uncached;

    for (int i = 0; i < numState; i++)
    {
        if (!(array[i]->GetSize() >= size && ((array[i]->GetSize() & 0xfff) == 0)))
        {
            array[i]->Close();
            continue;
        }

        array[i]->Initialize();

        uintptr_t addr = array[i]->GetAddress();
        result = nn::svc::SetMemoryAttribute(addr, size, mask, attribute);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        array[i]->CheckDefaultState();

        array[i]->Close();
    }
}

} // namespace

TEST(SetMemoryAttribute, AddressTest)
{
    nn::Result result;
    uintptr_t addr;
    size_t size = 0x1000;
    nn::Bit32 attribute = nn::svc::MemoryAttribute_Uncached;
    nn::Bit32 mask = nn::svc::MemoryAttribute_Uncached;

    TestHeap heap(HeapAlign);
    ASSERT_TRUE((heap.GetAddress() & 0xfff) == 0);

    // TEST 136-1
    // 4 KB のアライメントがされていると成功する
    for (addr = heap.GetAddress(); addr < heap.GetAddress() + heap.GetSize(); addr += 0x1000)
    {
        result = nn::svc::SetMemoryAttribute(addr, size, mask, attribute);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::SetMemoryAttribute(addr, size, mask, 0);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 136-2
    // 4KB のアライメントがされていないと失敗する
    for (addr = heap.GetAddress() + 1; addr < heap.GetAddress() + 0x1000; addr++)
    {
        result = nn::svc::SetMemoryAttribute(addr, size, mask, attribute);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());
    }
}

TEST(SetMemoryAttribute, SizeTest)
{
    nn::Result result;
    uintptr_t addr;
    size_t size;
    nn::Bit32 attribute = nn::svc::MemoryAttribute_Uncached;
    nn::Bit32 mask = nn::svc::MemoryAttribute_Uncached;

    TestHeap heap(HeapAlign);
    ASSERT_TRUE((heap.GetAddress() & 0xfff) == 0);
    addr = heap.GetAddress();

    // TEST 136-3
    // 4 KB のアライメントがされていると成功する
    for (size = 0x1000; size < heap.GetSize(); size += 0x1000)
    {
        result = nn::svc::SetMemoryAttribute(addr, size, mask, attribute);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::SetMemoryAttribute(addr, size, mask, 0);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 136-4
    // 4KB のアライメントがされていないと失敗する
    for (size = 1; size < 0x1000; size++)
    {
        result = nn::svc::SetMemoryAttribute(addr, size, mask, attribute);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());
    }

    // TEST 136-5
    // 0 を受け付けない
    result = nn::svc::SetMemoryAttribute(addr, 0, mask, attribute);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());
}

TEST(SetMemoryAttribute, MemoryAreaTest)
{
    nn::Result result;
    uintptr_t addr;
    size_t size = 0x1000;
    nn::Bit32 attribute = nn::svc::MemoryAttribute_Uncached;
    nn::Bit32 mask = nn::svc::MemoryAttribute_Uncached;

    nn::svc::MemoryState successState[] = {
        nn::svc::MemoryState_Normal,
        nn::svc::MemoryState_CodeData,
        nn::svc::MemoryState_Transfered,
#ifdef SUPPORT_ALIAS_CODE_DATA
        nn::svc::MemoryState_AliasCodeData,
#endif
    };
    const int NumSuccessState = sizeof(successState) / sizeof(nn::svc::MemoryState);

    TestMemoryInfo* header;
    GenerateMemoryStateList(
            &header, reinterpret_cast<uintptr_t>(g_Buffer), sizeof(g_Buffer));

    // 許可されたメモリ状態の領域を受け付ける
    TestMemoryInfo* successArea[NumSuccessState];
    header->GetTestListWithStates(successArea, successState, NumSuccessState);

    CheckSuccessMemoryArea(successArea, NumSuccessState);

    // 許可されていないメモリ状態の領域を受け付けない
    const int NumInvalid = NumTestMemoryState - NumSuccessState;
    TestMemoryInfo* invalidArea[NumInvalid];
    header->GetTestListExceptStates(invalidArea, NumInvalid, successState, NumSuccessState);

    CheckInvalidMemoryArea(invalidArea, NumInvalid);

    uint32_t successAttribute[] = {
        0,
        nn::svc::MemoryAttribute_Uncached,
        nn::svc::MemoryAttribute_DeviceShared,
        nn::svc::MemoryAttribute_DeviceShared | nn::svc::MemoryAttribute_Uncached,
    };

    const int NumSuccessAttr = sizeof(successAttribute) / sizeof(nn::svc::MemoryAttribute);
    TestMemoryInfo* successAttrArea[NumSuccessAttr];

    GenerateMemoryAttributeList(
            &header, reinterpret_cast<uintptr_t>(g_Buffer), sizeof(g_Buffer));
    header->GetTestListWithAttributes(successAttrArea, successAttribute, NumSuccessAttr);
    CheckSuccessMemoryArea(successAttrArea, NumSuccessAttr);

    // TEST 136-37
    // メモリ属性が付加されていると失敗する
    const int NumInvalidAttr = NumTestMemoryAttribute - NumSuccessAttr;
    TestMemoryInfo* invalidAttr[NumInvalidAttr];
    header->GetTestListExceptAttributes(
            invalidAttr, NumInvalidAttr, successAttribute, NumSuccessAttr);
    CheckInvalidMemoryArea(invalidAttr, NumInvalidAttr);

    {
        TestNormalMemoryState normal;
        normal.Initialize();

        // TEST 136-22
        // メモリ状態が一様ではないと失敗する
        // 前方
        addr = normal.GetAddress() - size;
        result = nn::svc::SetMemoryAttribute(addr, size * 2, mask, attribute);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        normal.CheckDefaultState();

        // 後方
        addr = normal.GetAddress() + normal.GetSize() - size;
        result = nn::svc::SetMemoryAttribute(addr, size * 2, mask, attribute);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        normal.CheckDefaultState();

        // TEST 136-23
        // オーバーフローする addr/size の組み合わせは失敗する
        addr = normal.GetAddress();
        size = static_cast<size_t>(-size);

        result = nn::svc::SetMemoryAttribute(addr, size, mask, attribute);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        normal.CheckDefaultState();
    }
} // NOLINT(readability/fn_size)

TEST(SetMemoryAttribute, MemoryAreaPermissionTest)
{
    nn::Result result;
    uintptr_t addr;
    size_t size = 0x1000;
    nn::Bit32 attribute = nn::svc::MemoryAttribute_Uncached;
    nn::Bit32 mask = nn::svc::MemoryAttribute_Uncached;
    nn::svc::MemoryInfo blockInfo;

    TestHeap heap(HeapAlign);
    ASSERT_TRUE((heap.GetAddress() & 0xfff) == 0);
    addr = heap.GetAddress();

    // TEST 136-24
    // MemoryPermission_ReadWrite の領域を受け付ける
    {
        GetMemoryInfo(&blockInfo, addr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);

        result = nn::svc::SetMemoryAttribute(addr, size, mask, attribute);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, addr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == attribute);

        result = nn::svc::SetMemoryAttribute(addr, size, mask, 0);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 136-25
    // MemoryPermission_Read の領域を受け付ける
    {
        TestMemoryPermission perm(addr, size, nn::svc::MemoryPermission_Read);
        GetMemoryInfo(&blockInfo, addr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_Read);
        ASSERT_TRUE(blockInfo.attribute == 0);

        result = nn::svc::SetMemoryAttribute(addr, size, mask, attribute);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, addr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_Read);
        ASSERT_TRUE(blockInfo.attribute == attribute);

        result = nn::svc::SetMemoryAttribute(addr, size, mask, 0);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 136-26
    // MemoryPermission_None の領域を受け付ける
    {
        TestMemoryPermission perm(addr, size, nn::svc::MemoryPermission_None);
        GetMemoryInfo(&blockInfo, addr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.attribute == 0);

        result = nn::svc::SetMemoryAttribute(addr, size, mask, attribute);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, addr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.attribute == attribute);

        result = nn::svc::SetMemoryAttribute(addr, size, mask, 0);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 136-29
    // メモリ領域のパーミッションが一様でないと失敗する
    {
        TestMemoryPermission perm(addr + size, size, nn::svc::MemoryPermission_Read);

        GetMemoryInfo(&blockInfo, addr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.baseAddress + blockInfo.size == addr + size);
        ASSERT_TRUE(blockInfo.attribute == 0);

        GetMemoryInfo(&blockInfo, addr + size);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_Read);
        ASSERT_TRUE(blockInfo.baseAddress == addr + size);
        ASSERT_TRUE(blockInfo.size == size);
        ASSERT_TRUE(blockInfo.attribute == 0);

        result = nn::svc::SetMemoryAttribute(addr, size * 2, mask, attribute);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        GetMemoryInfo(&blockInfo, addr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.baseAddress + blockInfo.size == addr + size);
        ASSERT_TRUE(blockInfo.attribute == 0);

        GetMemoryInfo(&blockInfo, addr + size);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_Read);
        ASSERT_TRUE(blockInfo.baseAddress == addr + size);
        ASSERT_TRUE(blockInfo.size == size);
        ASSERT_TRUE(blockInfo.attribute == 0);
    }
}

TEST(SetMemoryAttribute, AttributeTest)
{
    nn::Result result;
    uintptr_t addr = reinterpret_cast<uintptr_t>(g_Buffer);
    size_t size = 0x1000;
    nn::Bit32 mask = nn::svc::MemoryAttribute_Uncached;
    nn::Bit32 attribute;
    nn::svc::MemoryAttribute ValidAttribute[] = {
        nn::svc::MemoryAttribute_Uncached,
    };
    nn::svc::MemoryInfo blockInfo;

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);

    // TEST 136-31
    // 指定されているMemoryAttribute を受け付ける
    for (int32_t i = 0; i < static_cast<int32_t>(sizeof(ValidAttribute) / sizeof(attribute)); i++)
    {
        attribute = ValidAttribute[i];

        result = nn::svc::SetMemoryAttribute(addr, size, mask, attribute);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, addr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == attribute);

        result = nn::svc::SetMemoryAttribute(addr, size, mask, 0);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 136-32
    // MemoryAttribute ではない値を受け付けない
    attribute = ~(
            nn::svc::MemoryAttribute_Uncached
            | nn::svc::MemoryAttribute_Locked
            | nn::svc::MemoryAttribute_IpcLocked
            | nn::svc::MemoryAttribute_DeviceShared);

    result = nn::svc::SetMemoryAttribute(addr, size, mask, attribute);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCombination());

    nn::svc::MemoryAttribute InvalidAttribute[] = {
        nn::svc::MemoryAttribute_Locked,
        nn::svc::MemoryAttribute_IpcLocked,
        nn::svc::MemoryAttribute_DeviceShared,
    };

    // TEST 136-36
    // 許可されていない MemoryAttribute は指定できない
    for (int32_t i = 0; i < static_cast<int32_t>(sizeof(InvalidAttribute) / sizeof(attribute)); i++)
    {
        attribute = InvalidAttribute[i];

        result = nn::svc::SetMemoryAttribute(addr, size, mask, attribute);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCombination());
    }

    // TEST 136-39
    // 0 を受け付ける
    result = nn::svc::SetMemoryAttribute(addr, size, mask, 0);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SetMemoryAttribute, MaskTest)
{
    nn::Result result;
    uintptr_t addr = reinterpret_cast<uintptr_t>(g_Buffer);
    size_t size = 0x1000;
    nn::Bit32 mask = nn::svc::MemoryAttribute_Uncached;
    nn::Bit32 attribute = nn::svc::MemoryAttribute_Uncached;
    nn::svc::MemoryInfo blockInfo;

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);

    // TEST 136-40
    // 0 を受け付ける
    result = nn::svc::SetMemoryAttribute(addr, size, 0, 0);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 136-41
    // mask で立っていないビットに相当する attribute は指定できない
    result = nn::svc::SetMemoryAttribute(addr, size, 0, attribute);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCombination());

    // TEST 136-42
    // 指定されていない MemoryAttribute を指定できない
    mask = ~(nn::svc::MemoryAttribute_Uncached);
    result = nn::svc::SetMemoryAttribute(addr, size, mask, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCombination());

    nn::svc::MemoryAttribute InvalidMask[] = {
        nn::svc::MemoryAttribute_Locked,
        nn::svc::MemoryAttribute_IpcLocked,
        nn::svc::MemoryAttribute_DeviceShared,
    };

    for (int32_t i = 0; i < static_cast<int32_t>(sizeof(InvalidMask) / sizeof(mask)); i++)
    {
        mask = InvalidMask[i];

        result = nn::svc::SetMemoryAttribute(addr, size, mask, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCombination());
    }
}
