﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"

/* TEST 1-1, 1-2, 1-3, 1-4, 1-6, 1-7 */
/* SetHeapSize の正常系の確認をする */
TEST(SetHeapSizeTest, AllocationTest)
{
    uintptr_t addr;
    size_t size;
    size_t maxAllocation;
    nn::Result result;
    int i;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;

    size = 0;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 1-1
    size = HeapAlign;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 1-2
    // Heap Area Check
    ASSERT_TRUE(g_HeapAreaBegin <= addr);
    ASSERT_TRUE((addr + size - 1) <= g_HeapAreaEnd);
    maxAllocation = size;

    // TEST 1-6
    // リソースの限界を超えるとエラーになることを確認
    for (i = 1;; i++)
    {
        uintptr_t tmp;
        size = HeapAlign * i;
        result = nn::svc::SetHeapSize(&tmp, size);
        if (result.IsFailure())
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfMemory());
            break;
        }
        ASSERT_RESULT_SUCCESS(result);
        // TEST 1-4
        // Heapのトップが変更されないことの確認
        ASSERT_TRUE(tmp == addr);
        // TEST 1-2
        // Heap Area Check
        ASSERT_TRUE(g_HeapAreaBegin<= addr);
        ASSERT_TRUE((addr + size - 1) <= g_HeapAreaEnd);
        maxAllocation = size;
        // TEST 1-3
        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.baseAddress == addr);
        ASSERT_TRUE(blockInfo.size == size);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);

        // アクセス確認
        volatile char* p;
        size_t offset;
        for (p = reinterpret_cast<volatile char*>(addr), offset = 0; offset < size; offset += 0x1000)
        {
            *(p + offset);
            *(p + offset) = 1;
            *(p + offset + 0x1000 - 1);
            *(p + offset + 0x1000 - 1) = 1;
        }
    }

    size = 0;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_RESULT_SUCCESS(result);

    size = HeapAlign;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_RESULT_SUCCESS(result);

    // Heap Area Check
    ASSERT_TRUE(g_HeapAreaBegin <= addr);
    ASSERT_TRUE((addr + size - 1) <= g_HeapAreaEnd);

    // TEST 1-12
    // 再度ヒープ領域が同じ大きさだけ確保できることを確認
    for (i = 1;; i++)
    {
        uintptr_t tmp;
        size = HeapAlign * i;
        result = nn::svc::SetHeapSize(&tmp, size);
        if (result.IsFailure())
        {
            // 一度解放後に同じだけ取れることの確認
            ASSERT_TRUE(maxAllocation == HeapAlign * (i - 1));
            break;
        }
        ASSERT_RESULT_SUCCESS(result);
        // TEST 1-4
        // Heapのトップが変更されないことの確認
        ASSERT_TRUE(tmp == addr);
        // TEST 1-2
        // Heap Area Check
        ASSERT_TRUE(g_HeapAreaBegin <= addr);
        ASSERT_TRUE((addr + size - 1) <= g_HeapAreaEnd);

        // TEST 1-3
        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.baseAddress == addr);
        ASSERT_TRUE(blockInfo.size == size);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);

        // アクセス確認
        volatile char* p;
        size_t offset;
        for (p = reinterpret_cast<volatile char*>(addr), offset = 0; offset < size; offset += 0x1000)
        {
            *(p + offset);
            *(p + offset) = 1;
            *(p + offset + 0x1000 - 1);
            *(p + offset + 0x1000 - 1) = 1;
        }
    }

    // TEST 1-7
    for (i = maxAllocation / HeapAlign; i >= 0; i--)
    {
        uintptr_t tmp;
        size = HeapAlign * i;
        result = nn::svc::SetHeapSize(&tmp, size);
        // Heapのトップが変更されないことの確認
        ASSERT_RESULT_SUCCESS(result);
        // TEST 1-4
        ASSERT_TRUE(tmp == addr);
        // TEST 1-2
        // Heap Area Check
        ASSERT_TRUE(g_HeapAreaBegin <= addr);
        ASSERT_TRUE((addr + size - 1) <= g_HeapAreaEnd);

        if (i != 0)
        {
            // TEST 1-3
            result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(blockInfo.baseAddress == addr);
            ASSERT_TRUE(blockInfo.size == size);
            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);

            // アクセス確認
            volatile char* p;
            size_t offset;
            for (p = reinterpret_cast<volatile char*>(addr), offset = 0; offset < size; offset += 0x1000)
            {
                *(p + offset);
                *(p + offset) = 1;
                *(p + offset + 0x1000 - 1);
                *(p + offset + 0x1000 - 1) = 1;
            }
        }
    }

    size = 0;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_RESULT_SUCCESS(result);
} // NOLINT (readability/fn_size)

TEST(SetHeapSizeTest, FailureTest)
{
    uintptr_t addr;
    size_t size;
    size_t maxAllocation = 0;
    nn::Result result;
    uint32_t i;

    size = 0;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_RESULT_SUCCESS(result);

    for (i = 1;; i++)
    {
        uintptr_t tmp;
        size = HeapAlign * i;
        result = nn::svc::SetHeapSize(&tmp, size);
        if (result.IsFailure())
        {
            break;
        }
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(g_HeapAreaBegin <= addr);
        ASSERT_TRUE((addr + size - 1) <= g_HeapAreaEnd);
        maxAllocation = size;
    }

    // TEST 1-10
    // 最大容量以上のサイズが確保出来ないことを確認する
    for (i = (maxAllocation / HeapAlign) + 1;; i++)
    {
        uintptr_t tmp;
        size = HeapAlign * i;
        if ((CheckVirtualEnd == 0 && size == 0) || (CheckVirtualEnd > 0 && size > CheckVirtualEnd))
        {
            break;
        }
        result = nn::svc::SetHeapSize(&tmp, size);
        NN_ASSERT(result <= nn::svc::ResultOutOfMemory() || result <= nn::svc::ResultInvalidSize());
    }
    // TEST 1-10
    // 境界値
    {
        uintptr_t tmp;
        size = static_cast<size_t>(-HeapAlign);
        result = nn::svc::SetHeapSize(&tmp, size);
        NN_ASSERT(result <= nn::svc::ResultOutOfMemory() || result <= nn::svc::ResultInvalidSize());
    }

    size = 0;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 1-8
    // HeapAlign にアライメントされていないサイズが指定できないことを確認する
    for (i = 1; i < HeapAlign; i++)
    {
        size = HeapAlign + i;
        result = nn::svc::SetHeapSize(&addr, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());
    }

    size = 0;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_RESULT_SUCCESS(result);
}

