﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestProcess.h"
#include <nn/svc/svc_Dd.h>

#if defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK1) \
    || defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK2) \
    || defined (NN_BUILD_CONFIG_HARDWARE_NX)
#include "test_TestTmrDevice_Jetson.h"
#endif // defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK1)

namespace {

char g_Stack[DefaultStackSize] __attribute__((aligned(0x1000)));
char g_IpcClientBuffer[0x1000] __attribute__((aligned(0x1000)));

const uint32_t FailCodes[] = {
    0xe3a03000, // mov r3, #0
    0xe5930000, // ldr r0, [r3] ; ここでデータアボート
    0xef000000 | NN_SVC_ID_EXIT_PROCESS,
};


void IpcServerThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle* session = reinterpret_cast<nn::svc::Handle*>(arg);
    int32_t index;

    SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
    result = nn::svc::ReplyAndReceive(&index, session, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    NN_ASSERT_RESULT_SUCCESS(result);
    NN_ASSERT_EQUAL(GetIpcTag(nullptr), DefaultIpcTag_Send);

    SetOnlyIpcTag(nullptr, DefaultIpcTag_Reply);
    result = nn::svc::ReplyAndReceive(&index, session, 0, *session, 0);
    NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
}

void DummyThread(uintptr_t arg)
{
    NN_UNUSED(arg);
    AutoThreadExit autoExit;
}

} // namespace

TEST(ResetSignal, HandleTest)
{
    TestEventLeak leakTest;
    TestProcessLeak testProcessLeak;
    nn::Result result;
    nn::svc::Handle readableEvent;
    nn::svc::Handle writableEvent;
    int32_t index;

    // CreateEvent
    {
        result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose wEventCloser(writableEvent);
        AutoHandleClose rEventCloser(readableEvent);

        result = nn::svc::SignalEvent(writableEvent);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::WaitSynchronization(&index, &readableEvent, 1, 0);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 134-1
        // CreateEvent の ReadableEvent を受け付ける
        // TEST 134-11
        // シグナル状態なら成功する
        result = nn::svc::ResetSignal(readableEvent);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::WaitSynchronization(&index, &readableEvent, 1, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

        // TEST 134-15
        // シグナル状態でなければ失敗する
        result = nn::svc::ResetSignal(readableEvent);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());

        // TEST 134-2
        // CreateEvent の WritableEvent を受け付けない
        result = nn::svc::ResetSignal(writableEvent);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

        result = nn::svc::CloseHandle(readableEvent);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 134-10
        // クローズしたハンドルを受け付けない
        result = nn::svc::ResetSignal(readableEvent);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    }

    // InterruptEvent
#if defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK1) \
    || defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK2) \
    || defined (NN_BUILD_CONFIG_HARDWARE_NX)
    {
        TestTmrDriver driver;
        int32_t name = driver.GetInterruptNumber();
        result = nn::svc::CreateInterruptEvent(
                &readableEvent, name, nn::svc::InterruptType_Level);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose rEventCloser(readableEvent);

        driver.SetTimer(100);
        driver.StartTimer();

        result = nn::svc::WaitSynchronization(&index, &readableEvent, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        driver.ClearInterrupt();

        // TEST 134-3
        // CreateInterruptEvent の ReadableEvent を受け付ける
        result = nn::svc::ResetSignal(readableEvent);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::WaitSynchronization(&index, &readableEvent, 1, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

        // TEST 134-16
        // シグナル状態でなければ失敗する
        result = nn::svc::ResetSignal(readableEvent);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());
    }
#endif // defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK1)

    // SendAsyncRequestWithUserBuffer
    {
        nn::svc::Handle serverSession;
        nn::svc::Handle clientSession;

        result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sSessionCloser(serverSession);
        AutoHandleClose cSessionCloser(clientSession);

        uintptr_t pc = reinterpret_cast<uintptr_t>(IpcServerThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + DefaultStackSize;
        uintptr_t param = reinterpret_cast<uintptr_t>(&serverSession);
        int32_t priority = TestLowestThreadPriority;
        int32_t idealCore = g_ProcessIdealCore;

        TestThread thread(pc, param, sp, priority, idealCore);
        thread.Start();

        uintptr_t msgBuffer = reinterpret_cast<uintptr_t>(g_IpcClientBuffer);
        size_t msgSize = sizeof(g_IpcClientBuffer);

        nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(msgBuffer);
        SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
        nn::svc::Handle eventHandle;
        result = nn::svc::SendAsyncRequestWithUserBuffer(
                &eventHandle, msgBuffer, msgSize, clientSession);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::WaitSynchronization(&index, &eventHandle, 1, -1);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_EQ(GetIpcTag(pMsgBuffer), DefaultIpcTag_Reply);

        result = nn::svc::WaitSynchronization(&index, &eventHandle, 1, 0);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 134-4
        // SendAsyncRequestWithUserBuffer の ReadableEvent を受け付ける
        // TEST 134-13
        // シグナル状態であれば成功する
        result = nn::svc::ResetSignal(eventHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::WaitSynchronization(&index, &eventHandle, 1, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

        // TEST 134-17
        // シグナル状態でなければ失敗する
        result = nn::svc::ResetSignal(eventHandle);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());

        result = nn::svc::CloseHandle(eventHandle);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 134-10
        // クローズしたハンドルを受け付けない
        result = nn::svc::ResetSignal(eventHandle);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    }

    // Process (EnableJit)
    {
        nn::Bit32 flag[DefaultCapabilityFlagNum];
        SetDefaultCapability(flag, DefaultCapabilityFlagNum);
        TestProcess process(
                1, nn::svc::CreateProcessParameterFlag_EnableJitDebug,
                flag, DefaultCapabilityFlagNum);
        nn::svc::Handle processHandle = process.GetHandle();
        SetExecuteCode(
                processHandle, FailCodes, sizeof(FailCodes),
                process.GetCodeAddress(), process.GetCodeAreaSize());

        process.Start();
        nn::svc::ResetSignal(processHandle);

        result = nn::svc::WaitSynchronization(&index, &processHandle, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 134-5
        // Process のハンドルを受け付ける
        // TEST 134-14
        // プロセスがシグナル状態でかつTerminate の状態でなければ成功する
        result = nn::svc::ResetSignal(processHandle);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 134-18
        // シグナル状態でなければ失敗する
        result = nn::svc::ResetSignal(processHandle);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());

        result = nn::svc::TerminateProcess(processHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(processHandle);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 134-10
        // クローズしたハンドルを受け付けない
        result = nn::svc::ResetSignal(processHandle);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    }

    // Thread
    {
        uintptr_t pc = reinterpret_cast<uintptr_t>(DummyThread);
        TestThread thread(pc, 0, 0, TestLowestThreadPriority, 0);

        // TEST 134-6
        // スレッドのハンドルを受け付けない
        result = nn::svc::ResetSignal(thread.GetHandle());
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    }

    // TEST 134-7
    // INVALID_HANDLE_VALUE を受け付けない
    result = nn::svc::ResetSignal(nn::svc::INVALID_HANDLE_VALUE);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 134-8
    // スレッドの擬似ハンドルを受け付けない
    result = nn::svc::ResetSignal(nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 134-9
    // プロセスの擬似ハンドルを受け付ける
    result = nn::svc::ResetSignal(nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
    ASSERT_TRUE(!(result <= nn::svc::ResultInvalidHandle()));

    // Process
    {
        nn::Bit32 flag[DefaultCapabilityFlagNum];
        SetDefaultCapability(flag, DefaultCapabilityFlagNum);
        TestProcess process(1, 0, flag, DefaultCapabilityFlagNum);
        nn::svc::Handle processHandle = process.GetHandle();

        AssignExitCode(
                processHandle, process.GetCodeAddress(), process.GetCodeAreaSize());

        process.Start();
        nn::svc::ResetSignal(processHandle);

        // TEST 134-19
        // プロセスがシグナル状態でかつ Terminate の状態だと失敗する
        for(;;)
        {
            int64_t info;
            result = nn::svc::WaitSynchronization(&index, &processHandle, 1, -1);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::GetProcessInfo(&info, processHandle, nn::svc::ProcessInfoType_State);
            NN_ASSERT_RESULT_SUCCESS(result);
            nn::svc::ProcessState state = static_cast<nn::svc::ProcessState>(info);
            if (state == nn::svc::ProcessState_Terminated)
            {
                result = nn::svc::ResetSignal(processHandle);
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());
                break;
            }

            result = nn::svc::ResetSignal(processHandle);
            if (result.IsFailure())
            {
                result = nn::svc::GetProcessInfo(&info, processHandle, nn::svc::ProcessInfoType_State);
                NN_ASSERT_RESULT_SUCCESS(result);
                state = static_cast<nn::svc::ProcessState>(info);
                ASSERT_TRUE(state == nn::svc::ProcessState_Terminated);
                break;
            }
        }
    }
} // NOLINT(readability/fn_size)

