﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestProcess.h"

#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_BaseId.autogen.h>
#include <cstring>

extern "C" void nnMain();

namespace {

#ifdef INVALID_POINTER_TEST
const char ConstVar[sizeof(nn::svc::MemoryInfo)] = {0};
#endif

} // namespace

TEST(QueryProcessMemory, pBlockInfoTest)
{
    nn::Result result;
    nn::svc::MemoryInfo* pBlockInfo;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;
    nn::svc::Handle process = nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS;
    uint64_t addr = g_FreeAreaBegin;

    // TEST 118-1
    // MemoryPermission_ReadWrite の領域を指し示していると成功する

    // ローカル変数
    pBlockInfo = &blockInfo;
    result = nn::svc::QueryProcessMemory(pBlockInfo, &pageInfo, process, addr);
    ASSERT_RESULT_SUCCESS(result);

    // ヒープ
    {
        TestHeap heap(HeapAlign);
        pBlockInfo = reinterpret_cast<nn::svc::MemoryInfo*>(heap.GetAddress());
        result = nn::svc::QueryProcessMemory(pBlockInfo, &pageInfo, process, addr);
        ASSERT_RESULT_SUCCESS(result);
    }

#ifdef INVALID_POINTER_TEST
    // TEST 118-2
    // NULL を指定すると失敗する
    pBlockInfo = NULL;
    result = nn::svc::QueryProcessMemory(pBlockInfo, &pageInfo, process, addr);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 118-3
    // MemoryPermission_None  の領域を指し示していると失敗する
    pBlockInfo = reinterpret_cast<nn::svc::MemoryInfo*>(g_FreeAreaBegin);
    result = nn::svc::QueryProcessMemory(pBlockInfo, &pageInfo, process, addr);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 118-4
    // MemoryPermission_Read  の領域を指し示していると失敗する
    uint64_t tmpAddr = reinterpret_cast<uint64_t>(ConstVar);
    pBlockInfo = reinterpret_cast<nn::svc::MemoryInfo*>(tmpAddr);
    result = nn::svc::QueryProcessMemory(pBlockInfo, &pageInfo, process, addr);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 118-5
    // MemoryPermission_ReadExecute の領域を指し示していると失敗する
    pBlockInfo = reinterpret_cast<nn::svc::MemoryInfo*>(nnMain);
    result = nn::svc::QueryProcessMemory(pBlockInfo, &pageInfo, process, addr);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

}

TEST(QueryProcessMemory, pPageInfoTest)
{
    nn::Result result;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;
    nn::svc::PageInfo* pPageInfo;
    nn::svc::Handle process = nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS;
    uint64_t addr = g_FreeAreaBegin;

    // TEST 118-8
    // MemoryPermission_ReadWrite の領域を指し示していると成功する

    // ローカル変数
    pPageInfo = &pageInfo;
    result = nn::svc::QueryProcessMemory(&blockInfo, pPageInfo, process, addr);
    ASSERT_RESULT_SUCCESS(result);

    // ヒープ
    {
        TestHeap heap(HeapAlign);
        pPageInfo = reinterpret_cast<nn::svc::PageInfo*>(heap.GetAddress());
        result = nn::svc::QueryProcessMemory(&blockInfo, pPageInfo, process, addr);
        ASSERT_RESULT_SUCCESS(result);
    }

#ifdef INVALID_POINTER_TEST
    // TEST 118-9
    // NULL を指定すると失敗する
    pPageInfo = NULL;
    result = nn::svc::QueryProcessMemory(&blockInfo, pPageInfo, process, addr);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 118-10
    // MemoryPermission_None  の領域を指し示していると失敗する
    pPageInfo = reinterpret_cast<nn::svc::PageInfo*>(g_FreeAreaBegin);
    result = nn::svc::QueryProcessMemory(&blockInfo, pPageInfo, process, addr);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 118-11
    // MemoryPermission_Read  の領域を指し示していると失敗する
    uint64_t tmpAddr = reinterpret_cast<uint64_t>(ConstVar);
    pPageInfo = reinterpret_cast<nn::svc::PageInfo*>(tmpAddr);
    result = nn::svc::QueryProcessMemory(&blockInfo, pPageInfo, process, addr);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 118-12
    // MemoryPermission_ReadExecute の領域を指し示していると失敗する
    pPageInfo = reinterpret_cast<nn::svc::PageInfo*>(nnMain);
    result = nn::svc::QueryProcessMemory(&blockInfo, pPageInfo, process, addr);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

}

TEST(QueryProcessMemory, HandleTest)
{
    nn::Result result;

    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;

    nn::Bit32 flag[DefaultCapabilityFlagNum];
    SetDefaultCapability(flag, DefaultCapabilityFlagNum);

    TestProcess process(1, 0, flag, DefaultCapabilityFlagNum);
    nn::svc::Handle processHandle = process.GetHandle();

    size_t size = 0x1000;
    uint64_t addr = process.GetCodeAddress();
    AssignExitCode(processHandle, process.GetCodeAddress(), size);

    // TEST 118-15
    // 実行前のプロセスのハンドルを受け付ける
    result = nn::svc::QueryProcessMemory(&blockInfo, &pageInfo, processHandle, addr);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::StartProcess(
            processHandle, TestLowestThreadPriority, g_ProcessIdealCore, DefaultStackSize);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 118-16
    // 実行中のプロセスのハンドルを受け付ける
    result = nn::svc::QueryProcessMemory(&blockInfo, &pageInfo, processHandle, addr);
    ASSERT_RESULT_SUCCESS(result);

    WaitProcess(processHandle);

    // TEST 118-17
    // 実行後のプロセスのハンドルを受け付ける
    result = nn::svc::QueryProcessMemory(&blockInfo, &pageInfo, processHandle, addr);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 118-18
    // プロセスの擬似ハンドルを受け付ける
    result = nn::svc::QueryProcessMemory(
            &blockInfo, &pageInfo, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, addr);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 118-19
    // INVALID_HANDLE_VALUE を受け付けない
    result = nn::svc::QueryProcessMemory(
            &blockInfo, &pageInfo, nn::svc::INVALID_HANDLE_VALUE, addr);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 118-20
    // スレッドの擬似ハンドルを受け付けない
    result = nn::svc::QueryProcessMemory(
            &blockInfo, &pageInfo, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, addr);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 118-21
    // Close したプロセスのハンドルを受け付けない
    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::QueryProcessMemory(&blockInfo, &pageInfo, processHandle, addr);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

TEST(QueryProcessMemory, AddrTest)
{
    nn::Result result;

    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;
    nn::svc::Handle process = nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS;

    // TEST 118-22
    // 0 を受け付ける
    result = nn::svc::QueryProcessMemory(&blockInfo, &pageInfo, process, 0);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 118-23
    // アライメントを気にしない
    for (uint64_t i = 1; i <= 0x1000; i++)
    {
        result = nn::svc::QueryProcessMemory(&blockInfo, &pageInfo, process, i);
        ASSERT_RESULT_SUCCESS(result);
    }
}

