﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_MemoryState.h"
#include "util_TestMemory.h"
#include <nn/svc/svc_Dd.h>
#include <nn/svc/ipc/svc_SessionMessage.h>

namespace {
char g_Buffer[0x1000] __attribute__((aligned(0x1000)));

void CheckSuccessMemoryArea(TestMemoryInfo** array, int numState)
{
    nn::Result result;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;

    for (int i = 0; i < numState; i++)
    {
        if (array[i]->GetSize() == 0)
        {
            array[i]->Close();
            continue;
        }
        array[i]->Initialize();
        uintptr_t addr = array[i]->GetAddress();

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);

        ASSERT_TRUE(blockInfo.state == array[i]->GetState());
        ASSERT_TRUE(blockInfo.permission == array[i]->GetPermission());
        ASSERT_TRUE(blockInfo.attribute == array[i]->GetAttribute());

        array[i]->Close();
    }
}

}

extern "C" void nnMain();

TEST(QueryMemory, GetSpecificMemoryInfoTest)
{
    TestMemoryInfo* header;
    GenerateMemoryStateList(
            &header, reinterpret_cast<uintptr_t>(g_Buffer), sizeof(g_Buffer));

    // 許可されたメモリ状態の領域を受け付ける
    TestMemoryInfo* successArea[NumTestMemoryState];
    header->GetTestListWithStates(successArea, AllMemoryState, NumTestMemoryState);
    CheckSuccessMemoryArea(successArea, NumTestMemoryState);

    GenerateMemoryAttributeList(
            &header, reinterpret_cast<uintptr_t>(g_Buffer), sizeof(g_Buffer));
    TestMemoryInfo* successAttrArea[NumTestMemoryAttribute];
    header->GetTestListWithAttributes(
            successAttrArea, AllMemoryAttribute, NumTestMemoryAttribute);
    CheckSuccessMemoryArea(successAttrArea, NumTestMemoryAttribute);
} // NOLINT(readability/fn_size)

#ifdef INVALID_POINTER_TEST
TEST(QueryMemory, InvalidPointerTest)
{
    nn::Result result;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;
    uintptr_t heapPtr;

#ifdef INVALID_POINTER_TEST
    // TEST 5-3
    // pBlockInfo に NULL を渡すと失敗する
    result = nn::svc::QueryMemory(NULL, &pageInfo, g_FreeAreaBegin);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 5-4
    // pageInfo に NULL を渡すと失敗する
    result = nn::svc::QueryMemory(&blockInfo, NULL, g_FreeAreaBegin);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 5-12
    // pBlockInfo に MemoryPermission_None を渡すと失敗する
    result = nn::svc::QueryMemory(
            reinterpret_cast<nn::svc::MemoryInfo*>(g_FreeAreaBegin), &pageInfo, g_FreeAreaBegin);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 5-13
    // pBlockInfo に MemoryPermission_Read を渡すと失敗する
    result = nn::svc::SetHeapSize(&heapPtr, HeapAlign);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::SetMemoryPermission(heapPtr, 0x1000, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::QueryMemory(
            reinterpret_cast<nn::svc::MemoryInfo*>(heapPtr), &pageInfo, g_FreeAreaBegin);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
    result = nn::svc::SetMemoryPermission(heapPtr, 0x1000, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::SetHeapSize(&heapPtr, 0);
    ASSERT_RESULT_SUCCESS(result);
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 5-14
    // pBlockInfo に MemoryPermission_ReadExecute を渡すと失敗する
    result = nn::svc::QueryMemory(
            reinterpret_cast<nn::svc::MemoryInfo*>(nnMain), &pageInfo, g_FreeAreaBegin);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST


#ifdef INVALID_POINTER_TEST
    // TEST 5-15
    // pageInfo に MemoryPermission_None を渡すと失敗する
    result = nn::svc::QueryMemory(
            &blockInfo, reinterpret_cast<nn::svc::PageInfo*>(g_FreeAreaBegin), g_FreeAreaBegin);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 5-16
    // pageInfo に MemoryPermission_Read を渡すと失敗する
    result = nn::svc::SetHeapSize(&heapPtr, HeapAlign);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::SetMemoryPermission(heapPtr, 0x1000, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::QueryMemory(
            &blockInfo, reinterpret_cast<nn::svc::PageInfo*>(heapPtr), g_FreeAreaBegin);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
    result = nn::svc::SetMemoryPermission(heapPtr, 0x1000, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::SetHeapSize(&heapPtr, 0);
    ASSERT_RESULT_SUCCESS(result);
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 5-17
    // pageInfo に MemoryPermission_ReadExecute を渡すと失敗する
    result = nn::svc::QueryMemory(
            &blockInfo, reinterpret_cast<nn::svc::PageInfo*>(nnMain), g_FreeAreaBegin);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

}
#endif
