﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_TestIoMapping.h"
#include "test_Common.h"
#include "util_TestMemory.h"
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/ipc/svc_SessionMessage.h>

extern "C" void nnMain();

namespace {
#ifdef INVALID_POINTER_TEST
const int ConstVar = 0;
#endif

char g_TestBuffer[0x1000] __attribute__((aligned(0x1000)));

void CheckInvalidMemoryArea(TestMemoryInfo** array, int numState)
{
    nn::Result result;
    size_t size = 0x1;

    for (int i = 0; i < numState; i++)
    {
        if(array[i]->GetSize() == 0)
        {
            array[i]->Close();
            continue;
        }

        array[i]->Initialize();

        nn::svc::PhysicalMemoryInfo info;
        result = nn::svc::QueryPhysicalAddress(&info, array[i]->GetAddress());
        if (result.IsFailure())
        {
            array[i]->Close();
            continue;
        }

        nn::svc::PhysicalAddress phyAddr = info.physicalAddress;
        uintptr_t addr;

        result = nn::svc::QueryIoMapping(&addr, phyAddr, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultNotFound());

        array[i]->CheckDefaultState();

        array[i]->Close();
    }
}

} // namespace

#if defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK1) \
    || defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK2) \
    || defined (NN_BUILD_CONFIG_HARDWARE_NX)
TEST(QueryIoMapping, pOutTest)
{
    nn::Result result;
    uintptr_t addr;
    uintptr_t* pAddr;
    nn::svc::PhysicalAddress phyAddr = DefaultIoMappingAddress;
    size_t size = 0x1000;

    // TEST 125-1
    // MemoryPermission_ReadWrite の領域を指し示していると成功する
    // ローカル変数
    result = nn::svc::QueryIoMapping(&addr, phyAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // ヒープ変数
    TestHeap heap(HeapAlign);
    pAddr = reinterpret_cast<uintptr_t*>(heap.GetAddress());
    result = nn::svc::QueryIoMapping(pAddr, phyAddr, size);
    ASSERT_RESULT_SUCCESS(result);

#ifdef INVALID_POINTER_TEST
    // TEST 125-2
    // NULL を指定すると失敗する
    pAddr = NULL;
    result = nn::svc::QueryIoMapping(pAddr, phyAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 125-3
    // MemoryPermission_None の領域を指し示していると失敗する
    pAddr = reinterpret_cast<uintptr_t*>(g_FreeAreaBegin);
    result = nn::svc::QueryIoMapping(pAddr, phyAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 125-4
    // MemoryPermission_Read の領域を指し示していると失敗する
    uintptr_t tmpAddr = reinterpret_cast<uintptr_t>(&ConstVar);
    pAddr = reinterpret_cast<uintptr_t*>(tmpAddr);
    result = nn::svc::QueryIoMapping(pAddr, phyAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 125-5
    // MemoryPermission_ReadExecute の領域を指し示していると失敗する
    pAddr = reinterpret_cast<uintptr_t*>(nnMain);
    result = nn::svc::QueryIoMapping(pAddr, phyAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

}

TEST(QueryIoMapping, AddressTest)
{
    nn::Result result;
    uintptr_t addr;
    nn::svc::PhysicalAddress phyAddr = DefaultIoMappingAddress;
    size_t size = 0x1000;

    // TEST 125-8
    // アライメントを気にしない
    for (int32_t i = 0; i < 0x1000; i++)
    {
        result = nn::svc::QueryIoMapping(&addr, phyAddr + i, size);
        ASSERT_RESULT_SUCCESS(result);
    }
}

TEST(QueryIoMapping, SizeTest)
{
    nn::Result result;
    uintptr_t addr;
    nn::svc::PhysicalAddress phyAddr = DefaultIoMappingAddress;

    // TEST 125-9
    // 0 を受け付けない
    result = nn::svc::QueryIoMapping(&addr, phyAddr, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());

    // TEST 125-8
    // アライメントを気にしない
    for (int32_t i = 1; i < 0x1000; i++)
    {
        result = nn::svc::QueryIoMapping(&addr, phyAddr, i);
        ASSERT_RESULT_SUCCESS(result);
    }
}

TEST(QueryIoMapping, MemoryAreaTest)
{
    nn::Result result;
    uintptr_t addr;
    nn::svc::PhysicalAddress phyAddr;
    size_t size = 0x1;

    // 仮想メモリの領域を受け付けない
    TestMemoryInfo* header;
    GenerateMemoryStateList(
            &header, reinterpret_cast<uintptr_t>(g_TestBuffer), sizeof(g_TestBuffer));

    TestMemoryInfo* invalidArea[NumTestMemoryState];
    header->GetTestListWithStates(invalidArea, AllMemoryState, NumTestMemoryState);

    CheckInvalidMemoryArea(invalidArea, NumTestMemoryState);

    // TEST 125-18
    // MemoryState_Io の領域の情報を取得できる
    phyAddr = DefaultIoMappingAddress;
    result = nn::svc::QueryIoMapping(&addr, phyAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 125-27
    // physicalAddres/size で示す領域のメモリ状態が一様でないと失敗する
    // 前方
    size = 0x1000;
    phyAddr = DefaultIoMappingAddress - size;
    result = nn::svc::QueryIoMapping(&addr, phyAddr, size * 2);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultNotFound());

    // TEST 125-28
    // physicalAddres/size で示す領域のメモリ状態が一様でないと失敗する
    // 後方
    size = 0x1000;
    phyAddr = DefaultIoMappingAddress + DefaultIoMappingSize - size;
    result = nn::svc::QueryIoMapping(&addr, phyAddr, size);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::QueryIoMapping(&addr, phyAddr + size, 1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultNotFound());
    result = nn::svc::QueryIoMapping(&addr, phyAddr, size * 2);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultNotFound());
} // NOLINT(readability/fn_size)

// TEST 125-26
// マッピングしてある領域のどこでも状態が取得できる
TEST(QueryIoMapping, AccessAnyWhere)
{
    nn::Result result;
    uintptr_t addr;
    nn::svc::PhysicalAddress phyAddr = DefaultIoMappingAddress;
    size_t size = 1;
    size_t blockSize = DefaultIoMappingSize;

    for (uintptr_t i = 0; i < blockSize; i++)
    {
        result = nn::svc::QueryIoMapping(&addr, phyAddr + i, size);
        ASSERT_RESULT_SUCCESS(result);
    }

    for (size = 1; size < blockSize; size++)
    {
        result = nn::svc::QueryIoMapping(&addr, phyAddr, size);
        ASSERT_RESULT_SUCCESS(result);
    }

    for (size = 1; size < blockSize; size++)
    {
        result = nn::svc::QueryIoMapping(&addr, phyAddr + (DefaultIoMappingSize - size), size);
        ASSERT_RESULT_SUCCESS(result);
    }
}
#endif

