﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_DebugProcess.h"
#include "util_TestIpc.h"
#include "util_TestProcess.h"
#include "util_MemoryState.h"
#include "util_TestSdmmc.h"
#include <cstring>

#ifdef ENABLE_OUT_OF_RESOURCE_TEST
namespace {

char g_Buffer[0x1000] __attribute__((aligned(0x1000)));
const char* DummyPortName = "dummy";
const uint64_t SpaceAddr = 0;

} // namespace

TEST(OutOfResourceTest, Event)
{
    nn::Result result;

    nn::svc::Handle writableEvent;
    nn::svc::Handle readableEvent;

    {
        ConsumeEvent consume;

        // TEST 69-41
        // Event 数が上限に達している状態で CreateEvent を呼び出すと失敗する
        result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
    }

    // TEST 69-42
    // Event が解放されれば、再度確保できるようになる
    result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(writableEvent);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(readableEvent);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(OutOfResourceTest, InterruptEvent)
{
    // 未実装
    // テスト用にデバイスをハンドル数分用意することが難しいため
}

TEST(OutOfResourceTest, Thread)
{
    nn::Result result;

    nn::svc::Handle thread;
    uintptr_t pc = 0x1;

    {
        ConsumeThread consume;

        // TEST 7-20
        // Thread 数が上限に達している状態で CreateThread を呼び出すと失敗する
        result = nn::svc::CreateThread(&thread, pc, 0, 0, 0, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
    }

    // TEST 7-21
    // Thread が解放されれば、再度確保できるようになる
    result = nn::svc::CreateThread(&thread, pc, 0, 0, 0, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(thread);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(OutOfResourceTest, SharedMemory)
{
    nn::Result result;

    nn::svc::Handle handle;
    size_t sharedSize = 0x1000;
    nn::svc::MemoryPermission permission = nn::svc::MemoryPermission_ReadWrite;

    {
        ConsumeSharedMemory consume;

        // TEST 80-28
        // SharedMemory 数が上限に達している状態で CreateSharedMemory を呼び出すと失敗する
        result = nn::svc::CreateSharedMemory(&handle, sharedSize, permission, permission);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
    }

    // TEST 80-29
    // 作成した共有メモリを解放すれば、再度共有メモリを作成できる
    result = nn::svc::CreateSharedMemory(&handle, sharedSize, permission, permission);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(OutOfResourceTest, TransferMemory)
{
    nn::Result result;

    nn::svc::Handle handle;
    uintptr_t addr = reinterpret_cast<uintptr_t>(g_Buffer);

    {
        ConsumeTransferMemory consume;

        // TEST 126-39
        // TransferMemory 数が上限に達していると、失敗する
        result = nn::svc::CreateTransferMemory(
                &handle, addr, sizeof(g_Buffer), nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
    }

    // TEST 126-40
    // TransferMemory 数が上限よりも少なくなると呼び出しに成功する
    result = nn::svc::CreateTransferMemory(
            &handle, addr, sizeof(g_Buffer), nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(OutOfResourceTest, DeviceAddressSpace)
{
#if defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || \
    defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || \
    defined (NN_BUILD_CONFIG_HARDWARE_NX)

    nn::Result result;

    nn::svc::Handle handle;
    uint64_t spaceAddr = 0;
    size_t size = 0x200000; // 2MB

    {
        ConsumeDeviceAddressSpace consume;

        // TEST 129-13
        // DeviceAddressSpace 数が上限に達していると、失敗する
        result = nn::svc::CreateDeviceAddressSpace(&handle, spaceAddr, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
    }

    // TEST 129-14
    // DeviceAddressSpace 数が上限よりも少なくなると呼び出しに成功する
    result = nn::svc::CreateDeviceAddressSpace(&handle, spaceAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

#endif // defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK1)
}

TEST(OutOfResourceTest, Session)
{
    nn::Result result;
    int32_t index;

    nn::svc::Handle namedPort;

    NamedPortManager portManager(DummyPortName, 1);
    namedPort = portManager.GetHandle();

    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    result = nn::svc::CreatePort(&serverPort, &clientPort, 1, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);
    AutoHandleClose cPortCloser(clientPort);

    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;

    {
        ConsumeSession consume;

        // TEST 64-15
        // Normal Session 数が上限に達している状態で CreateSession を呼び出すと失敗する
        result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());

        // TEST 26-13
        // Normal Session 数が上限に達している状態で ConnectToNamedPort を呼び出すと失敗する
        result = nn::svc::ConnectToNamedPort(&serverSession, DummyPortName);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());

        // TEST
        // Normal Session 数が上限に達している状態で ConnectToPort を呼び出すと失敗する
        result = nn::svc::ConnectToPort(&serverSession, clientPort);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
    }

    // TEST 64-16
    // セッションが解放されると、CreateSession でセッションが作成できるようになる
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 26-14
    //  セッションが解放されると、ConnectToNamedPort でセッションが作成できるようになる
    result = nn::svc::ConnectToNamedPort(&clientSession, DummyPortName);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &namedPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::AcceptSession(&serverSession, namedPort);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);

    // TEST
    //  セッションが解放されると、ConnectToPort でセッションが作成できるようになる
    result = nn::svc::ConnectToPort(&clientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(OutOfResourceTest, Process)
{
    nn::Result result;

    nn::svc::CreateProcessParameter param = {};
    SetDefaultParam(&param);

    nn::Bit32 flags[DefaultCapabilityFlagNum];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);

    nn::svc::Handle process;

    {
        ConsumeProcess consume;

        // TEST 121-76
        // Process 数が上限に達している状態で CreatePort を呼び出すと失敗する
        result = nn::svc::CreateProcess(&process, param, flags, DefaultCapabilityFlagNum);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
    }

    // TEST 121-77
    // Process が解放されれば、再度確保できるようになる
    result = nn::svc::CreateProcess(&process, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(process);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(OutOfResourceTest, Port)
{
    nn::Result result;

    // Port 数が上限に達している状態で CreatePort を呼び出すと失敗する
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;

    {
        ConsumePort consume;

        // TEST 112-29
        // Normal Session
        result = nn::svc::CreatePort(&serverPort, &clientPort, 1, false, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());

        // TEST 112-30
        // Light Session
        result = nn::svc::CreatePort(&serverPort, &clientPort, 1, true, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
    }

    // TEST 112-31
    // Normal Session
    result = nn::svc::CreatePort(&serverPort, &clientPort, 1, false, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(serverPort);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientPort);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 112-32
    // Light Session
    result = nn::svc::CreatePort(&serverPort, &clientPort, 1, true, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(serverPort);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientPort);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(OutOfResourceTest, Dma)
{
// 未実装
}

TEST(OutOfResourceTest, Debug)
{
    nn::Result result;

    nn::Bit32 flags[DefaultCapabilityFlagNum + 1];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum], 1 << 0); // EnableDebug

    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum + 1);

    nn::Bit64 processId;
    result = nn::svc::GetProcessId(&processId, process.GetHandle());
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle debugHandle;
    {
        ConsumeDebug consume;
        // TEST
        // Debug 数が上限に達している状態で DebugActiveProcess を呼び出すことはできない
        result = nn::svc::DebugActiveProcess(&debugHandle, processId);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
    }

    // TEST
    // Debug ハンドルが解放されると、DebugActiveProcess を呼び出すことできるようになる
    result = nn::svc::DebugActiveProcess(&debugHandle, processId);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(debugHandle);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(OutOfResourceTest, NamedPort)
{
    nn::Result result;

    nn::svc::Handle handle;
    {
        ConsumeNamedPort consume;

        // TEST 113-26
        // NamedPort 数が上限に達している状態で ManageNamedPort を呼び出すと失敗する
        result = nn::svc::ManageNamedPort(&handle, DummyPortName, 1);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
    }

    // TEST 113-27
    // NamedPort が解放されれば、再度確保できるようになる
    result = nn::svc::ManageNamedPort(&handle, DummyPortName, 1);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ManageNamedPort(&handle, DummyPortName, 0);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(OutOfResourceTest, LightSession)
{
    nn::Result result;
    int32_t index;

    // チェック用にポートを作成する
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    result = nn::svc::CreatePort(&serverPort, &clientPort, 1, true, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);
    AutoHandleClose cPortCloser(clientPort);

    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    {
        ConsumeLightSession consume;

        // TEST
        // Light Session 数が上限に達している状態で CreateSession を呼び出すと失敗する
        result = nn::svc::CreateSession(&serverSession, &clientSession, true, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());

        // TEST
        // Light Session 数が上限に達している状態で ConnectToPort を呼び出すと失敗する
        result = nn::svc::ConnectToPort(&serverSession, clientPort);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
    }

    // TEST
    // セッションが解放されると、CreateSession でセッションが作成できるようになる
    result = nn::svc::CreateSession(&serverSession, &clientSession, true, 0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);

    // TEST
    //  セッションが解放されると、ConnectToPort でセッションが作成できるようになる
    result = nn::svc::ConnectToPort(&clientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(OutOfResourceTest, MapDeviceAddressSpace)
{
    TestResourceEnv resourceEnv;
    TestDeviceAddressSpaceLeak leakTest;

    nn::Result result;
    uint64_t spaceSize = 0x80000000;
    nn::svc::Handle handle;
    nn::svc::MemoryInfo blockInfo;
    uint64_t deviceAddress = 0x400000;
    size_t outSize;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    result = nn::svc::AttachDeviceAddressSpace(DeviceName_SdCard, handle);
    ASSERT_RESULT_SUCCESS(result);
    AutoDetachDevice devCloser(handle, DeviceName_SdCard);

    uint64_t addr = reinterpret_cast<uint64_t>(g_Buffer);
    size_t size = sizeof(g_Buffer);
    {
        ConsumeMemoryControlBlock consume;

        // TEST 132-55
        // メモリ管理数が上限に達している状態で呼び出すことが出来ない

        GetMemoryInfo(&blockInfo, addr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);

        result = nn::svc::MapDeviceAddressSpace(&outSize,
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                addr, size, deviceAddress,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
    }

    // TEST 132-56
    // メモリ管理数が減ると、再度呼び出しができるようになる
    result = nn::svc::MapDeviceAddressSpace(&outSize,
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, size, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, outSize, deviceAddress);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == 0);
}

TEST(OutOfResourceTest, MapDeviceAddressSpaceAligned)
{
    TestResourceEnv resourceEnv;
    TestDeviceAddressSpaceLeak leakTest;

    nn::Result result;
    uint64_t spaceSize = 0x80000000;
    nn::svc::Handle handle;
    nn::svc::MemoryInfo blockInfo;
    uint64_t align = 0x400000;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    result = nn::svc::AttachDeviceAddressSpace(DeviceName_SdCard, handle);
    ASSERT_RESULT_SUCCESS(result);
    AutoDetachDevice devCloser(handle, DeviceName_SdCard);

    uint64_t addr = reinterpret_cast<uint64_t>(g_Buffer);
    uint64_t size = sizeof(g_Buffer);
    uint64_t deviceAddress = addr & (align - 1);

    {
        ConsumeMemoryControlBlock consume;

        // TEST 132-55
        // メモリ管理数が上限に達している状態で呼び出すことが出来ない

        GetMemoryInfo(&blockInfo, addr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);

        result = nn::svc::MapDeviceAddressSpaceAligned(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                addr, size, deviceAddress,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
    }

    // TEST 132-56
    // メモリ管理数が減ると、再度呼び出しができるようになる
    result = nn::svc::MapDeviceAddressSpaceAligned(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, size, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, size, deviceAddress);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == 0);
}

TEST(OutOfResourceTest, MapDeviceAddressSpaceByForce)
{
    TestResourceEnv resourceEnv;
    TestDeviceAddressSpaceLeak leakTest;

    nn::Result result;
    uint64_t spaceSize = 0x80000000;
    nn::svc::Handle handle;
    nn::svc::MemoryInfo blockInfo;
    uint64_t deviceAddress = 0x400000;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    result = nn::svc::AttachDeviceAddressSpace(DeviceName_SdCard, handle);
    ASSERT_RESULT_SUCCESS(result);
    AutoDetachDevice devCloser(handle, DeviceName_SdCard);

    uint64_t addr = reinterpret_cast<uint64_t>(g_Buffer);
    uint64_t size = sizeof(g_Buffer);

    {
        ConsumeMemoryControlBlock consume;
        // TEST 132-55
        // メモリ管理数が上限に達している状態で呼び出すことが出来ない

        GetMemoryInfo(&blockInfo, addr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);

        result = nn::svc::MapDeviceAddressSpaceByForce(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                addr, size, deviceAddress,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
    }

    // TEST 132-56
    // メモリ管理数が減ると、再度呼び出しができるようになる
    result = nn::svc::MapDeviceAddressSpaceByForce(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, size, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, size, deviceAddress);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == 0);
}

TEST(OutOfResourceTest, MapMemory)
{
    TestResourceEnv resourceEnv;
    TestMemoryControlBlockLeak leakTest;

    nn::Result result;
    nn::svc::Handle processHandle;
    uintptr_t addr = reinterpret_cast<uintptr_t>(g_Buffer);
    size_t size = sizeof(g_Buffer);

    {
        ConsumeMemoryControlBlock consume;
#ifdef ENABLE_PAGE_ROLLBACK
        // TEST 3-33
        // メモリ領域の管理数が上限に達していると失敗する
        result = nn::svc::MapMemory(g_FreeAreaBegin, addr, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
#endif // ENABLE_PAGE_ROLLBACK
    }

    // TEST 3-34
    // メモリ管理数が減ると、再度確保できるようになる
    result = nn::svc::MapMemory(g_FreeAreaBegin, addr, size);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::UnmapMemory(g_FreeAreaBegin, addr, size);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(OutOfResourceTest, MapProcessCodeMemory)
{
    TestResourceEnv resourceEnv;
    TestMemoryControlBlockLeak leakTest;

    nn::Result result;

    TestMemoryStateProcess stateProcess;
    nn::svc::Handle processHandle = stateProcess.GetHandle();

    size_t size = 0x1000;
    uintptr_t toAddr = g_FreeAreaBegin;
    uint64_t fromAddr;
    stateProcess.GetNormalArea(&fromAddr, size);

    {
        ConsumeMemoryControlBlock consume;

#ifdef ENABLE_PAGE_ROLLBACK
        // TEST 119-52
        // メモリ領域の管理数が上限に達していると失敗する
        result = nn::svc::MapProcessCodeMemory(
                processHandle, toAddr, fromAddr, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
#endif // ENABLE_PAGE_ROLLBACK
    }

    // TEST 119-53
    // メモリ管理が解放されれば、再び SVC を呼び出すことが出来る
    result = nn::svc::MapProcessCodeMemory(
            processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapProcessCodeMemory(
            processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(OutOfResourceTest, MapProcessMemory)
{
    TestResourceEnv resourceEnv;
    TestMemoryControlBlockLeak leakTest;

    nn::Result result;

    TestProcess process(1);

    uintptr_t toAddr = g_FreeAreaBegin;
    uintptr_t fromAddr = process.GetCodeAddress();
    size_t size = 0x1000;

    {
        ConsumeMemoryControlBlock consume;

        // TEST 116-52
        // メモリ領域の管理数が上限に達していると失敗する
        result = nn::svc::MapProcessMemory(
                toAddr, process.GetHandle(), fromAddr, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
    }

    // TEST 115-53
    // メモリ管理が解放されれば、再び SVC を呼び出すことが出来る
    result = nn::svc::MapProcessMemory(toAddr, process.GetHandle(), fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapProcessMemory(toAddr, process.GetHandle(), fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(OutOfResourceTest, MapSharedMemory)
{
    TestResourceEnv resourceEnv;
    TestSharedMemoryLeak leakTest;

    nn::Result result;
    nn::svc::Handle processHandle;

    nn::svc::Handle handle;
    size_t size = 0x1000;
    result = nn::svc::CreateSharedMemory(
            &handle, size,
            nn::svc::MemoryPermission_ReadWrite, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sharedCloser(handle);

    {
        ConsumeMemoryControlBlock consume;

        // TEST 17-53
        // メモリ領域の管理数が上限に達していると失敗する
        result = nn::svc::MapSharedMemory(handle, g_FreeAreaBegin,
                size, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
    }

    // TEST 17-54
    // メモリ領域の管理数が上限を下回れば、再度呼び出しが可能になる
    result = nn::svc::MapSharedMemory(handle, g_FreeAreaBegin,
            size, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapSharedMemory(handle, g_FreeAreaBegin, size);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(OutOfResourceTest, MapTransferMemory)
{
    TestTransferMemoryLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::MemoryInfo blockInfo;
    size_t size = 0x1000;
    const nn::svc::MemoryPermission ValidPerm[] = {
        nn::svc::MemoryPermission_None,
        nn::svc::MemoryPermission_Read,
        nn::svc::MemoryPermission_ReadWrite,
    };

    for (int32_t i = 0; i < static_cast<int32_t>(sizeof(ValidPerm) / sizeof(nn::svc::MemoryPermission)); i++)
    {
        TestResourceEnv resourceEnv;
        nn::svc::MemoryPermission perm = ValidPerm[i];
        result = nn::svc::CreateTransferMemory(
                &handle, reinterpret_cast<uintptr_t>(g_Buffer), size, perm);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose autoCloser(handle);

        uintptr_t toAddr = g_FreeAreaBegin;

        GetMemoryInfo(&blockInfo, toAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.attribute == 0);

        {
            ConsumeMemoryControlBlock consume;

            // TEST 127-30
            // メモリ管理数が上限に達している状態で呼び出すことが出来ない
            result = nn::svc::MapTransferMemory(handle, toAddr, size, perm);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());

            GetMemoryInfo(&blockInfo, toAddr);
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
            ASSERT_TRUE(blockInfo.attribute == 0);
        }

        // TEST 127-31
        // メモリ管理数が減ると、再度呼び出しができるようになる
        result = nn::svc::MapTransferMemory(handle, toAddr, size, perm);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, toAddr);
        if (perm == nn::svc::MemoryPermission_None)
        {
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Transfered);
        }
        else
        {
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_SharedTransfered);
        }
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);

        result = nn::svc::UnmapTransferMemory(handle, toAddr, size);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, toAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.attribute == 0);
    }
}

#ifdef ENABLE_PAGE_ROLLBACK
#if 0
/*
     このケースはカーネルとして対応不可能なので、テストしない
 */
TEST(OutOfResourceTest, UnmapDeviceAddressSpace)
{
    int32_t beforeNum;
    int32_t afterNum;
    TestResourceEnv resourceEnv;

    CalcMaxMemoryControlBlockNum(&beforeNum);


    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    uint64_t spaceSize = 0x80000000;
    nn::svc::Handle handle;
    nn::svc::MemoryInfo blockInfo;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    uint64_t align = 0x400000;
    uint64_t addr = reinterpret_cast<uint64_t>(g_Buffer);
    uint64_t mapSize = 0x1000;
    uint64_t deviceAddress = addr & (align - 1);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);

    result = nn::svc::MapDeviceAddressSpaceAligned(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, mapSize, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle processHandle;
    int32_t numBlock;
    ConsumeMemoryControlBlock(&processHandle, &numBlock);
    ASSERT_TRUE(numBlock > 0);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

    // 133-39
    // メモリ管理数が上限に達している状態で呼び出すことが出来ない
    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, mapSize, deviceAddress);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);

    // 133-40
    // メモリ管理数が減ると、再度呼び出しができるようになる
    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, mapSize, deviceAddress);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == 0);

    CalcMaxMemoryControlBlockNum(&afterNum);
    ASSERT_TRUE(beforeNum == afterNum);
}
#endif
#endif // ENABLE_PAGE_ROLLBACK

TEST(OutOfResourceTest, UnmapMemory)
{
    TestResourceEnv resourceEnv;
    TestMemoryControlBlockLeak leakTest;

    nn::Result result;
    nn::svc::Handle processHandle;
    uintptr_t addr = reinterpret_cast<uintptr_t>(g_Buffer);
    size_t size = sizeof(g_Buffer);

    result = nn::svc::MapMemory(g_FreeAreaBegin, addr, size);
    ASSERT_RESULT_SUCCESS(result);

    {
        ConsumeMemoryControlBlock consume;

        // TEST 4-22
        // メモリ領域の管理数が上限に達していると失敗する
        result = nn::svc::UnmapMemory(g_FreeAreaBegin, addr, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
    }

    // TEST 4-23
    // メモリ管理数が減ると、再度呼び出しが出来るようになる
    result = nn::svc::UnmapMemory(g_FreeAreaBegin, addr, size);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(OutOfResourceTest, UnmapProcessCodeMemory)
{
    TestResourceEnv resourceEnv;
    TestMemoryControlBlockLeak leakTest;

    nn::Result result;

    TestMemoryStateProcess stateProcess;
    nn::svc::Handle processHandle = stateProcess.GetHandle();

    size_t size = 0x1000;
    uintptr_t toAddr = g_FreeAreaBegin;
    uint64_t fromAddr;
    stateProcess.GetNormalArea(&fromAddr, size);
    result = nn::svc::MapProcessCodeMemory(
            processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    {
        ConsumeMemoryControlBlock consume;

#ifdef ENABLE_PAGE_ROLLBACK
        // メモリ領域の管理数が上限に達していると失敗する
        result = nn::svc::UnmapProcessCodeMemory(
                processHandle, toAddr, fromAddr, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
#endif // ENABLE_PAGE_ROLLBACK
    }

    // メモリ管理が解放されれば、再び SVC を呼び出すことが出来る
    result = nn::svc::UnmapProcessCodeMemory(
            processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(OutOfResourceTest, UnmapProcessMemory)
{
    TestResourceEnv resourceEnv;
    TestMemoryControlBlockLeak leakTest;

    nn::Result result;

    TestProcess process(1);

    uintptr_t toAddr = g_FreeAreaBegin;
    uintptr_t fromAddr = process.GetCodeAddress();
    size_t size = 0x1000;

    result = nn::svc::MapProcessMemory(
            toAddr, process.GetHandle(), fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    {
        ConsumeMemoryControlBlock consume;

        // メモリ領域の管理数が上限に達していると失敗する
        result = nn::svc::UnmapProcessMemory(
                toAddr, process.GetHandle(), fromAddr, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
    }

    // メモリ管理が解放されれば、再び SVC を呼び出すことが出来る
    result = nn::svc::UnmapProcessMemory(
            toAddr, process.GetHandle(), fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(OutOfResourceTest, UnmapSharedMemory)
{
    TestResourceEnv resourceEnv;
    TestSharedMemoryLeak leakTest;

    nn::Result result;
    nn::svc::Handle processHandle;

    nn::svc::Handle handle;
    size_t size = 0x1000;
    result = nn::svc::CreateSharedMemory(
            &handle, size,
            nn::svc::MemoryPermission_ReadWrite, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sharedCloser(handle);

    result = nn::svc::MapSharedMemory(handle, g_FreeAreaBegin,
            size, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    {
        ConsumeMemoryControlBlock consume;

        // TEST 18-26
        // メモリ領域の管理数が上限に達していると失敗する
        result = nn::svc::UnmapSharedMemory(handle, g_FreeAreaBegin, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
    }

    // TEST 18-27
    // メモリ領域の管理数が上限を下回れば、再度呼び出しが可能になる
    result = nn::svc::UnmapSharedMemory(handle, g_FreeAreaBegin, size);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(OutOfResourceTest, UnmapTransferMemory)
{
    TestTransferMemoryLeak leakTest;
    TestResourceEnv resourceEnv;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::MemoryInfo blockInfo;
    const nn::svc::MemoryPermission ValidPerm[] = {
        nn::svc::MemoryPermission_None,
        nn::svc::MemoryPermission_Read,
        nn::svc::MemoryPermission_ReadWrite,
    };

    for (int32_t i = 0; i < static_cast<int32_t>(sizeof(ValidPerm) / sizeof(nn::svc::MemoryPermission)); i++)
    {
        nn::svc::MemoryPermission perm = ValidPerm[i];
        uintptr_t bufferAddr = reinterpret_cast<uintptr_t>(g_Buffer);
        size_t size = sizeof(g_Buffer);

        GetMemoryInfo(&blockInfo, bufferAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);

        result = nn::svc::CreateTransferMemory(&handle, bufferAddr, size, perm);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose transferCloser(handle);

        uintptr_t addr = g_FreeAreaBegin;
        GetMemoryInfo(&blockInfo, addr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.attribute == 0);

        result = nn::svc::MapTransferMemory(handle, addr, size, perm);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, addr);
        if (perm == nn::svc::MemoryPermission_None)
        {
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Transfered);
        }
        else
        {
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_SharedTransfered);
        }
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);

        {
            ConsumeMemoryControlBlock consume;

            // TEST 128-29
            // メモリ管理数が上限に達している状態で呼び出すことが出来ない
            result = nn::svc::UnmapTransferMemory(handle, addr, size);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());

            GetMemoryInfo(&blockInfo, addr);
            if (perm == nn::svc::MemoryPermission_None)
            {
                ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Transfered);
            }
            else
            {
                ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_SharedTransfered);
            }
            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
            ASSERT_TRUE(blockInfo.attribute == 0);
        }

        // TEST 128-30
        // メモリ管理数が減ると、再度呼び出しができるようになる
        result = nn::svc::UnmapTransferMemory(handle, addr, size);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, addr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.attribute == 0);
    }
}

TEST(OutOfResourceTest, SetHeapSize)
{
    TestResourceEnv resourceEnv;
    TestMemoryControlBlockLeak leakTest;

    nn::Result result;
    nn::svc::Handle processHandle;
    uintptr_t addr;

    {
        ConsumeMemoryControlBlock consume;

        // TEST 1-16
        // メモリ領域の管理数が上限に達していると失敗する
        result = nn::svc::SetHeapSize(&addr, HeapAlign);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
    }

    // TEST 1-17
    // メモリ管理数が減ると、再度確保できるようになる
    result = nn::svc::SetHeapSize(&addr, HeapAlign);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::SetHeapSize(&addr, 0);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(OutOfResourceTest, SetMemoryAttribute)
{
    TestResourceEnv resourceEnv;
    TestMemoryControlBlockLeak leakTest;

    nn::Result result;
    uintptr_t addr = reinterpret_cast<uintptr_t>(g_Buffer);
    size_t size = 0x1000;
    nn::Bit32 attribute = nn::svc::MemoryAttribute_Uncached;
    nn::Bit32 mask = nn::svc::MemoryAttribute_Uncached;

    {
        ConsumeMemoryControlBlock consume;

        // TEST 136-33
        // メモリ管理数が上限に達していると呼び出しに失敗する
        result = nn::svc::SetMemoryAttribute(addr, size, mask, attribute);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
    }

    // TEST 136-34
    // メモリ管理数が上限を下回ると呼び出しに成功する
    result = nn::svc::SetMemoryAttribute(addr, size, mask, attribute);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::SetMemoryAttribute(addr, size, mask, 0);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(OutOfResourceTest, SetMemoryPermission)
{
    TestResourceEnv resourceEnv;
    TestMemoryControlBlockLeak leakTest;

    nn::Result result;
    uintptr_t addr = reinterpret_cast<uintptr_t>(g_Buffer);
    size_t size = sizeof(g_Buffer);

    {
        ConsumeMemoryControlBlock consume;

        // TEST 2-57
        // メモリ領域の管理数が上限に達していると失敗する
        result = nn::svc::SetMemoryPermission(addr, size, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
    }

    // TEST 2-58
    // メモリ管理数が減ると、再度確保できるようになる
    result = nn::svc::SetMemoryPermission(addr, size, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::SetMemoryPermission(addr, size, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(OutOfResourceTest, SetProcessMemoryPermission)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestResourceEnv resourceEnv;

    {
        int32_t maxNumPage = NN_KERN_SLAB_OBJ_NUM_SYS_MEMORY_BLOCK + 1;
        TestProcess process(maxNumPage);
        nn::svc::Handle processHandle = process.GetHandle();

        int32_t numMemoryBlock = 0;
        size_t size = 0x1000;
        uintptr_t addr = process.GetCodeAddress();
        uintptr_t endAddr = addr + process.GetCodeAreaSize();
        nn::svc::MemoryPermission permissions[] = {
            nn::svc::MemoryPermission_None,
            nn::svc::MemoryPermission_Read,
            nn::svc::MemoryPermission_ReadWrite,
            nn::svc::MemoryPermission_ReadExecute,
        };
        int32_t permSize = sizeof(permissions) / sizeof(nn::svc::MemoryPermission);

        // 出来るだけメモリ管理をする
        while((addr + size) < endAddr)
        {
            result = nn::svc::SetProcessMemoryPermission(
                    processHandle, addr, size, permissions[numMemoryBlock % permSize]);
            if (result.IsFailure())
            {
                // TEST 115-36
                // メモリ領域の管理数が上限に達していると失敗する
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
                break;
            }
            numMemoryBlock++;
            addr += size;
        }
        ASSERT_TRUE((addr + size) < endAddr);

        result = nn::svc::CloseHandle(processHandle);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 115-37
    // メモリ管理が解放されれば、再び SVC を呼び出すことが出来る
    TestProcess process(1);
    nn::svc::Handle processHandle = process.GetHandle();

    size_t size = 0x1000;
    uintptr_t addr = process.GetCodeAddress();
    result = nn::svc::SetProcessMemoryPermission(
            processHandle, addr, size, nn::svc::MemoryPermission_ReadExecute);
    ASSERT_RESULT_SUCCESS(result);
}

#endif // ENABLE_OUT_OF_RESOURCE_TEST

