﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestProcess.h"
#include "util_TestMemory.h"
#include "util_MemoryState.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_BaseId.autogen.h>
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <nn/TargetConfigs/build_Compiler.h>
#include <cstring>

extern "C" void nnMain();

namespace {
} // namespace

TEST(MapProcessCodeMemory, HandleTest)
{
    nn::Result result;

    TestMemoryStateProcess stateProcess;
    nn::svc::Handle processHandle = stateProcess.GetHandle();

    size_t size = 0x1000;
    uint64_t toAddr = g_FreeAreaBegin;
    uint64_t fromAddr;
    stateProcess.GetNormalArea(&fromAddr, size);

    // TEST 119-2
    // 実行中のプロセスのハンドルを受け付ける
    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::TerminateProcess(processHandle);
    ASSERT_RESULT_SUCCESS(result);

    WaitProcess(processHandle);

    stateProcess.SetDead();

    // TEST 119-3
    // 実行後のプロセスのハンドルを受け付ける
    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 119-4
    // プロセスの擬似ハンドルを受け付けない
    result = nn::svc::MapProcessCodeMemory(
            nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, toAddr, fromAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 119-5
    // INVALID_HANDLE_VALUE を受け付けない
    result = nn::svc::MapProcessCodeMemory(
            nn::svc::INVALID_HANDLE_VALUE, toAddr, fromAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 119-6
    // スレッドの擬似ハンドルを受け付けない
    result = nn::svc::MapProcessCodeMemory(
            nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, toAddr, fromAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 119-7
    // Close したプロセスのハンドルを受け付けない
    result = nn::svc::MapProcessCodeMemory(
            processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

TEST(MapProcessCodeMemory, AddrTest)
{
    nn::Result result;

    TestMemoryStateProcess stateProcess;
    nn::svc::Handle processHandle = stateProcess.GetHandle();

    size_t size = 0x1000;
    uint64_t toAddr = g_FreeAreaBegin;
    uint64_t fromAddr;
    stateProcess.GetNormalArea(&fromAddr, size);

    // TEST 119-8
    // toAddr のアラインメントが4KBに揃っていると成功する
    // TEST 119-10
    // fromAddr のアラインメントが4KBに揃っていると成功する
    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 119-9
    // toAddr のアラインメントが4KBに揃っていないと失敗する
    for (uint64_t i = toAddr + 1; i < toAddr + 0x1000; i++)
    {
        result = nn::svc::MapProcessCodeMemory(processHandle, i, fromAddr, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());
    }

    // TEST 119-11
    // fromAddr のアラインメントが4KBに揃っていないと失敗する
    for (uint64_t i = fromAddr + 1; i < fromAddr + 0x1000; i++)
    {
        result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, i, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());
    }
}

TEST(MapProcessCodeMemory, SizeTest)
{
    nn::Result result;

    TestMemoryStateProcess stateProcess;
    nn::svc::Handle processHandle = stateProcess.GetHandle();

    size_t size = 0x1000;
    uint64_t toAddr = g_FreeAreaBegin;
    uint64_t fromAddr;
    stateProcess.GetNormalArea(&fromAddr, size);

    // TEST 119-12
    // size のアラインメントが4KBに揃っていると成功する
    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 119-13
    // size のアラインメントが4KBに揃っていないと失敗する
    for (uint64_t i = 1; i < 0x1000; i++)
    {
        result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, i);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());
    }

    // TEST 119-14
    // size は 0 を受け付けない
    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());
}

TEST(MapProcessCodeMemory, ToAddrAreaTest)
{
    nn::Result result;

    TestMemoryStateProcess stateProcess;
    nn::svc::Handle processHandle = stateProcess.GetHandle();

    uint64_t mapSize = 0x1000;

    uint64_t normalAreaSize = mapSize * 3;
    uint64_t heapAddr;
    stateProcess.GetNormalArea(&heapAddr, normalAreaSize);

    uint64_t fromAddr = heapAddr;
    uint64_t toAddr;
    uint64_t size;

    // 許可されていないメモリ状態を受け付けない
    for (int i = 0; i < NumTestMemoryState; i++)
    {
        nn::svc::MemoryState state = AllMemoryState[i];
        if (state == nn::svc::MemoryState_Free)
        {
            continue;
        }

        CheckProcessMemory(
                processHandle,
                fromAddr,
                nn::svc::MemoryState_Normal,
                nn::svc::MemoryPermission_ReadWrite,
                0);

        stateProcess.GetMemoryArea(&toAddr, &size, state);

        if (size >= mapSize)
        {
            result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, mapSize);
            if (!IsInProcessAslrRegion(processHandle, toAddr))
            {
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());
            }
            else
            {
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
            }
        }

        stateProcess.FreeMemoryArea();
    }

    CheckProcessMemory(
            processHandle,
            fromAddr,
            nn::svc::MemoryState_Normal,
            nn::svc::MemoryPermission_ReadWrite,
            0);

    toAddr = g_FreeAreaBegin + mapSize;
    CheckProcessMemory(
            processHandle,
            toAddr,
            nn::svc::MemoryState_Free,
            nn::svc::MemoryPermission_None,
            0);

    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, mapSize);
    ASSERT_RESULT_SUCCESS(result);

    CheckProcessMemory(
            processHandle,
            fromAddr,
            nn::svc::MemoryState_Normal,
            nn::svc::MemoryPermission_None,
            nn::svc::MemoryAttribute_Locked);

    CheckProcessMemory(
            processHandle,
            toAddr,
            nn::svc::MemoryState_AliasCode,
            nn::svc::MemoryPermission_None,
            0);

    fromAddr += mapSize;

    // TEST 119-30
    // toAddr/size で示す領域 のメモリ状態が一様でないと失敗する (前方)
    toAddr = g_FreeAreaBegin + mapSize;
    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, mapSize * 2);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 119-31
    // toAddr/size で示す領域 のメモリ状態が一様でないと失敗する (後方)
    toAddr = g_FreeAreaBegin;
    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, mapSize * 2);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    toAddr = g_FreeAreaBegin + mapSize;
    fromAddr = heapAddr;
    result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr, fromAddr, mapSize);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 119-32
    // オーバーフローする組み合わせを指定できない
    mapSize = static_cast<uint64_t>(-0x1000);
    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, mapSize);
    ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() ||
            result <= nn::svc::ResultInvalidRegion());

    // TEST 119-51
    // toAddr/size で示す領域 のメモリ状態がAslr に含まれていないと失敗する
    // Reserved 領域
    mapSize = 0x1000;
    {
        nn::Bit64 reservedAddr;
        result = nn::svc::GetInfo(
                &reservedAddr, nn::svc::InfoType_ReservedRegionAddress,
                processHandle, 0);
        NN_ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::MapProcessCodeMemory(processHandle, reservedAddr, fromAddr, mapSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());
    }

    // Large 領域
    {
        bool check = GetProcessLargeFreeArea(&toAddr, mapSize, processHandle, stateProcess.Is64Bit());
#if defined NN_BUILD_CONFIG_CPU_ARM_V7A
        ASSERT_TRUE(!check);
#else
        ASSERT_TRUE(check);
#endif
        if(check)
        {
#ifdef FIX_MAP_PROCESS_CODE_MEMORY_LARGE_AREA
            result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, mapSize);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());
#endif
        }
    }
} // NOLINT(readability/fn_size)

TEST(MapProcessCodeMemory, FromAddrAreaTest)
{
    nn::Result result;

    TestMemoryStateProcess stateProcess;
    nn::svc::Handle processHandle = stateProcess.GetHandle();

    uint64_t mapSize = 0x1000;
    uint64_t normalAreaSize = mapSize * 3;

    uint64_t heapAddr;
    stateProcess.GetNormalArea(&heapAddr, normalAreaSize);

    uint64_t fromAddr;
    uint64_t toAddr = g_FreeAreaEnd - mapSize * 3;
    uint64_t size;

    CheckProcessMemory(
            processHandle,
            toAddr,
            nn::svc::MemoryState_Free,
            nn::svc::MemoryPermission_None,
            0);

    // TEST 119-34
    // fromAddr/size で示す領域が MemoryState_Normal だと成功する
    fromAddr = heapAddr + mapSize;
    CheckProcessMemory(
            processHandle,
            fromAddr,
            nn::svc::MemoryState_Normal,
            nn::svc::MemoryPermission_ReadWrite,
            0);

    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, mapSize);
    ASSERT_RESULT_SUCCESS(result);

    CheckProcessMemory(
            processHandle,
            fromAddr,
            nn::svc::MemoryState_Normal,
            nn::svc::MemoryPermission_None,
            nn::svc::MemoryAttribute_Locked);

    CheckProcessMemory(
            processHandle,
            toAddr,
            nn::svc::MemoryState_AliasCode,
            nn::svc::MemoryPermission_None,
            0);

    toAddr += mapSize;

    // 許可されていないメモリ状態を受け付けない
    for (int i = 0; i < NumTestMemoryState; i++)
    {
        nn::svc::MemoryState state = AllMemoryState[i];
        if (state == nn::svc::MemoryState_Normal)
        {
            continue;
        }

        CheckProcessMemory(
                processHandle,
                toAddr,
                nn::svc::MemoryState_Free,
                nn::svc::MemoryPermission_None,
                0);

        stateProcess.GetMemoryArea(&fromAddr, &size, state);

        if (size >= mapSize)
        {
            result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, mapSize);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

            CheckProcessMemory(
                    processHandle,
                    toAddr,
                    nn::svc::MemoryState_Free,
                    nn::svc::MemoryPermission_None,
                    0);
        }

        stateProcess.FreeMemoryArea();
    }

    // TEST 119-48
    // fromAddr/size で示す領域 のメモリ状態が一様でないと失敗する (前方)
    fromAddr = heapAddr;
    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, mapSize * 2);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 119-49
    // fromAddr/size で示す領域 のメモリ状態が一様でないと失敗する (後方)
    fromAddr = heapAddr + mapSize;
    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, mapSize * 2);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    toAddr -= mapSize;
    result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr, fromAddr, mapSize);

    // TEST 119-50
    // オーバーフローする組み合わせを指定できない
    fromAddr = heapAddr;
    mapSize = static_cast<uint64_t>(-0x1000);
    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, mapSize);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // fromAddr/size で示す領域にメモリ属性が付加されている失敗する
    {
        mapSize = 0x1000;
        stateProcess.SetUncached(fromAddr, mapSize);

        CheckProcessMemory(
                processHandle,
                fromAddr,
                nn::svc::MemoryState_Normal,
                nn::svc::MemoryPermission_ReadWrite,
                nn::svc::MemoryAttribute_Uncached);

        result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, mapSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        stateProcess.UnsetUncached(fromAddr, mapSize);
    }
} // NOLINT(readability/fn_size)

