﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"

namespace {

const int32_t BeginValue = 1;
const int32_t MiddleValue = 3;
const int32_t EndValue = 7;
const int32_t MapSize = 0x1000;

char g_Buffer[0x2000] __attribute__((aligned(0x1000)));

void WriteData(uintptr_t addr, size_t size)
{
    int32_t *ptr = reinterpret_cast<int32_t*>(addr);
    int32_t maxIndex = size / sizeof(int32_t);
    // 全部を読み書きしていると時間が掛かるので、３か所チェックする
    ptr[0] = BeginValue;
    ptr[maxIndex / 8] = MiddleValue;
    ptr[maxIndex - 1] = EndValue;
}

void CheckReadData(uintptr_t addr, size_t size)
{
    int32_t *ptr = reinterpret_cast<int32_t*>(addr);
    int32_t maxIndex = size / sizeof(int32_t);
    // 全部を読み書きしていると時間が掛かるので、３か所チェックする
    ASSERT_TRUE(ptr[0] == BeginValue);
    ASSERT_TRUE(ptr[maxIndex / 8] == MiddleValue);
    ASSERT_TRUE(ptr[maxIndex - 1] == EndValue);
}

}

/* TEST 3-25 */
/* マップした領域のデータが読み書きできることを確認する */
TEST(MapMemoryTest, ReadWriteTest)
{
    nn::Result result;
    uintptr_t bssAddr;
    uintptr_t heapPtr;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;

    bssAddr = reinterpret_cast<uintptr_t>(g_Buffer);

    result = nn::svc::SetHeapSize(&heapPtr, HeapAlign);
    ASSERT_RESULT_SUCCESS(result);

    // BSS領域が Normal State であることを確認する
    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, bssAddr);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);

    // ヒープ領域が Normal State であることを確認する
    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, heapPtr);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);

    // FreeArea領域が 空いていて、Free State であることを確認する
    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, g_FreeAreaBegin);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
    ASSERT_TRUE(g_FreeAreaEnd - 1 <= blockInfo.baseAddress + blockInfo.size - 1);

    result = nn::svc::MapMemory(g_FreeAreaBegin, heapPtr, MapSize);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, g_FreeAreaBegin);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Stack);
    ASSERT_TRUE(blockInfo.attribute == 0);

    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, heapPtr);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_Locked);

    // マップ先で書き込み
    WriteData(g_FreeAreaBegin, MapSize);

    result = nn::svc::UnmapMemory(g_FreeAreaBegin, heapPtr, MapSize);
    ASSERT_RESULT_SUCCESS(result);

    // マップ元で確認
    CheckReadData(heapPtr, MapSize);

    // マップ元で書き込み
    WriteData(bssAddr, MapSize);

    result = nn::svc::MapMemory(g_FreeAreaBegin, bssAddr, MapSize);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, g_FreeAreaBegin);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Stack);
    ASSERT_TRUE(blockInfo.attribute == 0);

    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, bssAddr);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_Locked);

    // マップ先で確認
    CheckReadData(g_FreeAreaBegin, MapSize);

    result = nn::svc::UnmapMemory(g_FreeAreaBegin, bssAddr, MapSize);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::SetHeapSize(&heapPtr, 0);
    ASSERT_RESULT_SUCCESS(result);
}

