﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestLoader.h"
#include "test_ManageNamedPort.h"
#include <nn/svc/ipc/svc_SessionMessage.h>

extern char BinManageNamedPort_begin[];
extern char BinManageNamedPort_end[];

namespace {

char g_Stack[DefaultStackSize] __attribute__((aligned(0x1000)));

void NamedPortClientThread(uintptr_t arg)
{
    NN_UNUSED(arg);
    AutoThreadExit autoExit;
    nn::Result result;

    nn::svc::Handle clientSession;
    result = nn::svc::ConnectToNamedPort(&clientSession, DelPortName);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose cSessionCloser(clientSession);

    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();

    result = nn::svc::SendSyncRequest(clientSession);
    ASSERT_RESULT_SUCCESS(result);
}

} // namespace

TEST(ManageNamedPort, ReAssignFromOtherProcess)
{
    uintptr_t begin = reinterpret_cast<uintptr_t>(BinManageNamedPort_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinManageNamedPort_end);

    TestLoader loader(BinManageNamedPort_begin, end - begin);

    TestNamedPortLeak leakTest;
    nn::Result result;

    nn::svc::Handle process;
    loader.SpawnProcess(&process);
    loader.StartProcess(process);
    AutoHandleClose processCloser(process);

    nn::svc::Handle serverPort;
    result = nn::svc::ManageNamedPort(&serverPort, PortName, 1);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle serverSession;
    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);

    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();

    result = nn::svc::ReplyAndReceive(
            &index, &serverSession, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle processHandle;
    {
        nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
        nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
        NN_ASSERT(ipcHeader.GetTag() == 0x0001);
        NN_ASSERT(ipcHeader.GetPointerNum() == 0);
        NN_ASSERT(ipcHeader.GetSendNum() == 0);
        NN_ASSERT(ipcHeader.GetReceiveNum() == 0);
        NN_ASSERT(ipcHeader.GetExchangeNum() == 0);
        NN_ASSERT(ipcHeader.GetRawNum() == 0);
        NN_ASSERT(ipcHeader.GetReceiveListNum() == 0);
        NN_ASSERT(ipcHeader.GetSpecialNum() == 1);
        NN_ASSERT(!ipcSpecial.GetProcessIdFlag());
        NN_ASSERT(ipcSpecial.GetCopyHandleNum() == 1);
        NN_ASSERT(ipcSpecial.GetMoveHandleNum() == 0);

        // Copy Handle
        {
            int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);
            processHandle = ipcMsg.GetHandle(offset);
        }
    }

    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceive(
            &index, &serverSession, 0, serverSession, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    result = nn::svc::TerminateProcess(processHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle delPort;
    result = nn::svc::ManageNamedPort(&delPort, DelPortName, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ManageNamedPort(&delPort, DelPortName, 1);
    ASSERT_RESULT_SUCCESS(result);

    // Client Session のスレッド作成
    uintptr_t pc = reinterpret_cast<uintptr_t>(NamedPortClientThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t param = 0;
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

    TestThread thread(pc, param, sp, priority, idealCore);
    thread.Start();

    // DelPort 用のセッションを作成し、クライアントとやり取りをする
    {
        result = nn::svc::WaitSynchronization(&index, &delPort, 1, -1);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose delPortCloser(delPort);

        nn::svc::Handle delServerSession;
        result = nn::svc::AcceptSession(&delServerSession, delPort);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose delServerSessionCloser(delServerSession);

        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceive(
                &index, &delServerSession, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);

        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceive(
                &index, &delServerSession, 0, delServerSession, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }

    thread.Wait();

    result = nn::svc::ManageNamedPort(&delPort, DelPortName, 0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::ManageNamedPort(&serverPort, PortName, 0);
    ASSERT_RESULT_SUCCESS(result);
}

