﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Kernel.h>

namespace {

const size_t StackSize = 0x2000;

volatile uintptr_t g_SavedParam0;
const int64_t SleepTime = 100 * 1000 * 1000;
char g_Buffers[2][DefaultStackSize] __attribute__((aligned(0x1000)));

void TestThreadFunction1(uintptr_t arg)
{
    NN_UNUSED(arg);
    AutoThreadExit autoExit;
    while (g_SavedParam0 == 0)
    {
        nn::svc::SleepThread(SleepTime);
    }
}

void CreateTestThreads(uintptr_t heapPtr, size_t heapSize, int32_t threadMax, int32_t *outNumThreadSlab, bool doCheck)
{
    nn::svc::Handle *childHandles = NULL;
    int32_t threadNum = 0;
    nn::Result result = nn::ResultSuccess();
    int32_t priority = 32;
    int32_t idealCore = 0;
    uintptr_t pc = 0;
    uintptr_t sp = 0;
    int32_t beforeThreadNum;
    int32_t afterThreadNum;

    /*
       Capability を作成するときの、flag によってスレッドが生成出来る数の最大値が決定する。
       しかし、SLAB が足りなくても生成できないので、SLAB < flag の時は最大値が変動しうる。
       現状では、SLAB < flag のケースはない。
     */
    ASSERT_TRUE(heapPtr != 0);
    ASSERT_TRUE(heapSize > sizeof(nn::svc::Handle) * (threadMax + 1) + StackSize * (threadMax + 1));

    childHandles = reinterpret_cast<nn::svc::Handle*>(heapPtr);
    pc = reinterpret_cast<uintptr_t>(TestThreadFunction1);

    // スレッドを作れるだけ作る
    g_SavedParam0 = 0;
    for (threadNum = 0; threadNum <= threadMax; threadNum++)
    {
        sp = heapPtr + heapSize - StackSize * threadNum;
        result = nn::svc::CreateThread(&childHandles[threadNum], pc, 0, sp, priority, idealCore);
        if (result.IsFailure())
        {
            break;
        }

        result = nn::svc::StartThread(childHandles[threadNum]);
        ASSERT_RESULT_SUCCESS(result);
    }

    if (doCheck)
    {
        // 上限値を超えた時にエラーになっているかを調べる
        ASSERT_TRUE(threadNum == threadMax);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
    }

    if (outNumThreadSlab != NULL)
    {
        *outNumThreadSlab = threadNum;
    }

    g_SavedParam0 = 1;
    result = nn::svc::GetThreadList(&beforeThreadNum, NULL, 0, nn::svc::INVALID_HANDLE_VALUE);
    ASSERT_RESULT_SUCCESS(result);
    for (uint32_t i = 0; i < static_cast<uint32_t>(threadNum); i++)
    {
        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &childHandles[i], 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(childHandles[i]);
        ASSERT_RESULT_SUCCESS(result);
    }

    // スレッドのインスタンスがワーカースレッドによって解放されるのを待つ
    afterThreadNum = beforeThreadNum;
    while (threadNum > (beforeThreadNum - afterThreadNum))
    {
        result = nn::svc::GetThreadList(&afterThreadNum, NULL, 0, nn::svc::INVALID_HANDLE_VALUE);
        ASSERT_RESULT_SUCCESS(result);
        nn::svc::SleepThread(SleepTime);
    }
    g_SavedParam0 = 0;
}

int32_t g_SequenceCount = -1;

void LowPriorityThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::svc::Handle* handle = reinterpret_cast<nn::svc::Handle*>(arg);

    NN_ASSERT(g_SequenceCount == 1);
    g_SequenceCount = 2;

    int32_t index;
    nn::Result result = nn::svc::WaitSynchronization(&index, &*handle, 1, SleepTime);
    NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    g_SequenceCount = 3;
}

void HighPriorityThread(uintptr_t arg)
{
    NN_UNUSED(arg);
    AutoThreadExit autoExit;

    ASSERT_TRUE(g_SequenceCount == 0);
    g_SequenceCount = 1;

    // 低優先度のスレッドに移る
    while(g_SequenceCount != 2)
    {
        nn::svc::SleepThread(SleepTime);
    }

    // タイムアウトになる時間分ビジーループする
    volatile int64_t begin = nn::svc::GetSystemTick();
    for(;;)
    {
        volatile int64_t end = nn::svc::GetSystemTick();
        if (((end - begin) / (NN_HW_TICKS_PER_SECOND / (1000 * 1000)) * 1000) >= SleepTime)
        {
            break;
        }
    }

    NN_ASSERT(g_SequenceCount == 2);

    // 低優先度のスレッドが終了するまで待つ
    while(g_SequenceCount != 3)
    {
        nn::svc::SleepThread(SleepTime);
    }
}

} // namespace

/* 大量にスレッドを作成し、上限値まで作る*/
/* TEST: 7-2, TEST: 7-3 */
TEST(CreateThreadTest, MaxNumberThreadTest)
{
    TestThreadLeak leakTest;
    nn::Result result;
    size_t heapSize;
    uintptr_t heapPtr;
    int32_t maxNumThread;

    heapSize =
        (sizeof(nn::svc::Handle) * NN_KERN_SLAB_OBJ_NUM_THREAD + StackSize * NN_KERN_SLAB_OBJ_NUM_THREAD + HeapAlign - 1)
        & ~(HeapAlign - 1);
    result = nn::svc::SetHeapSize(&heapPtr, heapSize);
    ASSERT_RESULT_SUCCESS(result);

    CreateTestThreads(heapPtr, heapSize, NN_KERN_SLAB_OBJ_NUM_THREAD, &maxNumThread, false);
    ASSERT_TRUE(maxNumThread < NN_KERN_SLAB_OBJ_NUM_THREAD);
    CreateTestThreads(heapPtr, heapSize, maxNumThread, NULL, true);

    result = nn::svc::SetHeapSize(&heapPtr, 0);
    ASSERT_RESULT_SUCCESS(result);
}

#ifdef EXHAUST_HEAP_TEST
/*
   Heap を使い切る
   現在、Heap を使い切って、なおかつ、スレッドを大量に作ると、
   カーネルのヒープが足りなくなってカーネルパニックを起こします。
   このケースは現在考慮にいれていないため、テストも実施しません。
 */
/* TEST: 7-5 */
TEST(CreateThreadTest, ExhaustHeapTest)
{
    TestThreadLeak leakTest;
    nn::Result result;
    size_t heapSize = g_HeapAreaEnd - g_HeapAreaBegin + 1;
    uintptr_t heapPtr;
    int32_t maxNumThread;
    int32_t numThread;

    // Heap を使い切る
    ASSERT_TRUE((heapSize & (HeapAlign - 1)) == 0);
    result = nn::svc::SetHeapSize(&heapPtr, heapSize);
    if (result.IsFailure())
    {
        heapSize = 0;
        do
        {
            heapSize += HeapAlign;
            result = nn::svc::SetHeapSize(&heapPtr, heapSize);
        } while (result.IsSuccess());
        heapSize -= HeapAlign;
        result = nn::svc::SetHeapSize(&heapPtr, heapSize);
        ASSERT_RESULT_SUCCESS(result);
    }

    // Heap を使い切っている方が生成出来るスレッド数が少なくなるはず
    CreateTestThreads(heapPtr, heapSize, NN_KERN_SLAB_OBJ_NUM_THREAD, &numThread, false);
    ASSERT_TRUE(numThread < NN_KERN_SLAB_OBJ_NUM_THREAD);

    result = nn::svc::SetHeapSize(&heapPtr, 0);
    ASSERT_RESULT_SUCCESS(result);
}
#endif

/*
   優先度が高いスレッドは、スリープや終了をしない限り、処理が低優先度のスレッドに移ることがない
*/
// TEST 7-19
TEST(CreateThreadTest, PriorityTest)
{
    TestThreadLeak leakTest;
    nn::Result result;
    nn::svc::Handle handles[2];
    uintptr_t pc[2];
    uintptr_t sp[2];
    uintptr_t param[2];
    int32_t idealCore = 0;

    pc[0] = reinterpret_cast<uintptr_t>(HighPriorityThread);
    pc[1] = reinterpret_cast<uintptr_t>(LowPriorityThread);
    sp[0] = reinterpret_cast<uintptr_t>(g_Buffers[0]) + sizeof(g_Buffers[0]);
    sp[1] = reinterpret_cast<uintptr_t>(g_Buffers[1]) + sizeof(g_Buffers[1]);
    param[0] = 0;

    for(int32_t priority = TestHighestThreadPriority;
            priority < TestLowestThreadPriority;
            priority++)
    {
        g_SequenceCount = 0;

        result = nn::svc::CreateThread(
                &handles[0], pc[0], param[0], sp[0], priority, idealCore);
        ASSERT_RESULT_SUCCESS(result);
        param[1] = reinterpret_cast<uintptr_t>(&handles[0]);

        result = nn::svc::CreateThread(
                &handles[1], pc[1], param[1], sp[1], priority + 1, idealCore);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::StartThread(handles[0]);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::StartThread(handles[1]);
        ASSERT_RESULT_SUCCESS(result);

        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &handles[0], 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::WaitSynchronization(&index, &handles[1], 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(handles[0]);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(handles[1]);
        ASSERT_RESULT_SUCCESS(result);
    }
}

