﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Tcb.h>

namespace {

const size_t StackSize = 0x1000;
const size_t BufferSize = 0x1000;

char g_Buffer[StackSize] __attribute__((aligned(0x1000)));
char g_ServerBuffer[BufferSize] __attribute__((aligned(0x1000)));
char g_ClientBuffer[BufferSize] __attribute__((aligned(0x1000)));

void SendSyncRequestThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::svc::Handle* handle = reinterpret_cast<nn::svc::Handle*>(arg);

    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    nn::Result result = nn::svc::SendSyncRequest(*handle);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(GetIpcTag(nullptr), DefaultIpcTag_Reply);
}

void SendSyncRequestWithUserBufferThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::svc::Handle* handle = reinterpret_cast<nn::svc::Handle*>(arg);

    uintptr_t addr = reinterpret_cast<uintptr_t>(g_ClientBuffer);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(addr);
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    nn::Result result = nn::svc::SendSyncRequestWithUserBuffer(addr, BufferSize, *handle);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(GetIpcTag(pMsgBuffer), DefaultIpcTag_Reply);
}

void SendAsyncRequestWithUserBufferThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::svc::Handle* handle = reinterpret_cast<nn::svc::Handle*>(arg);

    uintptr_t addr = reinterpret_cast<uintptr_t>(g_ClientBuffer);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(addr);
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    nn::svc::Handle event;
    nn::Result result = nn::svc::SendAsyncRequestWithUserBuffer(&event, addr, BufferSize, *handle);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose eventCloser(event);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(GetIpcTag(pMsgBuffer), DefaultIpcTag_Reply);
}

} // namespace

#ifdef ENABLE_MAX_HANDLE_TEST
TEST(CreateSession, MaxHandleTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;

    {
        ConsumeHandle maxHandle;

        // TEST 64-13
        // ハンドル数が上限に達している状態で CreateSession を呼び出すと失敗する
        result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultMaxHandle());
    }

    // TEST 64-14
    // ハンドル数が上限に達した後、ハンドルが解放されれば、CreateSession で作成できるようになる
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);
}
#endif // ENABLE_MAX_HANDLE_TEST

TEST(CreateSession, UseSession1)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;

    // 使用するセッションを作成
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendSyncRequestThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer) + StackSize;
    uintptr_t param = reinterpret_cast<uintptr_t>(&clientSession);

    // IPC の設定
    int32_t index;
    nn::svc::Handle handles[1] = { serverSession };

    for (int32_t idealCore = 0; idealCore < NumCore; idealCore++)
    {
        for (int32_t priority = TestHighestThreadPriority;
                priority <= TestLowestThreadPriority;
                priority++)
        {
            TestThread thread(pc, param, sp, priority, idealCore);
            thread.Start();

            SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
            // TEST 64-17
            // CreateSession で作成したセッションを用いて通信を行うことが出来る
            // SendSyncRequest & ReplyAndReceive
            result = nn::svc::ReplyAndReceive(&index, handles, 1,
                    nn::svc::INVALID_HANDLE_VALUE, -1);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_EQ(GetIpcTag(nullptr), DefaultIpcTag_Send);

            SetOnlyIpcTag(nullptr, DefaultIpcTag_Reply);
            result = nn::svc::ReplyAndReceive(&index, handles, 0,
                    serverSession, 0);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

            thread.Wait();
        }
    }
}

TEST(CreateSession, UseSession2)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;

    // 使用するセッションを作成
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendSyncRequestWithUserBufferThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer) + StackSize;
    uintptr_t param = reinterpret_cast<uintptr_t>(&clientSession);

    // IPC の設定
    int32_t index;
    nn::svc::Handle handles[1] = { serverSession };
    uintptr_t userBufferAddr = reinterpret_cast<uintptr_t>(g_ServerBuffer);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(userBufferAddr);

    for (int32_t idealCore = 0; idealCore < NumCore; idealCore++)
    {
        for (int32_t priority = TestHighestThreadPriority;
                priority <= TestLowestThreadPriority;
                priority++)
        {
            TestThread thread(pc, param, sp, priority, idealCore);
            thread.Start();

            SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Receive);
            // TEST 64-18
            // CreateSession で作成したセッションを用いて通信を行うことが出来る
            // SendSyncRequestWithUserBuffer & ReplyAndReceiveWithUserBuffer
            result = nn::svc::ReplyAndReceiveWithUserBuffer(
                    &index, userBufferAddr, BufferSize,
                    handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_EQ(GetIpcTag(pMsgBuffer), DefaultIpcTag_Send);

            SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Reply);
            result = nn::svc::ReplyAndReceiveWithUserBuffer(
                    &index, userBufferAddr, BufferSize,
                    handles, 0, serverSession, 0);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

            thread.Wait();
        }
    }
}

TEST(CreateSession, UseSession3)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;

    // 使用するセッションを作成
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendAsyncRequestWithUserBufferThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer) + StackSize;
    uintptr_t param = reinterpret_cast<uintptr_t>(&clientSession);

    // IPC の設定
    int32_t index;
    nn::svc::Handle handles[1] = { serverSession };
    uintptr_t userBufferAddr = reinterpret_cast<uintptr_t>(g_ServerBuffer);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(userBufferAddr);

    for (int32_t idealCore = 0; idealCore < NumCore; idealCore++)
    {
        for (int32_t priority = TestHighestThreadPriority;
                priority <= TestLowestThreadPriority;
                priority++)
        {
            TestThread thread(pc, param, sp, priority, idealCore);
            thread.Start();

            SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Receive);
            // TEST 64-19
            // CreateSession で作成したセッションを用いて通信を行うことが出来る
            // SendAsyncRequestWithUserBuffer & ReplyAndReceiveWithUserBuffer
            result = nn::svc::ReplyAndReceiveWithUserBuffer(
                    &index, userBufferAddr, BufferSize,
                    handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_EQ(GetIpcTag(pMsgBuffer), DefaultIpcTag_Send);

            SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Reply);
            result = nn::svc::ReplyAndReceiveWithUserBuffer(
                    &index, userBufferAddr, BufferSize,
                    handles, 0, serverSession, 0);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

            thread.Wait();
        }
    }
}

