﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"

extern "C" void nnMain();

namespace {

const int64_t SleepTime = 100 * 1000 * 1000;
volatile nn::Bit32 g_Lock;
char g_Stack[DefaultStackSize] __attribute__((aligned(0x1000)));
bool g_LoopEnd;
bool g_CheckFlag;
uintptr_t g_TargetAddress;

void DummyThread(uintptr_t arg)
{
    NN_UNUSED(arg);
    AutoThreadExit autoExit;
}

void UnlockThread(uintptr_t arg)
{
    NN_UNUSED(arg);
    AutoThreadExit autoExit;
    nn::Result result;

    while(!g_LoopEnd)
    {
        if (g_CheckFlag)
        {
            g_CheckFlag = false;
            result = nn::svc::ArbitrateUnlock(g_TargetAddress);
            //NN_ASSERT_RESULT_SUCCESS(result);
        }
        nn::svc::SleepThread(SleepTime);
    }
}


} // namespace

TEST(ArbitrateLock, HandleTest)
{
    nn::Result result;
    uintptr_t addr = reinterpret_cast<uintptr_t>(&g_Lock);
    g_TargetAddress = addr;
    nnHandle handle;
    g_Lock = 0;

    // Thread
    {
        uintptr_t pc = reinterpret_cast<uintptr_t>(UnlockThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
        uintptr_t param = 0;
        int32_t priority = TestLowestThreadPriority;
        int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

        g_LoopEnd = false;
        g_CheckFlag = true;
        TestThread thread(pc, param, sp, priority, idealCore);
        thread.Start();

        handle = thread.GetHandle();
        g_Lock = handle.value | nn::svc::Handle::WaitMask;

        // TEST 135-1
        // スレッドのハンドルを受け付ける
        result = nn::svc::ArbitrateLock(thread.GetHandle(), addr, 0);
        ASSERT_RESULT_SUCCESS(result);

        g_LoopEnd = true;

        thread.Wait();

        result = nn::svc::CloseHandle(thread.GetHandle());
        ASSERT_RESULT_SUCCESS(result);

        // TEST 135-6
        // Close したハンドラ を受け付けない
        handle = thread.GetHandle();
        g_Lock = handle.value | nn::svc::Handle::WaitMask;
        result = nn::svc::ArbitrateLock(thread.GetHandle(), addr, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    }

    // Process
    {
        TestProcess process(1);

        // TEST 135-2
        // プロセスのハンドルを受け付けない
        handle = process.GetHandle();
        g_Lock = handle.value | nn::svc::Handle::WaitMask;
        result = nn::svc::ArbitrateLock(process.GetHandle(), addr, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    }

    // TEST 135-3
    // INVALID_HANDLE_VALUE を受け付けない
    handle = nn::svc::INVALID_HANDLE_VALUE;
    g_Lock = handle.value | nn::svc::Handle::WaitMask;
    result = nn::svc::ArbitrateLock(nn::svc::INVALID_HANDLE_VALUE, addr, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 135-4
    // スレッドの擬似ハンドルをを受け付けない
    handle = nn::svc::PSEUDO_HANDLE_CURRENT_THREAD;
    g_Lock = handle.value | nn::svc::Handle::WaitMask;
    result = nn::svc::ArbitrateLock(nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, addr, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 135-5
    // プロセスの擬似ハンドルをを受け付けない
    handle = nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS;
    g_Lock = handle.value | nn::svc::Handle::WaitMask;
    result = nn::svc::ArbitrateLock(nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, addr, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

TEST(ArbitrateLock, AddressTest)
{
    nn::Result result;
    nnHandle handle;
    uintptr_t addr;
    size_t align = 4;
    size_t size = 0x1000;

    TestHeap heap(HeapAlign);
    addr = heap.GetAddress();
    ASSERT_TRUE((addr % align) == 0);
    nn::Bit32* ptr = reinterpret_cast<nn::Bit32*>(addr);
    nn::Bit32 value = 0;
    *ptr = 0;

    uintptr_t pc = reinterpret_cast<uintptr_t>(UnlockThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t param = 0;
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

    g_LoopEnd = false;

    TestThread thread(pc, param, sp, priority, idealCore);
    thread.Start();

    nn::svc::Handle threadHandle = thread.GetHandle();
    handle = threadHandle;
    value = handle.value | nn::svc::Handle::WaitMask;

    // TEST 135-7
    // 4B にアライメントされている値を受け付ける
    for (uintptr_t i = 0; i < 0x100; i += align)
    {
        g_TargetAddress = addr + i;
        ptr = reinterpret_cast<nn::Bit32*>(g_TargetAddress);
        *ptr = value;
        g_CheckFlag = true;
        result = nn::svc::ArbitrateLock(threadHandle, g_TargetAddress, 0);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 135-8
    // 4B にアライメントされていないと失敗する
    for (uintptr_t i = 1; i < align; i++)
    {
        g_TargetAddress = addr + i;
        ptr = reinterpret_cast<nn::Bit32*>(addr + i);
        *ptr = value;
        g_CheckFlag = true;
        result = nn::svc::ArbitrateLock(threadHandle, addr + i, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());
    }

    ptr = reinterpret_cast<nn::Bit32*>(addr);
    *ptr = value;
    g_TargetAddress = addr;

    // TEST 135-9
    // MemoryPermission_None の領域は受け付けない
    {
        g_CheckFlag = true;
        TestMemoryPermission perm(addr, size, nn::svc::MemoryPermission_None);
        result = nn::svc::ArbitrateLock(threadHandle, addr, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }

    // TEST 135-10
    // MemoryPermission_Read の領域は受け付けない
    {
        g_CheckFlag = true;
        TestMemoryPermission perm(addr, size, nn::svc::MemoryPermission_Read);
        result = nn::svc::ArbitrateLock(threadHandle, addr, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }

    // TEST 135-13
    // MemoryPermission_ReadWrite の領域を受け付ける
    {
        g_CheckFlag = true;
        TestMemoryPermission perm(addr, size, nn::svc::MemoryPermission_ReadWrite);
        result = nn::svc::ArbitrateLock(threadHandle, addr, 0);
        ASSERT_RESULT_SUCCESS(result);
    }

    g_LoopEnd = true;
    thread.Wait();
}

TEST(ArbitrateLock, NewValueTest)
{
    nn::Result result;
    uintptr_t addr = reinterpret_cast<uintptr_t>(&g_Lock);
    size_t align = 4;
    ASSERT_TRUE((addr % align) == 0);
    g_Lock = 0;

    uintptr_t pc = reinterpret_cast<uintptr_t>(DummyThread);
    TestThread thread(pc, 0, 0, TestLowestThreadPriority, 0);
    nn::svc::Handle threadHandle = thread.GetHandle();

    nn::Bit32 validValue[] = {0, 1, 0x1000, 0xFFFF, 0x10000, 0xFFFFFFFF};

    // TEST 135-15
    // Bit32 の値を受け付ける
    for (int32_t i = 0; i < static_cast<int32_t>(sizeof(validValue) / sizeof(nn::Bit32)); i++)
    {
        result = nn::svc::ArbitrateLock(threadHandle, addr, validValue[i]);
        ASSERT_RESULT_SUCCESS(result);
    }
}

