﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Tcb.h>

extern nn::Result ClientSendLightRequest(nn::svc::Handle handle);
extern nn::Result ServerReceiveLightRequest(nn::svc::Handle handle);
extern nn::Result ServerReplyLightRequest(nn::svc::Handle handle);

namespace {

const char* PortName = "Test6501";
const size_t StackSize = DefaultStackSize;
char g_Buffer[StackSize] __attribute__((aligned(0x1000)));

void NamedPortClient(uintptr_t arg)
{
    NN_UNUSED(arg);
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle clientSession;

    result = nn::svc::ConnectToNamedPort(&clientSession, PortName);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose cSessionCloser(clientSession);

    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequest(clientSession);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(GetIpcTag(nullptr), DefaultIpcTag_Reply);
}

void NormalPortClient(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle* clientPort = reinterpret_cast<nn::svc::Handle*>(arg);
    nn::svc::Handle clientSession;

    result = nn::svc::ConnectToPort(&clientSession, *clientPort);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose cSessionCloser(clientSession);

    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequest(clientSession);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(GetIpcTag(nullptr), DefaultIpcTag_Reply);
}

void LightPortClient(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle* clientPort = reinterpret_cast<nn::svc::Handle*>(arg);
    nn::svc::Handle clientSession;

    result = nn::svc::ConnectToPort(&clientSession, *clientPort);
    ASSERT_RESULT_SUCCESS(result);

    result = ClientSendLightRequest(clientSession);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);
}

} // namespace

TEST(AcceptSession, AcceptNamedPort)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverPort;

    // NamedPort を作成
    result = nn::svc::ManageNamedPort(&serverPort, PortName, 1);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);

    // Thread の設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(NamedPortClient);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer) + StackSize;
    uintptr_t param = 0;
    uintptr_t priority = TestLowestThreadPriority;
    uintptr_t idealCore = nn::svc::IdealCoreUseProcessValue;

    TestThread thread(pc, param, sp, priority, idealCore);
    thread.Start();

    // TEST 65-1
    // クライアントからの接続を受け付ける
    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle serverSession;
    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 65-4
    // Normal Session を渡すと、Normal Session が返ってくる
    {
        nn::svc::Handle handles[1] = {serverSession};
        int32_t index;

        result = ServerReceiveLightRequest(serverSession);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

        SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
        result = nn::svc::ReplyAndReceive(
                &index, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_EQ(GetIpcTag(nullptr), DefaultIpcTag_Send);
        SetOnlyIpcTag(nullptr, DefaultIpcTag_Reply);
        result = nn::svc::ReplyAndReceive(&index, handles, 0, serverSession, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }

    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);

    thread.Wait();

    result = nn::svc::ManageNamedPort(&serverPort, PortName, 0);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(AcceptSession, AcceptNormalPort)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;

    // Port を作成
    result = nn::svc::CreatePort(&serverPort, &clientPort, 1, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);
    AutoHandleClose cPortCloser(clientPort);

    // Thread の設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(NormalPortClient);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer) + StackSize;
    uintptr_t param = reinterpret_cast<uintptr_t>(&clientPort);
    uintptr_t priority = TestLowestThreadPriority;
    uintptr_t idealCore = nn::svc::IdealCoreUseProcessValue;

    TestThread thread(pc, param, sp, priority, idealCore);
    thread.Start();

    // TEST 65-2
    // クライアントからの接続を受け付ける
    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 65-4
    // Normal Session を渡すと、Normal Session が返ってくる
    nn::svc::Handle serverSession;
    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);

    {
        nn::svc::Handle handles[1] = {serverSession};
        int32_t index;

        result = ServerReceiveLightRequest(serverSession);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

        SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
        result = nn::svc::ReplyAndReceive(
                &index, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_EQ(GetIpcTag(nullptr), DefaultIpcTag_Send);
        SetOnlyIpcTag(nullptr, DefaultIpcTag_Reply);
        result = nn::svc::ReplyAndReceive(&index, handles, 0, serverSession, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }

    thread.Wait();
}

TEST(AcceptSession, AcceptLightPort)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;

    // Port を作成
    result = nn::svc::CreatePort(&serverPort, &clientPort, 1, true, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);
    AutoHandleClose cPortCloser(clientPort);

    // Thread の設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(LightPortClient);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer) + StackSize;
    uintptr_t param = reinterpret_cast<uintptr_t>(&clientPort);
    uintptr_t priority = TestLowestThreadPriority;
    uintptr_t idealCore = nn::svc::IdealCoreUseProcessValue;

    TestThread thread(pc, param, sp, priority, idealCore);
    thread.Start();

    // TEST 65-3
    // クライアントからの接続を受け付ける
    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 65-5
    // port に Light Session を渡すと、pOut に Light Session が渡される
    nn::svc::Handle serverSession;
    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);

    {
        nn::svc::Handle handles[1] = {serverSession};
        int32_t index;

        SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
        result = nn::svc::ReplyAndReceive(
                &index, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

        result = ServerReceiveLightRequest(serverSession);
        ASSERT_RESULT_SUCCESS(result);
        result = ServerReplyLightRequest(serverSession);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());
    }

    thread.Wait();
}

#ifdef ENABLE_MAX_HANDLE_TEST
TEST(AcceptSession, MaxHandleTest)
{
    TestSessionLeak leakTest;
    nn::Result result;

    // ポートの設定
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;

    result = nn::svc::CreatePort(&serverPort, &clientPort, 1, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);
    AutoHandleClose cPortCloser(clientPort);

    // クライアントの接続
    nn::svc::Handle clientSession;
    result = nn::svc::ConnectToPort(&clientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose cSessionCloser(clientSession);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle serverSession;
    {
        ConsumeHandle maxHandle;

        // TEST 65-18
        // ハンドル数が上限に達している状態で呼び出すと失敗する
        result = nn::svc::AcceptSession(&serverSession, serverPort);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultMaxHandle());
    }

    // ハンドル数が上限に達した状態が解除されると、再度確保できる
    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
}
#endif // ENABLE_MAX_HANDLE_TEST

