﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sdmmc_BaseDeviceAccessor.h"
#include "sdmmc_Log.h"
#include <nn/result/result_HandlingUtility.h>
#if (defined(NN_DETAIL_SDMMC_THREAD_SAFE_ENABLE))
    #include <mutex>
#endif

namespace nn { namespace sdmmc1 {
namespace detail {

#if (defined(NN_DETAIL_SDMMC_THREAD_SAFE_ENABLE))
    #define NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD   std::lock_guard<nn::os::Mutex> lock(m_pBaseDevice->m_DeviceMutex)
#else
    #define NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD
#endif

void BaseDevice::GetLegacyCapacityParameters(uint8_t* pOutCSizeMult, uint8_t* pOutReadBlLen) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutCSizeMult);
    NN_ABORT_UNLESS_NOT_NULL(pOutReadBlLen);

    // m_Csd[2]: CSD[55:40]
    // CSD[49:47] C_SIZE_MULT の取得
    *pOutCSizeMult = static_cast<uint8_t>((m_Csd[2] & 0x0380U) >> 7);

    // m_Csd[4]: CSD[87:72]
    // CSD[83:80] READ_BL_LEN の取得
    *pOutReadBlLen = static_cast<uint8_t>((m_Csd[4] & 0x0F00U) >> 8);
}

Result BaseDevice::SetLegacyMemoryCapacity() NN_NOEXCEPT
{
    // m_Csd[4]: CSD[87:72], m_Csd[3]: CSD[71:56]
    // CSD[73:62]: C_SIZE の取得
    uint32_t cSize = ((m_Csd[4] & 0x0003U) << 10) | ((m_Csd[3] & 0xFFC0U) >> 6);

    uint8_t cSizeMult = 0;
    uint8_t readBlLen = 0;
    GetLegacyCapacityParameters(&cSizeMult, &readBlLen);
    NN_DETAIL_SDMMC_DEBUG_LOG("Legacy C_SIZE: %u, C_SIZE_MULT: %u, READ_BL_LEN: %u\n", cSize, static_cast<uint32_t>(cSizeMult), static_cast<uint32_t>(readBlLen));
    if (((cSizeMult + 2) + readBlLen) >= 9)
    {
        // 桁あふれしないように、べき乗をまとめて計算
        // BLOCKNR = (C_SIZE+1)^MULT;
        // MULT = 2^(C_SIZE_MULT+2);
        // BLOCK_LEN = 2^READ_BL_LEN;
        // memory capacity(byte) = BLOCKNR * BLOCK_LEN;
        // memory capacity(sector) = memory capacity(byte) / 512;
        m_MemoryCapacity = (cSize + 1) << ((cSizeMult + 2) + readBlLen - 9);
        m_IsValidMemoryCapacity = true;
        return ResultSuccess();
    }
    else
    {
        return ResultUnexpectedDeviceCsdValue();
    }
}

Result BaseDevice::CheckDeviceStatus(uint32_t deviceStatus) const NN_NOEXCEPT
{
    // 簡易チェック
    if ((deviceStatus & DeviceStatus_Mask_AllErrors) == 0)
    {
        return ResultSuccess();
    }

    // エラー Result として優先する順にチェックする
    if ((deviceStatus & DeviceStatus_Bit_ComCrcError) != 0)
    {
        return ResultDeviceStatusComCrcError();
    }
    if ((deviceStatus & DeviceStatus_Bit_DeviceEccFailed) != 0)
    {
        return ResultDeviceStatusDeviceEccFailed();
    }
    if ((deviceStatus & DeviceStatus_Bit_CcError) != 0)
    {
        return ResultDeviceStatusCcError();
    }
    if ((deviceStatus & DeviceStatus_Bit_Error) != 0)
    {
        return ResultDeviceStatusError();
    }
    if (GetDeviceType() == DeviceType_Mmc)
    {
        if ((deviceStatus & DeviceStatus_Bit_SwitchError) != 0)
        {
            return ResultDeviceStatusSwitchError();
        }
    }
    if ((deviceStatus & DeviceStatus_Bit_AddressMisalign) != 0)
    {
        return ResultDeviceStatusAddressMisalign();
    }
    if ((deviceStatus & DeviceStatus_Bit_BlockLenError) != 0)
    {
        return ResultDeviceStatusBlockLenError();
    }
    if ((deviceStatus & DeviceStatus_Bit_EraseSeqError) != 0)
    {
        return ResultDeviceStatusEraseSeqError();
    }
    if ((deviceStatus & DeviceStatus_Bit_EraseParam) != 0)
    {
        return ResultDeviceStatusEraseParam();
    }
    if ((deviceStatus & DeviceStatus_Bit_WpViolation) != 0)
    {
        return ResultDeviceStatusWpViolation();
    }
    if ((deviceStatus & DeviceStatus_Bit_LockUnlockFailed) != 0)
    {
        return ResultDeviceStatusLockUnlockFailed();
    }
    if ((deviceStatus & DeviceStatus_Bit_CidCsdOverwrite) != 0)
    {
        return ResultDeviceStatusCidCsdOverwrite();
    }
    if ((deviceStatus & DeviceStatus_Bit_WpEraseSkip) != 0)
    {
        return ResultDeviceStatusWpEraseSkip();
    }
    if ((deviceStatus & DeviceStatus_Bit_EraseReset) != 0)
    {
        return ResultDeviceStatusEraseReset();
    }
    if ((deviceStatus & DeviceStatus_Bit_IllegalCommand) != 0)
    {
        // SDA Physical Spec Version 2.00 未満への CMD8 発行で発生する場合がある
        return ResultDeviceStatusIllegalCommand();
    }
    if ((deviceStatus & DeviceStatus_Bit_AddressOutOfRange) != 0)
    {
        // 終端セクタのリードアクセスで発生する場合がある
        return ResultDeviceStatusAddressOutOfRange();
    }
    return ResultSuccess();
}

DeviceState BaseDevice::GetDeviceState(uint32_t r1Response) const NN_NOEXCEPT
{
    return static_cast<DeviceState>((r1Response & DeviceStatus_Mask_CurrentState) >> DeviceStatus_Pos_CurrentState);
}

Result BaseDeviceAccessor::IssueCommandAndCheckR1(uint32_t* pOutResponse, uint32_t commandIndex, uint32_t commandArgument, bool isBusy, DeviceState expectedState, uint32_t deviceStatusBitsIgnored) const NN_NOEXCEPT
{
    ResponseType responseType = ResponseType_R1;

    Command command(commandIndex, commandArgument, responseType, isBusy);
    Result result = m_pHostController->IssueCommand(&command);
    if (result.IsFailure())
    {
        return result;
    }

    NN_ABORT_UNLESS_NOT_NULL(pOutResponse);
    m_pHostController->GetLastResponse(pOutResponse, sizeof(uint32_t), responseType);
    if (deviceStatusBitsIgnored != 0)
    {
        *pOutResponse &= (~deviceStatusBitsIgnored);
    }
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    result = m_pBaseDevice->CheckDeviceStatus(*pOutResponse);
    if (result.IsFailure())
    {
        return result;
    }

    if (expectedState != DeviceState_Unknown)
    {
        if (m_pBaseDevice->GetDeviceState(*pOutResponse) != expectedState)
        {
            return ResultUnexpectedDeviceState();
        }
    }

    return ResultSuccess();
}

Result BaseDeviceAccessor::IssueCommandGoIdleState() const NN_NOEXCEPT
{
    // argument[31:0] stuff bit
    Command command(0, 0, ResponseType_R0, false);
    return m_pHostController->IssueCommand(&command);
}

Result BaseDeviceAccessor::IssueCommandAllSendCid(void* pOutCidBuffer, size_t cidBufferSize) const NN_NOEXCEPT
{
    ResponseType responseType = ResponseType_R2;

    // argument[31:0] stuff bit
    Command command(2, 0, responseType, false);
    Result result = m_pHostController->IssueCommand(&command);
    if (result.IsFailure())
    {
        return result;
    }

    NN_ABORT_UNLESS_NOT_NULL(pOutCidBuffer);
    NN_ABORT_UNLESS((reinterpret_cast<uintptr_t>(pOutCidBuffer) % sizeof(uint32_t)) == 0);
    NN_ABORT_UNLESS(cidBufferSize >= DeviceCidSize);
    m_pHostController->GetLastResponse(reinterpret_cast<uint32_t*>(pOutCidBuffer), DeviceCidSize, responseType);
    NN_DETAIL_SDMMC_DEBUG_LOG("CID:\n");
    NN_DETAIL_SDMMC_DATA_LOG(pOutCidBuffer, DeviceCidSize);

    return ResultSuccess();
}

Result BaseDeviceAccessor::IssueCommandSelectCard() const NN_NOEXCEPT
{
    // argument[31:16] RCA
    // argument[15: 0] stuff bits
    // R1b, このコマンドで状態遷移するため、state チェックは除外する
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    uint32_t argument = static_cast<uint32_t>(m_pBaseDevice->GetRca()) << 16;
    return IssueCommandAndCheckR1(7, argument, true, DeviceState_Unknown);
}

Result BaseDeviceAccessor::IssueCommandSendCsd(void* pOutCsdBuffer, size_t csdBufferSize) const NN_NOEXCEPT
{
    ResponseType responseType = ResponseType_R2;

    // argument[31:16] RCA
    // argument[15: 0] stuff bits
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    uint32_t argument = static_cast<uint32_t>(m_pBaseDevice->GetRca()) << 16;
    Command command(9, argument, responseType, false);
    Result result = m_pHostController->IssueCommand(&command);
    if (result.IsFailure())
    {
        return result;
    }

    NN_ABORT_UNLESS_NOT_NULL(pOutCsdBuffer);
    NN_ABORT_UNLESS((reinterpret_cast<uintptr_t>(pOutCsdBuffer) % sizeof(uint32_t)) == 0);
    NN_ABORT_UNLESS(csdBufferSize >= DeviceCsdSize);
    m_pHostController->GetLastResponse(reinterpret_cast<uint32_t*>(pOutCsdBuffer), DeviceCsdSize, responseType);
    NN_DETAIL_SDMMC_DEBUG_LOG("CSD:\n");
    NN_DETAIL_SDMMC_DATA_LOG(pOutCsdBuffer, DeviceCsdSize);

    return ResultSuccess();
}

Result BaseDeviceAccessor::IssueCommandSendStatus(uint32_t* pOutDeviceStatus, uint32_t deviceStatusBitsIgnored) const NN_NOEXCEPT
{
    // argument[31:16] RCA
    // argument[15: 0] stuff bits
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    uint32_t argument = static_cast<uint32_t>(m_pBaseDevice->GetRca()) << 16;
    return IssueCommandAndCheckR1(pOutDeviceStatus, 13, argument, false, DeviceState_Tran, deviceStatusBitsIgnored);
}

Result BaseDeviceAccessor::IssueCommandSetBlockLenToSectorSize() const NN_NOEXCEPT
{
    // argument[31: 0] block length, SectorSize 固定
    return IssueCommandAndCheckR1(16, SectorSize, false, DeviceState_Tran);
}

Result BaseDeviceAccessor::IssueCommandMultipleBlock(uint32_t* pOutTransferredNumBlocks, uint32_t sectorIndex, uint32_t numSectors, void* pBuffer, bool isRead) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);

    ResponseType responseType = ResponseType_R1;

    // argument[31:0] data address
    uint32_t commandArgument;
    if (m_pBaseDevice->IsHighCapacity())
    {
        commandArgument = sectorIndex;
    }
    else
    {
        commandArgument = sectorIndex * SectorSize;
    }
    uint32_t commandIndex;
    TransferDirection transferDirection;
    if (isRead)
    {
        commandIndex = 18;
        transferDirection = TransferDirection_ReadFromDevice;
    }
    else
    {
        commandIndex = 25;
        transferDirection = TransferDirection_WriteToDevice;
    }
    Command command(commandIndex, commandArgument, responseType, false);
    TransferData transferData(pBuffer, SectorSize, numSectors, transferDirection, true, true);
    Result result = m_pHostController->IssueCommand(&command, &transferData, pOutTransferredNumBlocks);
    if (result.IsFailure())
    {
        return result;
    }

    uint32_t response;
    m_pHostController->GetLastResponse(&response, sizeof(response), responseType);
    result = m_pBaseDevice->CheckDeviceStatus(response);
    if (result.IsFailure())
    {
        return result;
    }

    m_pHostController->GetLastStopTransmissionResponse(&response, sizeof(response));
    if (isRead)
    {
        if ((sectorIndex + numSectors) == m_pBaseDevice->GetMemoryCapacity())
        {
            // 終端セクタを含むリードの ADDRESS_OUT_OF_RANGE を無視する（デバイスによっては発生する）
            response &= (~DeviceStatus_Bit_AddressOutOfRange);
        }
    }
    result = m_pBaseDevice->CheckDeviceStatus(response);
    if (result.IsFailure())
    {
        return result;
    }

    return ResultSuccess();
}

void BaseDeviceAccessor::Deactivate() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    if (m_pBaseDevice->IsActive())
    {
        m_pHostController->Shutdown();

        m_pBaseDevice->Deactivate();
    }
}

Result BaseDeviceAccessor::ReadWrite(uint32_t sectorIndex, uint32_t numSectors, void* pBuffer, size_t bufferSize, bool isRead) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_pBaseDevice->CheckAccessible());

    if (numSectors == 0)
    {
        // 転送セクタ数が 0 ならば、何もしない
        return ResultSuccess();
    }

    // バッファサイズが十分であることを確認
    NN_ABORT_UNLESS((bufferSize / SectorSize) >= numSectors);

    uint32_t currentSectorIndex = sectorIndex;
    uint32_t restNumSectors = numSectors;
    uint8_t* pCurrentBuffer = reinterpret_cast<uint8_t*>(pBuffer);
    while (true)
    {
        uint32_t transferredNumBlocks;
        Result result = IssueCommandMultipleBlock(&transferredNumBlocks, currentSectorIndex, restNumSectors, pCurrentBuffer, isRead);
        if (result.IsFailure())
        {
            // TODO: エラーからの復旧
            return result;
        }

        uint32_t deviceStatusBitsIgnored = 0;
        if (isRead)
        {
            if ((currentSectorIndex + transferredNumBlocks) == m_pBaseDevice->GetMemoryCapacity())
            {
                // 終端セクタを含むリードの ADDRESS_OUT_OF_RANGE を無視する（デバイスによっては発生する）
                deviceStatusBitsIgnored = DeviceStatus_Bit_AddressOutOfRange;
            }
        }
        result = IssueCommandSendStatus(deviceStatusBitsIgnored);
        if (result.IsFailure())
        {
            // TODO: エラーからの復旧
            return result;
        }


        restNumSectors -= transferredNumBlocks;
        if (restNumSectors <= 0)
        {
            break;  // 完了
        }
        currentSectorIndex += transferredNumBlocks;
        pCurrentBuffer += (SectorSize * transferredNumBlocks);
    }

    return ResultSuccess();
}

Result BaseDeviceAccessor::GetMemoryCapacity(uint32_t* pOutNumSectors) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_pBaseDevice->CheckAccessible());
    NN_ABORT_UNLESS_NOT_NULL(pOutNumSectors);
    *pOutNumSectors = m_pBaseDevice->GetMemoryCapacity();
    return ResultSuccess();
}

Result BaseDeviceAccessor::GetDeviceStatus(uint32_t* pOutDeviceStatus) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_pBaseDevice->CheckAccessible());
    return IssueCommandSendStatus(pOutDeviceStatus, 0);
}

Result BaseDeviceAccessor::GetOcr(uint32_t* pOutOcr) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_pBaseDevice->CheckAccessible());
    NN_ABORT_UNLESS_NOT_NULL(pOutOcr);
    *pOutOcr = m_pBaseDevice->GetOcr();
    return ResultSuccess();
}

Result BaseDeviceAccessor::GetRca(uint16_t* pOutRca) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_pBaseDevice->CheckAccessible());
    NN_ABORT_UNLESS_NOT_NULL(pOutRca);
    *pOutRca = m_pBaseDevice->GetRca();
    return ResultSuccess();
}

Result BaseDeviceAccessor::GetCid(void* pOutCid, size_t cidSize) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_pBaseDevice->CheckAccessible());
    m_pBaseDevice->GetCid(pOutCid, cidSize);
    return ResultSuccess();
}

Result BaseDeviceAccessor::GetCsd(void* pOutCsd, size_t csdSize) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_pBaseDevice->CheckAccessible());
    m_pBaseDevice->GetCsd(pOutCsd, csdSize);
    return ResultSuccess();
}

#if (defined(NN_DETAIL_SDMMC_ISSUE_COMMAND_FOR_DEBUG))
    Result BaseDeviceAccessor::IssueCommandForDebug(uint32_t* pOutResponse, uint32_t commandIndex, uint32_t commandArgument,
        DataTransfer* pDataTransfer, bool isBusy) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
        NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;

        // R1 コマンド固定
        ResponseType responseType = ResponseType_R1;
        Command command(commandIndex, commandArgument, responseType, isBusy);

        Result result;
        if (pDataTransfer == nullptr)
        {
            result = m_pHostController->IssueCommand(&command);
        }
        else
        {
            // バッファサイズが十分であることを確認
            NN_ABORT_UNLESS((pDataTransfer->dataBufferSize / pDataTransfer->blockSize) >= pDataTransfer->numBlocks);

            TransferDirection transferDirection = TransferDirection_WriteToDevice;
            if (pDataTransfer->isRead)
            {
                transferDirection = TransferDirection_ReadFromDevice;
            }

            bool isMultiBlockTransfer = false;
            if ((commandIndex == 18) || (commandIndex == 25))
            {
                // CMD18, CMD25 は Multi Block 転送固定
                isMultiBlockTransfer = true;
            }

            // CMD12 自動発行なし固定
            TransferData transferData(pDataTransfer->pDataBuffer,
                pDataTransfer->blockSize, pDataTransfer->numBlocks, transferDirection, isMultiBlockTransfer, false);

            result = m_pHostController->IssueCommand(&command, &transferData);
        }
        if (result.IsFailure())
        {
            return result;
        }

        m_pHostController->GetLastResponse(pOutResponse, sizeof(uint32_t), responseType);
        result = m_pBaseDevice->CheckDeviceStatus(*pOutResponse);
        if (result.IsFailure())
        {
            return result;
        }

        return ResultSuccess();
    }
#endif

} // namespace detail {
}} // namespace nn { namespace sdmmc1 {
