﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   I2C ドライバのバスアクセス機能の API 宣言。
 */

#pragma once

#include <nn/nn_Common.h>
#include "../i2c_Result.h"
#include "../i2c_Type.h"


namespace nn {
namespace i2c {
namespace driver {

/**
 * @brief セッション用パラメータ
 */
struct I2cSession
{
    int       _sessionId;
    uint64_t  _baseClock;
    I2cDevice _i2cDevice;
};

/**
 * @brief 指定したデバイスがボード上に存在するかどうかを返します。
 *
 * @param[in]   device      デバイス識別子
 * @return      指定したデバイスが存在するかどうかを返します。
 * @retval      true        指定したデバイスが存在します。
 * @retval      false       指定したデバイスは存在しないか、または使用できません。
 * @pre         ライブラリは初期化済の状態である必要があります。
 * @details    　デバイス識別子に該当するデバイスとの通信を行い、
 *              その結果からデバイスの有無を判定します。
 */
bool HasDevice(I2cDevice device) NN_NOEXCEPT;

/**
 * @brief 指定したスレーブデバイスとのセッションをオープンします。すなわちスレーブデバイスとの通信を可能な状態にします。
 *
 * @param[out]  pOutSession                    セッション用パラメータ
 * @param[in]   device                         デバイス識別子
 * @pre         ライブラリは初期化済の状態である必要があります。
 *              指定したデバイスはボード上に存在している必要があります。
 * @post        指定したデバイスが使用可能な状態になります。
 * @details     セッションのオープンに成功した場合、有効な　I2cSession　が設定されます。
 *              この関数は指定したスレーブアドレスのデバイスの接続状態の確認は行いません。
 */
void OpenSession(I2cSession* pOutSession, I2cDevice device) NN_NOEXCEPT;

/**
 * @brief 指定したデータ列を送信します。
 *
 * @param[in]   session      セッション用パラメータ
 * @param[in]   pInData      送信するデータ列
 * @param[in]   dataBytes    送信するデータ列の長さ [bytes]
 * @param[in]   inOption     送信時のオプションパラメータ
 * @return      処理の結果を返します。
 * @retval      ResultSuccess             指定した長さのデータ送信に成功しました。
 * @retval      ResultNoAck               対向デバイスとの接続が確立されていません。デバイスの接続状態を確認ください。
 * @retval      ResultBusBusy             バスが BUSY 状態です。通信間隔が短い場合や同一バスのトランザクションが込み合っている際に発生する可能性があります。
 * @pre         指定したセッションはオープン状態である必要があります。
 * @post        すべてのデータ列の送信が予約された状態になります。
 */
Result Send( const I2cSession& session, const void* pInData, size_t dataBytes, TransactionOption inOption ) NN_NOEXCEPT;

/**
 * @brief 指定した長さに達するまでデータ列を受信します。
 *
 * @param[out]  pOutData     受信データを格納するバッファ。
 *                           dataBytes で指定した以上のサイズで確保されている必要があります。
 * @param[in]   session      セッション用パラメータ
 * @param[in]   dataBytes    受信するデータ列の長さ [bytes]
 * @param[in]   inOption     送信時のオプションパラメータ
 * @return      処理の結果を返します。
 * @retval      ResultSuccess             指定した長さのデータ受信が完了しました。
 * @retval      ResultNoAck               対向デバイスとの接続が確立されていません。デバイスの接続状態を確認ください。
 * @retval      ResultBusBusy             バスが BUSY 状態です。通信間隔が短い場合や同一バスのトランザクションが込み合っている際に発生する可能性があります。
 * @pre         指定したデバイスとのセッションはオープン状態である必要があります。
 * @post        指定した長さのデータ列の受信が完了した状態になります。
 */
Result Receive( void* pOutData, const I2cSession& session, size_t dataBytes, TransactionOption inOption ) NN_NOEXCEPT;

/**
 * @brief コマンドリストの一括送信を行います
 *
 * @param[out]  pReceiveBuffer      コマンド実行結果の受信バッファ
 * @param[in]   receiveBufferSize   受信バッファサイズ[bytes]
 * @param[in]   session             セッション用パラメータ
 * @param[in]   pCommandList        実行コマンドのリスト
 * @param[in]   commandListLength   コマンドリスト長
 * @return      処理の結果を返します。
 * @retval      ResultSuccess       コマンドリストの実行に成功しました
 * @retval      ResultNoAck         対向デバイスとの接続が確立されていません。デバイスの接続状態を確認ください。
 * @retval      ResultBusBusy       バスが BUSY 状態です。通信間隔が短い場合や同一バスのトランザクションが込み合っている際に発生する可能性があります。
 * @pre         指定したデバイスとのセッションはオープン状態である必要があります。
 *              pCommandList != nullptr, pReceiveBuffer != nullptr である必要があります。
 * @post        指定したコマンド列が実行されます。
 *              Read コマンドの実行結果は pReceiveBuffer に格納されます。
 */
Result ExecuteCommandList( void* pReceiveBuffer, size_t receiveBufferSize, const I2cSession& session, const void* pCommandList, size_t commandListLength ) NN_NOEXCEPT;

/**
 * @brief 指定したセッションをクローズします。
 *
 * @param[in]   session    セッション用パラメータ
 * @pre         ライブラリは初期化済の状態である必要があります。
 * @post        指定したセッションがクローズ状態になります。
 * @details     オープンされていないセッションに対して実行した場合は何もしません。
 *              送信予約されているがまだ送信されていないデータについては失われる可能性があります。
 */
void CloseSession( const I2cSession& session ) NN_NOEXCEPT;

} // driver
} // i2c
} // nn


