﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file Pinmux 関係のレジスタ定義を切り出したファイル
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/dd.h>
#include "../i2c_DdUtil.h"

namespace nn {
namespace i2c {
namespace driver {
namespace detail {

struct PinmuxRegInfo
{
public:
    void Set(int busIdx) NN_NOEXCEPT
    {
        /**
         * @brief Pinmux レジスタが含まれる APB_MISC の物理アドレス ([Ref1] 2.0)
         */
        static const nn::dd::PhysicalAddress   ApbMiscPhysicalAddress = 0x070000000ull;
        static const size_t                    ApbMiscAddressSize     = 0x4000;

        /**
         * @brief レジスタのビット定義
         */
        /*
        struct PinmuxRegister
        {
            typedef nn::util::BitPack32::Field< 12,  1, bool> ESchmt;   // Enable schmitt trigger
            typedef nn::util::BitPack32::Field< 11,  1, bool> EOd;      // DISABLED: DO NOT USE
            typedef nn::util::BitPack32::Field< 10,  1, bool> EIoHv;    // Enable open-drain pull up capability to 3.3V
            typedef nn::util::BitPack32::Field<  9,  1, bool> EHsm;     // Enable high speed mode
            typedef nn::util::BitPack32::Field<  8,  1, bool> ELpdr;    // Enable only one bsae driver when set high
                                                                        // Typically set for I/F like I2C
            typedef nn::util::BitPack32::Field<  7,  1, bool> Lock;     // Lock control for writing for security
            typedef nn::util::BitPack32::Field<  6,  1, bool> EInput;   // Enable input receiver
            typedef nn::util::BitPack32::Field<  5,  1, bool> Park;     // Parking state holds control during deep sleep
            typedef nn::util::BitPack32::Field<  4,  1, bool> Tristate; // Tristate (High-Z) option
            typedef nn::util::BitPack32::Field<  2,  2,  int> Pupd;     // Pull up / pull down
            typedef nn::util::BitPack32::Field<  0,  2,  int> Pm;       //
        };
        */

        // TORIAEZU: 現時点では明示的な設定の必要があることがわかるまでレジスタ操作は行わない
        static const uintptr_t SdlRegOffsetTable[] =
        {
            0x0000, // I2C1
            0x0000, // I2C2
            0x0000, // I2C3
            0x0000, // I2C4
            0x0000, // I2C5
            0x0000  // I2C6
        };
        static const uintptr_t SdlRegValueTable[] =
        {
            0x0, // I2C1
            0x0, // I2C2
            0x0, // I2C3
            0x0, // I2C4
            0x0, // I2C5
            0x0  // I2C6
        };
        static const uintptr_t SdlRegMaskTable[] =
        {
            0x0, // I2C1
            0x0, // I2C2
            0x0, // I2C3
            0x0, // I2C4
            0x0, // I2C5
            0x0  // I2C6
        };
        NN_UNUSED(SdlRegMaskTable);
        static const uintptr_t SdaRegOffsetTable[] =
        {
            0x0000, // I2C1
            0x0000, // I2C2
            0x0000, // I2C3
            0x0000, // I2C4
            0x0000, // I2C5
            0x0000  // I2C6
        };
        static const uintptr_t SdaRegValueTable[] =
        {
            0x0, // I2C1
            0x0, // I2C2
            0x0, // I2C3
            0x0, // I2C4
            0x0, // I2C5
            0x0  // I2C6
        };
        static const uintptr_t SdaRegMaskTable[] =
        {
            0x0, // I2C1
            0x0, // I2C2
            0x0, // I2C3
            0x0, // I2C4
            0x0, // I2C5
            0x0  // I2C6
        };
        NN_UNUSED(SdaRegMaskTable);

        NN_SDK_ASSERT(0 <= busIdx && busIdx < static_cast<int>( sizeof(SdlRegOffsetTable) / sizeof(SdlRegOffsetTable[0]) ) );

        // IO マッピングを取得する
        uintptr_t baseAddr = GetVirtualAddress(ApbMiscPhysicalAddress, ApbMiscAddressSize); // APB_MISC

        pScl        = reinterpret_cast<volatile uint32_t*>(baseAddr + SdlRegOffsetTable[busIdx]);
        sclBitValue = SdlRegValueTable[busIdx];
        sclBitMask  = SdlRegValueTable[busIdx];
        pSda        = reinterpret_cast<volatile uint32_t*>(baseAddr + SdaRegOffsetTable[busIdx]);
        sdaBitValue = SdaRegValueTable[busIdx];
        sdaBitMask  = SdaRegValueTable[busIdx];
    }

public:
    volatile uint32_t*  pScl;
    uint32_t            sclBitValue;
    uint32_t            sclBitMask;
    volatile uint32_t*  pSda;
    uint32_t            sdaBitValue;
    uint32_t            sdaBitMask;
};

} // detail
} // driver
} // i2c
} // nn
