﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <climits>
#include <mutex>

#include <nn/nn_Common.h>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>

#include "../../include/nn/i2c/driver/i2c.h"
#include "../../include/nn/i2c/driver/i2c_BusDev.h"

#include "i2c_Util.h"
#include "i2c_TargetSpec.h"
#include "i2c_ResourceManager.h"

namespace nn {
namespace i2c {
namespace driver {
namespace detail {

void ResourceManager::Initialize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock( m_InitializeCountMutex );

    NN_SDK_ASSERT( m_InitializeCount < INT_MAX ); // 値域チェック

    m_InitializeCount++;

    // 初期化済みならば何も行わない
    if ( m_InitializeCount > 1)
    {
        NN_DETAIL_I2C_LOG("%s(%d) Already Initialized. COUNT=%d\n", __FUNCTION__, __LINE__ , m_InitializeCount );
        return;
    }
    NN_SDK_REQUIRES( m_InitializeCount == 1); // 念のためチェック

    NN_DETAIL_I2C_LOG("%s(%d) Initialize Done.\n", __FUNCTION__, __LINE__ );
}

void ResourceManager::Finalize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock( m_InitializeCountMutex );

    NN_SDK_ASSERT( m_InitializeCount > INT_MIN ); // 値域チェック

    m_InitializeCount--;

    // 終了回数が初期化された回数に達していない場合は何も行わない
    if ( m_InitializeCount > 0 )
    {
        NN_DETAIL_I2C_LOG("%s(%d) Not Finalized. COUNT=%d.\n", __FUNCTION__, __LINE__ , m_InitializeCount );
        return;
    }
    NN_SDK_REQUIRES( m_InitializeCount == 0);

    std::lock_guard<nn::os::Mutex> lockSession( m_SessionIsOpenMutex );

    for( int i = 0; i < MaxSessions; i++ )
    {
        m_Sessions[i].Close();
    }

    NN_DETAIL_I2C_LOG("%s(%d) Finalize Done.\n", __FUNCTION__, __LINE__ );
}

int ResourceManager::FindFreeSessionId() const NN_NOEXCEPT
{
    for(int i = 0; i < MaxSessions; i++ )
    {
        if (!m_Sessions[i].IsOpened())
        {
            return i; // 見つけた
        }
    }

    return -1; // 全部オープン済だった
}

void ResourceManager::OpenSession(I2cSession* pOutSession, int busIdx, Bit16 slaveAddress, AddressingMode addressingMode, SpeedMode speedMode ) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock( m_SessionIsOpenMutex );

    NN_SDK_REQUIRES_NOT_NULL(pOutSession);
    NN_SDK_REQUIRES(busIdx < MaxBuses);

    int sessionId = FindFreeSessionId();
    NN_ABORT_UNLESS( 0 <= sessionId && sessionId < MaxSessions, "No available sessions on I2C library");

    pOutSession->_sessionId = sessionId;
    m_Sessions[sessionId].Open(busIdx, slaveAddress, addressingMode, speedMode, &m_BusAccessor[busIdx]);

    NN_DETAIL_I2C_LOG("%s(%d) OpenSession at sessionId=%d\n", __FUNCTION__, __LINE__, sessionId );
}


void ResourceManager::CloseSession( const I2cSession& session ) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock( m_SessionIsOpenMutex );

    auto      sessionId = session._sessionId; // ユーザー領域へのアクセスは最小限に

    NN_SDK_ASSERT( m_Sessions[sessionId].IsOpened(), "Session Already Closed");

    m_Sessions[sessionId].Close();

    NN_DETAIL_I2C_LOG("%s(%d) CloseSession at sessionId=%d\n", __FUNCTION__, __LINE__, sessionId );
}

} // detail
} // driver
} // i2c
} // nn
