﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/account.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/os.h>

#include <nn/srepo.h>
#include <nn/srepo/srepo_StateNotifier.h>

TEST(Save, WithoutUser)
{
    nn::srepo::SystemReport report("srepo_basic");
    char buffer[1024];

    report.SetBuffer(buffer, sizeof (buffer));

    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x", 1.23));

    EXPECT_TRUE(nn::srepo::ResultInvalidKey::Includes(report.Add("*", 1.23)));
    EXPECT_TRUE(nn::srepo::ResultDuplicatedKey::Includes(report.Add("Position:x", 1.23)));
    EXPECT_TRUE(nn::srepo::ResultDuplicatedKey::Includes(report.Add("POSITION:X", 1.23)));

    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x123", 1.23));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x12", 1.2));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x1", 1.0));
    report.Clear();

    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x", 1.23));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:y", 0.00));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:z", -5.55));
    EXPECT_EQ(report.GetCount(), 3);
    report.Clear();

    nn::srepo::Any64BitId idDec = {1};
    nn::srepo::Any64BitId idHex = {0x1234567890ABCDEF};
    nn::srepo::Any64BitId idBeef = {0xbeefBEEFbeefBEEF};

    NNT_EXPECT_RESULT_SUCCESS(report.Add("id:dec", idDec));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("id:hex", idHex));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("id:beef", idBeef));
    EXPECT_EQ(report.GetCount(), 3);

    NNT_EXPECT_RESULT_SUCCESS(report.SetEventId("abcdefghijklmnopqrstuvwxyz_"));
    NNT_EXPECT_RESULT_SUCCESS(report.SetEventId("0123456789"));
    NNT_EXPECT_RESULT_SUCCESS(report.SetEventId("_"));

    EXPECT_TRUE(nn::srepo::ResultInvalidEventId::Includes(report.SetEventId("ABCDEFGHIJKLMNOPQRSTUVWXYZ")));
    EXPECT_TRUE(nn::srepo::ResultInvalidEventId::Includes(report.SetEventId(":")));

    NNT_EXPECT_RESULT_SUCCESS(report.SetEventId("srepo_basic"));

    EXPECT_TRUE(nn::srepo::ResultInvalidApplicationId::Includes(report.Save()));

    nn::ApplicationId applicationId = {0x0123456789abcdef};
    NNT_EXPECT_RESULT_SUCCESS(report.SetApplicationId(applicationId)); // Application ID を自己申告する。

    NNT_EXPECT_RESULT_SUCCESS(report.Save());
}

TEST(Save, WithUser)
{
    int userCount;
    nn::account::Uid users[nn::account::UserCountMax] = {};

    nn::account::InitializeForSystemService();

    NNT_ASSERT_RESULT_SUCCESS(nn::account::ListAllUsers(&userCount,
        users, nn::account::UserCountMax));
    if (userCount == 0)
    {
        NN_LOG("Skip the test because it needs one or more users.\n");
        return;
    }

    nn::srepo::SystemReport report("srepo_basic_with_user");
    char buffer[1024];

    report.SetBuffer(buffer, sizeof (buffer));

    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x", 1.23));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:y", 0.00));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:z", -5.55));
    EXPECT_EQ(report.GetCount(), 3);

    NNT_EXPECT_RESULT_SUCCESS(report.SetEventId("abcdefghijklmnopqrstuvwxyz_"));
    NNT_EXPECT_RESULT_SUCCESS(report.SetEventId("0123456789"));
    NNT_EXPECT_RESULT_SUCCESS(report.SetEventId("_"));

    NNT_EXPECT_RESULT_SUCCESS(report.SetEventId("srepo_basic"));

    NNT_EXPECT_RESULT_FAILURE(nn::srepo::ResultInvalidApplicationId, report.Save(users[0]));

    nn::ApplicationId applicationId = {0x0123456789abcdef};
    NNT_EXPECT_RESULT_SUCCESS(report.SetApplicationId(applicationId)); // Application ID を自己申告する。

    NNT_EXPECT_RESULT_SUCCESS(report.Save(users[0]));
}

TEST(StateDurationReport, NotifyApi) // NotifyXXX Api が叩けることだけ確認
{
    nn::account::InitializeForSystemService();

    auto GetUid = [](nn::account::Uid* pOut, int index) NN_NOEXCEPT
    {
        int len;
        nn::account::Uid uids[nn::account::UserCountMax];
        NNT_ASSERT_RESULT_SUCCESS(nn::account::ListAllUsers(&len, uids, NN_ARRAY_SIZE(uids)));
        ASSERT_GT(len, index);
        *pOut = uids[index];
    };

    {
        // NotifyNetworkRequestState
        nn::srepo::CompletedNetworkRequestType types[] =
        {
            nn::srepo::CompletedNetworkRequestType::None,
            nn::srepo::CompletedNetworkRequestType::WideAreaNetwork,
            nn::srepo::CompletedNetworkRequestType::LocalAreaNetwork,
            nn::srepo::CompletedNetworkRequestType::LocalDirectNetwork,
            nn::srepo::CompletedNetworkRequestType::NeighborDetection,
        };
        for(auto type : types)
        {
            nn::srepo::NotifyCompletedNetworkRequestChanged(type);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
            nn::srepo::NotifyCompletedNetworkRequestChanged(type);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        }
    }
    {
        // NotifyFriendPresenceChanged
        nn::account::Uid uid;
        ASSERT_NO_FATAL_FAILURE(GetUid(&uid, 0));
        nn::srepo::NotifyFriendPresenceChanged(uid, nn::srepo::FriendPresence::Online);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        nn::srepo::NotifyFriendPresenceChanged(uid, nn::srepo::FriendPresence::OnlinePlay);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        nn::srepo::NotifyFriendPresenceChanged(uid, nn::srepo::FriendPresence::Offline);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }
    {
        // NotifyNotificationConnectivityChanged
        nn::srepo::NotifyNotificationConnectivityChanged(nn::srepo::NotificationConnectivity::Available);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        nn::srepo::NotifyNotificationConnectivityChanged(nn::srepo::NotificationConnectivity::Unavailable);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }
    {
        // NotifyDeviceOperationModeChanged
        nn::srepo::NotifyDeviceOperationModeChanged(nn::srepo::DeviceOperationMode::Handheld);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        nn::srepo::NotifyDeviceOperationModeChanged(nn::srepo::DeviceOperationMode::Console);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }
    {
        // NotifySystemPowerStateChanged
        nn::srepo::NotifySystemPowerStateChanged(nn::srepo::SystemPowerState::FullAwake);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        nn::srepo::NotifySystemPowerStateChanged(nn::srepo::SystemPowerState::MinimumAwake);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        nn::srepo::NotifySystemPowerStateChanged(nn::srepo::SystemPowerState::MinimumAwakeForLowBatterySign);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        nn::srepo::NotifySystemPowerStateChanged(nn::srepo::SystemPowerState::MinimumAwakeForBackgroundTask);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        nn::srepo::NotifySystemPowerStateChanged(nn::srepo::SystemPowerState::SleepReady);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        nn::srepo::NotifySystemPowerStateChanged(nn::srepo::SystemPowerState::ShutdownReady);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }

    {
        // NotifyForegroundProgramChanged
        nn::ncm::ProgramId programId = {0x0123456789abcdef};
        nn::srepo::NotifyForegroundProgramChanged(programId);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        nn::srepo::NotifyForegroundProgramChanged(programId);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }
    {
        // NotifyControllerCountChanged
        nn::srepo::NotifyControllerCountChanged(2, 0);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        nn::srepo::NotifyControllerCountChanged(2, 1);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        nn::srepo::NotifyControllerCountChanged(2, 2);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }
} // NOLINT(impl/function_size)
