﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Runtime.Serialization;
using System.IO;
using Nintendo.Foundation.IO;

namespace ToolVersionChecker
{
    public class Program
    {
        public class CommandlineOptions
        {
            [CommandLineOption("git-path", IsRequired = true, Description = "Specify git.exe path.")]
            public string GitPath { get; set; }

            [CommandLineOption("repository-root", IsRequired = true, Description = "Specify the SigloSdk repository root path.")]
            public string RepositoryRoot { get; set; }

            [CommandLineOption("old-sdk-root", IsRequired = true, Description = "Specify the old NintendoSDK root path.")]
            public string OldSdkRootRoot { get; set; }

            [CommandLineOption("old-sdk-code-freeze-date", IsRequired = false, Description = "Specify the code freeze date for the old NintendoSDK. For example, 2016-06-22 00:00:00.")]
            public string OldSdkCodeFreezeDate { get; set; }

            [CommandLineOption("old-sdk-branch", IsRequired = false, Description = "Specify the previous release branch name. For example, origin/release/sdk/v1.0")]
            public string OldSdkBranch { get; set; }

            [CommandLineOption("target-tool-definition-file", IsRequired = true, Description = "Specify the target tool defition file.")]
            public string TargetToolDefinitionFile { get; set; }
        }

        [DataContract]
        public class ToolInfo
        {
            public ToolInfo(string toolPath)
            {
                ToolPath = toolPath;
                SourcePaths = new string[1] {
                    System.IO.Path.Combine(
                        "Programs/Iris/Sources",
                        System.IO.Path.GetDirectoryName(toolPath),
                        System.IO.Path.GetFileNameWithoutExtension(toolPath))
                };
            }

            [DataMember(Order = 0)]
            public string ToolPath { get; set; } = string.Empty;

            [DataMember(Order = 1)]
            public string[] SourcePaths { get; set; } = null;
        }

        [DataContract]
        public class ToolVersionCheckInfo
        {
            [DataMember(Order = 0)]
            public ToolInfo[] TargetToolInfos { get; set; }
        }

        static void Main()
        {
            CommandlineOptions options;
            try
            {
                if (!(new CommandLineParser().ParseArgs(Environment.GetCommandLineArgs().Skip(1), out options)))
                {
                    return;
                }
            }
            catch
            {
                return;
            }

            string gitPath = options.GitPath;
            if (!System.IO.File.Exists(gitPath))
            {
                WriteErrorLine($"{gitPath} が見つかりません。");
                return;
            }

            string reposRoot = options.RepositoryRoot;
            if (!System.IO.Directory.Exists(reposRoot))
            {
                WriteErrorLine($"{reposRoot} が見つかりません。");
                return;
            }

            string oldSdkRoot = options.OldSdkRootRoot;
            string oldSdkCodeFreezeDate = options.OldSdkCodeFreezeDate;
            string oldSdkBranch = options.OldSdkBranch;
            if (string.IsNullOrEmpty(oldSdkCodeFreezeDate) == string.IsNullOrEmpty(oldSdkBranch))
            {
                WriteErrorLine("--old-sdk-code-freeze-date か --old-sdk-branch のどちらか一方を指定してください");
                return;
            }

            ToolVersionCheckInfo checkInfo;
            try
            {
                checkInfo = DeserializeTargetToolInfos(options.TargetToolDefinitionFile);
            }
            catch (Exception exception)
            {
                WriteErrorLine($"{options.TargetToolDefinitionFile} のデシリアライズに失敗しました。\n{exception.ToString()}");
                return;
            }

            CheckToolVersion(
                checkInfo,
                reposRoot,
                oldSdkRoot,
                oldSdkCodeFreezeDate,
                oldSdkBranch,
                gitPath,
                null,
                Encoding.UTF8);
        }

        private static ToolVersionCheckInfo DeserializeTargetToolInfos(string path)
        {
            StreamReader reader = new StreamReader(path);
            string jsonString = reader.ReadToEnd();
            reader.Close();

            return JsonSerializerHelper.Deserialize<ToolVersionCheckInfo>(jsonString);
        }

        private class GitLogOptions
        {
            public string GitPath { get; set; } = string.Empty;

            public string After { get; set; } = string.Empty;

            public string Before { get; set; } = string.Empty;

            public string OldBranch { get; set; } = string.Empty;

            public int? MaxCount { get; set; } = null;

            public string TargetPath { get; set; } = string.Empty;

            public bool SkipesMergeLog { get; set; } = true;

            public bool IsCommandLineLogEnabled { get; set; } = false;

            public Encoding LogEncoding { get; set; } = Encoding.UTF8;

            public string[] ExceptStrings { get; set; } = null;
        }

        private static void WriteLog(string message)
        {
            System.Console.Write(message);
        }

        private static void WriteLogLine(string message)
        {
            WriteLog(message + "\n");
        }

        private static void WriteErrorLine(string message)
        {
            WriteLogLine("エラー: " + message);
        }

        private static void WriteWarningLine(string message)
        {
            WriteLogLine("警告: " + message);
        }

        private static void ReplaceSourcePath(
            ref List<string> paths,
            string containsStringForTarget, string replacePath)
        {
            int index3dEditor = paths.FindIndex(x => x.Contains(containsStringForTarget));
            if (index3dEditor != -1)
            {
                paths[index3dEditor] = replacePath;
            }
        }

        private static ToolVersion ExtractVersion(string log)
        {
            string[] spaceSplited = log.Split(' ');
            foreach (var spaceSplitedElem in spaceSplited)
            {
                string[] dotSplited = spaceSplitedElem.Split('.');
                if (dotSplited.Length >= 3)
                {
                    // . を 3 つ以上含む文字列をバージョン文字列としておく。必要があれば対応
                    return new ToolVersion(Int32.Parse(dotSplited[0]), Int32.Parse(dotSplited[1]), Int32.Parse(dotSplited[2]));
                }
            }

            return ToolVersion.InvalidVersion;
        }

        private static GitLog[] GetGitLogs(
            out string stdout, out string stderr,
            GitLogOptions options)
        {
            string logFormat = "%H %s(%cn, %cd)";
            string args = "--no-pager log --pretty=\"" + logFormat + "\"";
            if (!string.IsNullOrEmpty(options.After))
            {
                args += $" --after \"{options.After}\"";
            }
            if (!string.IsNullOrEmpty(options.Before))
            {
                args += $" --before \"{options.Before}\"";
            }
            if (!string.IsNullOrEmpty(options.OldBranch))
            {
                args += $" {options.OldBranch}..HEAD";
            }
            if (options.MaxCount != null)
            {
                args += $" --max-count {options.MaxCount}";
            }
            args += " \"" + options.TargetPath + "\"";

            string standardOut, standardError;
            Utility.ExecuteCommand(
                out standardOut,
                out standardError,
                options.GitPath, args, options.TargetPath,
                options.LogEncoding);
            stdout = "\"" + options.GitPath + "\" " + args + "\n" + standardOut;
            stderr = standardError;

            GitLog[] tmpLogs;
            Utility.ConvertToGitLogs(out tmpLogs, standardOut, options.SkipesMergeLog);
            return tmpLogs;
        }
        private static void MergeLogsByTaskUrl(ref List<GitLog> gitLogs, GitLog[] tmpLogs)
        {
            foreach (var log in tmpLogs)
            {
                GitLog matchedLog = gitLogs.FirstOrDefault(x => x.TaskUrl == log.TaskUrl);
                if (matchedLog != null)
                {
                    matchedLog.Message += "\n" + log.Message;
                }
                else
                {
                    gitLogs.Add(log);
                }
            }
        }

        private static string CreateGitLogTextForTask(
            GitLogOptions options,
            params string[] paths)
        {
            List<GitLog> gitLogs = new List<GitLog>();
            StringBuilder logStringBuilder = new StringBuilder();
            foreach (var path in paths)
            {
                string standardOut, standardError;
                options.TargetPath = path;
                GitLog[] tmpLogs = GetGitLogs(out standardOut, out standardError, options);
                if (options.IsCommandLineLogEnabled)
                {
                    logStringBuilder.AppendLine(standardOut);
                    logStringBuilder.AppendLine(standardError);
                }

                MergeLogsByTaskUrl(ref gitLogs, tmpLogs);
            }

            foreach (var log in gitLogs)
            {
                if (Utility.HasSubstrings(log.Message, options.ExceptStrings))
                {
                    continue;
                }

                logStringBuilder.AppendLine();
                logStringBuilder.AppendLine(log.Message);
                logStringBuilder.AppendLine(log.TaskUrl);
            }

            return logStringBuilder.ToString();
        }

        private static void CheckToolVersion(
            ToolVersionCheckInfo checkInfo,
            string reposRoot,
            string oldSdkRoot,
            string oldSdkCodeFreezeDate,
            string oldSdkBranch,
            string gitPath,
            string[] exceptStringsForGitLog,
            Encoding gitLogEncoding)
        {
            foreach (var toolInfo in checkInfo.TargetToolInfos)
            {
                string toolPath = toolInfo.ToolPath;
                WriteLogLine($"{System.IO.Path.GetFileName(toolPath)}");

                // バージョンの表示
                {
                    string oldVersionPath = System.IO.Path.Combine(oldSdkRoot, toolPath);
                    string currentVersionPath = System.IO.Path.Combine(reposRoot, toolPath);

                    ToolVersion oldVersion = null;
                    if (System.IO.File.Exists(oldVersionPath))
                    {
                        try
                        {
                            string standardOut, standardError;
                            Utility.ExecuteCommand(
                                out standardOut, out standardError,
                                oldVersionPath, "--version",
                                gitLogEncoding);
                            oldVersion = ExtractVersion(standardOut);
                        }
                        catch (Exception exception)
                        {
                            WriteWarningLine($"{oldVersionPath} --version の実行に失敗しました。{exception.Message}\nバージョン取得をスキップします。");
                        }
                    }
                    else
                    {
                        WriteWarningLine($"ファイル {oldVersionPath} が見つからないのでバージョン取得をスキップします");
                    }

                    ToolVersion currentVersion = null;
                    if (System.IO.File.Exists(currentVersionPath))
                    {
                        try
                        {
                            string standardOut, standardError;
                            Utility.ExecuteCommand(
                                out standardOut, out standardError,
                                currentVersionPath, "--version",
                                gitLogEncoding);
                            currentVersion = ExtractVersion(standardOut);
                        }
                        catch (Exception exception)
                        {
                            WriteWarningLine($"{currentVersionPath} --version の実行に失敗しました。{exception.Message}\nバージョン取得をスキップします。");
                        }
                    }
                    else
                    {
                        WriteWarningLine($"ファイル {currentVersionPath} が見つからないのでバージョン取得をスキップします");
                    }

                    WriteLogLine($"        {oldVersion} => {currentVersion}");
                }

                // ログの表示
                {
                    List<string> reposSourcePaths = new List<string>();
                    foreach (var sourcePath in toolInfo.SourcePaths)
                    {
                        string reposSourcePath = System.IO.Path.Combine(reposRoot, sourcePath);
                        if (!System.IO.Directory.Exists(reposSourcePath))
                        {
                            WriteErrorLine($"フォルダ {reposSourcePath} が見つかりません");
                            continue;
                        }

                        reposSourcePaths.Add(reposSourcePath);
                    }

                    string gitLog = CreateGitLogTextForTask(
                        new GitLogOptions()
                        {
                            After = oldSdkCodeFreezeDate,
                            Before = DateTime.Now.ToString(),
                            OldBranch = oldSdkBranch,
                            GitPath = gitPath,
                            LogEncoding = gitLogEncoding,
                            IsCommandLineLogEnabled = false,
                            SkipesMergeLog = true,
                            ExceptStrings = exceptStringsForGitLog
                        },
                        reposSourcePaths.ToArray());
                    gitLog = "        " + gitLog.Replace("\n", "\n        ");
                    WriteLogLine(gitLog);
                }
            }
        }
    }
}
