﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <vector>

#include <nn/nn_Log.h>

#include "TestAppSmallSimple_CommonUtil.h"
#include "TestAppSmallSimple_RomInfoUtil.h"

#include "../TestApplicationSimple/TestAppSimple_Utility.h"
#include "../TestApplicationSimple/TestAppSimple_FsUtilities.h"

void MetaInfo::SetInitParam() NN_NOEXCEPT
{
    this->contentType = "???";
    this->id = "???";
    this->version = "";
    this->releaseVersion = "";
    this->privateVersion = "";
    this->createdTime = "???";
}

void MetaInfo::ConvertVersion() NN_NOEXCEPT
{
    VersionHolder vh;
    vh.Convert(this->releaseVersion, this->privateVersion, this->version);

    // それぞれの値を確定させる
    this->releaseVersion = Convert::ToString(vh.releaseVer);
    this->privateVersion = Convert::ToString(vh.privateVer);
    this->version = Convert::ToString(vh.unifiedVer);
}

void RomInfoUtil::ReadExpectedHashValue(char outVal[]) NN_NOEXCEPT
{
    auto result = CommonUtil::ReadExpectedHashValue(outVal, ROM_ROOT_NAME);
    if (result.IsFailure())
    {
        NN_LOG("[Error] RomInfoUtil::ReadExpectedHashValue() Failed : result = 0x%08x\n", result.GetInnerValueForDebug());
    }
}

void RomInfoUtil::GetRomDataHashValue(char outVal[]) NN_NOEXCEPT
{
    const auto dummyRomFilePath = std::string(ROM_ROOT_NAME) + ":/DummyData.dat";
    auto result = CommonUtil::GetFileDataHashValue(outVal, dummyRomFilePath);
    if (result.IsFailure())
    {
        NN_LOG("[Error] RomInfoUtil::GetRomDataHashValue() Failed : result = 0x%08x\n", result.GetInnerValueForDebug());
    }
}

void RomInfoUtil::GetMetaInfoData(MetaInfo& outMetaInfo) NN_NOEXCEPT
{
    // 初期値設定
    outMetaInfo.SetInitParam();

    nn::Result result;
    fsutil::File file;

    result = file.Open("Contents:/MetaList.txt", nn::fs::OpenMode_Read);
    if (result.IsFailure())
    {
        NN_LOG("[Error] GetMetaInfoData() : OpenFile() Fialed : Path = Contents:/MetaList.txt\n");
        return;
    }

    auto fileSize = file.GetSize();
    if (file.GetLastResult().IsFailure() || fileSize <= 0)
    {
        NN_LOG("[Error] GetMetaInfoData() : GetFileSize() Failed : fileSize = %lld\n", fileSize);
        return;
    }

    const size_t readSize = static_cast<size_t>(fileSize);
    std::unique_ptr<char[]> buf(new char[readSize + 1]);
    buf[readSize] = '\0';

    size_t outSize = 0;
    result = file.Read(&outSize, 0, buf.get(), readSize);
    if (result.IsFailure())
    {
        // ファイル読み込み処理で失敗
        NN_LOG("[Error] GetMetaInfoData() : ReadFile() Failed : readSize = %d\n", readSize);
        return;
    }

    ParseMetaInfoList(buf.get(), outMetaInfo);
}

void RomInfoUtil::ParseMetaInfoList(const std::string& inMetaFileData, MetaInfo& outMetaInfo)
{
    const char* divStr = "\"";
    const std::string& docStr = inMetaFileData;

    struct MetaSetting
    {
        std::string item;
        std::string value;
    };
    std::vector<MetaSetting> metaInfoList;

    std::string::size_type pos = 0;
    while (pos < docStr.size())
    {
        // 左辺値の取得
        pos = docStr.find(divStr, pos);
        if (pos == std::string::npos)
        {
            break;
        }

        ++pos;
        auto nextPos = docStr.find(divStr, pos);
        if (nextPos == std::string::npos)
        {
            break;
        }

        MetaSetting setting;
        setting.item = docStr.substr(pos, nextPos - pos);

        // 右辺値の取得
        pos = docStr.find(divStr, nextPos + 1);
        if (pos == std::string::npos)
        {
            break;
        }

        ++pos;
        nextPos = docStr.find(divStr, pos);
        if (nextPos == std::string::npos)
        {
            break;
        }

        setting.value = docStr.substr(pos, nextPos - pos);
        pos = nextPos + 1;

        metaInfoList.push_back(setting);
    }

    const auto np = std::string::npos;
    for (auto& setting : metaInfoList)
    {
        const auto& it = setting.item;

        // ラムダ式でバージョン系の条件判定を共通化しておく
        auto isIncludeVersion = [&] (const std::string& verStr) -> bool {
            if (it.find(verStr) != np && (it.find("App/") != np || it.find("SystemProgram/") != np))
            {
                return true;
            }
            return false;
        };

        if ("ContentType" == it)
        {
            outMetaInfo.contentType = setting.value;
        }
        else if (it.find("/ApplicationId") != np || it.find("/ProgramId") != np)
        {
            outMetaInfo.id = setting.value;
        }
        else if (isIncludeVersion("/Version"))
        {
            outMetaInfo.version = setting.value;
        }
        else if (isIncludeVersion("/ReleaseVersion"))
        {
            outMetaInfo.releaseVersion = setting.value;
        }
        else if (isIncludeVersion("/PrivateVersion"))
        {
            outMetaInfo.privateVersion = setting.value;
        }
        else if ("CreatedTime" == it)
        {
            outMetaInfo.createdTime = setting.value;
        }
    }

    // バージョン系の値を確定させる
    outMetaInfo.ConvertVersion();
}
