﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/init.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/oe/oe_Common.h>
#include <nn/oe/oe_ApplicationControlApis.h>

#include "TestAppSmallSimple_CommonUtil.h"
#include "TestAppSmallSimple_RomInfoUtil.h"
#include "TestAppSmallSimple_AocInfo.h"

namespace {

// ヒープサイズ
const size_t MemoryHeapSize = 8 * 1024 * 1024;
const size_t MallocHeapSize = 6 * 1024 * 1024;

void* Allocate(size_t size) NN_NOEXCEPT
{
    return std::malloc(size);
}

void Deallocate(void* addr, size_t) NN_NOEXCEPT
{
    std::free(addr);
}

} // namespace

// アプリケーションのメモリ管理機構を初期化
extern "C" void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::SetMemoryHeapSize(MemoryHeapSize));

    uintptr_t address = uintptr_t();
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::AllocateMemoryBlock(&address, MallocHeapSize));

    nn::init::InitializeAllocator(
        reinterpret_cast<void*>(address), MallocHeapSize);
}

// アプリ情報をログ出力
void PrintAppInfo() NN_NOEXCEPT
{
    MetaInfo metaInfo;
    RomInfoUtil::GetMetaInfoData(metaInfo);

    NN_LOG(" CreatedTime    : %s\n", metaInfo.createdTime.c_str());
    NN_LOG(" ContentType    : %s\n", metaInfo.contentType.c_str());

    std::string idTypeStr = "ApplicationID ";
    if (metaInfo.contentType == "SystemProgram")
    {
        idTypeStr = "ProgramID     ";
    }
    NN_LOG(" %s : %s\n", idTypeStr.c_str(), metaInfo.id.c_str());

    NN_LOG(" Version        : %s\n", metaInfo.version.c_str());
    NN_LOG(" ReleaseVersion : %s\n", metaInfo.releaseVersion.c_str());
    NN_LOG(" PrivateVersion : %s\n", metaInfo.privateVersion.c_str());

    { // DisplayVersion を取得＆ログ出力
        nn::oe::DisplayVersion dv;
        nn::oe::GetDisplayVersion(&dv);
        NN_LOG(" DisplayVersion : %s\n", dv.value);
    }

    NN_LOG("\n");
}

// AddOnContent のハッシュ値チェック
void CheckAocData() NN_NOEXCEPT
{
    AocInfo ai;
    const auto aocCount = ai.GetCount();
    NN_LOG(" AddOnContent Count : %d\n", aocCount);
    if (aocCount == 0)
    {
        // コンテンツが無ければ意味が無いので以下のチェック処理は行わない
        NN_LOG("\n");
        return;
    }

    NN_LOG(" AocDataHash Check ... Start\n");

    const auto isOK = ai.CheckAllHash();
    if (isOK)
    {
        NN_LOG("  HashCheck(AocData): OK\n");
    }
    else
    {
        NN_ABORT("  HashCheck(AocData): NG\n");
    }

    NN_LOG(" AocDataHash Check ... End\n");
    NN_LOG("\n");
}

// (ダミー)ROMファイルのハッシュ値チェック
void CheckRomData() NN_NOEXCEPT
{
    NN_LOG(" RomHash Check ... Start\n");

    const size_t HashBufSize = HASH_STRING_SIZE + 1;
    char expHash[HashBufSize] = { 0 };
    RomInfoUtil::ReadExpectedHashValue(expHash);

    char actHash[HashBufSize] = { 0 };
    RomInfoUtil::GetRomDataHashValue(actHash);

    if (memcmp(expHash, actHash, HashBufSize) == 0)
    {
        NN_LOG("  HashCheck : OK\n");
    }
    else
    {
        NN_LOG("NG\n");
        NN_LOG("  Expected Hash = \"%s\"\n", expHash);
        NN_LOG("  Actual   Hash = \"%s\"\n", actHash);
        NN_ABORT("  HashCheck : NG");
    }

    NN_LOG(" RomHash Check ... End\n");
    NN_LOG("\n");
}

extern "C" void nnMain()
{
    nn::fs::SetAllocator(Allocate, Deallocate);

    // nn::fs の初期化
    size_t mountRomCacheBufferSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::QueryMountRomCacheSize(&mountRomCacheBufferSize));
    void* mountRomCacheBuffer = Allocate(mountRomCacheBufferSize);
    NN_ABORT_UNLESS_NOT_NULL(mountRomCacheBuffer);
    NN_UTIL_SCOPE_EXIT
    {
        Deallocate(mountRomCacheBuffer, mountRomCacheBufferSize);
    };
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom(ROM_ROOT_NAME, mountRomCacheBuffer, mountRomCacheBufferSize));

    // nn::oe の初期化 (For GetDisplayVersion)
    nn::oe::Initialize();

    // 自分自身のメタ情報の表示
    PrintAppInfo();

    // AddOnContent のハッシュ値チェック
    CheckAocData();

    // ROM ハッシュ値チェック
    CheckRomData();

    nn::fs::Unmount(ROM_ROOT_NAME);

    return;
}
