﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <nn/os.h>

class WorkContractor;
class Work
{
    friend class WorkContractor;
    NN_DISALLOW_COPY(Work);
    NN_DISALLOW_MOVE(Work);

public:
    Work() NN_NOEXCEPT;
    virtual ~Work() NN_NOEXCEPT;

    bool IsPrepared() const NN_NOEXCEPT;
    bool IsExecuted() const NN_NOEXCEPT;
    bool IsCancelRequested() const NN_NOEXCEPT;
    nn::Result GetLastResult() const NN_NOEXCEPT;
    nn::TimeSpan GetElapsedTime() const NN_NOEXCEPT;

    void Cleanup() NN_NOEXCEPT;

protected:
    nn::Result m_LastResult;

    virtual bool Exeute() NN_NOEXCEPT = 0;
    void SetStateToNotPrepared() NN_NOEXCEPT;
    void SetStateToPrepared() NN_NOEXCEPT;

private:
    enum State
    {
        State_NotPrepared,
        State_Prepared,
        State_Executing,
        State_Executed,
    };

    mutable nn::os::Mutex m_Lock;
    State m_State;
    nn::os::Tick m_Begin;
    nn::os::Tick m_End;
    bool m_IsCancelRequested;

    void SetStateToExecuting() NN_NOEXCEPT;
    void SetStateToExecuted() NN_NOEXCEPT;
    void RequestToCancel() NN_NOEXCEPT;
};

class WorkContractor
{
    NN_DISALLOW_COPY(WorkContractor);
    NN_DISALLOW_MOVE(WorkContractor);

public:
    WorkContractor(void* pStack, size_t stackSize) NN_NOEXCEPT;
    ~WorkContractor() NN_NOEXCEPT;

    void Initialize() NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

    void Start(Work* pWork) NN_NOEXCEPT;
    void Start(Work* pWorkArray[], size_t workCount) NN_NOEXCEPT;
    void WaitFinish() NN_NOEXCEPT;
    bool IsFinished() NN_NOEXCEPT;
    bool IsBusy() NN_NOEXCEPT;
    void Cancel() NN_NOEXCEPT;

private:
    enum State
    {
        State_NotInitialized,
        State_Runnable,
        State_Stopping,
    };

    nn::os::Mutex m_Lock;
    State m_State;
    void* m_pStackBuffer;
    size_t m_StackSize;
    std::vector<Work*> m_Works;
    nn::os::EventType m_StartEvent;
    nn::os::EventType m_FinishEvent;
    nn::os::ThreadType m_Thread;

    bool IsRunnable() NN_NOEXCEPT;
    bool RequestToStop() NN_NOEXCEPT;

    Work* FrontWork() NN_NOEXCEPT;
    bool PopFrontWork(Work* pWork) NN_NOEXCEPT;

    static void ThreadFunc(void* arg) NN_NOEXCEPT;
};
