﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/oe/oe_InStoreDemoApis.h>

#include "TestAppSimple_OperationEnvScene.h"

namespace {
    // 言語コードと表記の対応リスト定義
    struct LanguagePair
    {
        const char* code;
        const char* display;
    } LanguagePairList[] = {
        { "ja", "Japanese" },
        { "en-US", "AmericanEnglish" },
        { "fr", "French" },
        { "de", "German" },
        { "it", "Italian" },
        { "es", "Spanish" },
        { "ko", "Korean" },
        { "nl", "Dutch" },
        { "pt", "Portuguese" },
        { "ru", "Russian" },
        { "en-GB", "BritishEnglish" },
        { "fr-CA", "CanadianFrench" },
        { "es-419", "LatinAmericanSpanish" },
        { "zh-Hans", "SimplifiedChinese" },
        { "zh-Hant", "TraditionalChinese" },
    };

    // OperationMode と表記の対応リスト定義
    struct OperationModePair
    {
        nn::oe::OperationMode mode;
        const char* display;
    } OperationPairList[] = {
        { nn::oe::OperationMode_Handheld, "Handheld" },
        { nn::oe::OperationMode_Console, "Console" },
    };

    // PerformanceMode と表記の対応リスト定義
    struct PerformanceModePair
    {
        nn::oe::PerformanceMode mode;
        const char* display;
    } PerformancePairList[] = {
        { nn::oe::PerformanceMode_Invalid, "Invalid" },
#if (defined(NN_BUILD_CONFIG_SPEC_NX))
        { nn::oe::PerformanceMode_Normal, "Normal" },
        { nn::oe::PerformanceMode_Boost, "Boost" },
#endif // (defined(NN_BUILD_CONFIG_SPEC_NX))
    };

    // FocusState と表記の対応リスト定義
    struct FocusStatePair
    {
        nn::oe::FocusState state;
        const char* display;
    } FocusStatePairList[] = {
        { nn::oe::FocusState_InFocus, "InFocus" },
        { nn::oe::FocusState_OutOfFocus, "OutOfFocus" },
        { nn::oe::FocusState_Background, "Background" },
    };

    // FocusHandlingMode と表記の対応リスト定義
    struct FocusHandlingModePair
    {
        nn::oe::FocusHandlingMode mode;
        const char* display;
    } FocusHandlingModePairList[] = {
        { nn::oe::FocusHandlingMode_Suspend, "Suspend" },
        { nn::oe::FocusHandlingMode_Notify, "Notify" },
        { nn::oe::FocusHandlingMode_SuspendAndNotify, "SuspendAndNotify" },
        { nn::oe::FocusHandlingMode_InFocusOnly, "InFocusOnly" },
    };

    // 起動パラメータ表示の行数
    const int LaunchParamDisplayLineCnt = 12;

    // 1行に表示する起動パラメータのバイトサイズ
    const int LaunchParamLineByteSize = 16;

    // 起動パラメータ表示の文字幅
    const float LaunchParamCharWidth = 16.0f;

    // 起動パラメータ16進数表記の文字数(スペースが入るので3文字分)
    const int LaunchParamHexCharCnt = 3;

    // 起動パラメータの最大表示行数
    int g_LaunchParamMaxLine;

    // スクロールを止める行数
    int g_LaunchParamMaxTopLine;

    // 起動パラメータ表示をスクロール
    void ScrollLaunchParamTopLine(int* pOutTopLine, int scrollLine) NN_NOEXCEPT
    {
        *pOutTopLine += scrollLine;

        if (*pOutTopLine < 0)
        {
            *pOutTopLine = 0;
        }
        else if (*pOutTopLine > g_LaunchParamMaxTopLine)
        {
            *pOutTopLine = g_LaunchParamMaxTopLine;
        }
    }
};

OperationEnvScene::OperationEnvScene() NN_NOEXCEPT
    : m_State(State_None),
      m_CurrentFocusHandlingModeValue(nn::oe::FocusHandlingMode_Suspend),
      m_FocusHandlingMode("Suspend")
{
}

void OperationEnvScene::InternalSetup() NN_NOEXCEPT
{
    // アプリ起動時に取得できる静的情報を取得しておく
    this->GetStaticInformation();

    // FocusHandlingMode を切り替えるタッチ位置の定義
    {
        m_FocusHandlingSwitchRange.pos.x = 60.0f;
        m_FocusHandlingSwitchRange.pos.y = 560.0f;
        m_FocusHandlingSwitchRange.labelStr = "Y: Switch Focus Handling";
    }

    // 試遊台向け体験版を終了するタッチ位置の定義
    {
        m_ExitInStoreDemoRange.pos.x = 880.0f;
        m_ExitInStoreDemoRange.pos.y = 560.0f;
        m_ExitInStoreDemoRange.labelStr = "X: Exit In Store Demo";
    }

    // 起動パラメータ関連タッチ位置の定義
    {
        // 表示ボタンの位置は高さをそろえるため項目表示部で指定
        m_ShowLaunchParamRange.labelStr = "B: Show Launch Parameter";

        m_HideLaunchParamRange.pos.x = 928.0f;
        m_HideLaunchParamRange.pos.y = 120.0f;
        m_HideLaunchParamRange.labelStr = "X: Hide Launch Parameter";

        m_ScrollDownLaunchParamRange.pos.x = 60.0f;
        m_ScrollDownLaunchParamRange.pos.y = 600.0f;
        m_ScrollDownLaunchParamRange.labelStr = "Scroll Down";

        m_ScrollUpLaunchParamRange.pos.x = 1020.0f;
        m_ScrollUpLaunchParamRange.pos.y = 600.0f;
        m_ScrollUpLaunchParamRange.labelStr = "Scroll Up";
    }

    m_WarningView.SetTitle("!! WARNING !!");
    m_WarningView.PushMessage("Do you really want to execute \"", White);
    m_WarningView.PushMessage("Exit In Store Demo", Coral);
    m_WarningView.PushMessage("\" ?", White);
}

void OperationEnvScene::GetStaticInformation() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // DesiredLanguage の取得
    {
        auto langCode = nn::oe::GetDesiredLanguage();
        // 画面表記用の文字列に変換しておく
        std::string codeStr = langCode.string;
        std::string displayStr;
        for (auto& langPair : LanguagePairList)
        {
            if (langPair.code == codeStr)
            {
                displayStr = langPair.display;
                break;
            }
        }
        // 画面に表示する文字列を作成
        m_DesiredLanguage = displayStr + " (" + codeStr + ")";
    }

    // DisplayVersion の取得
    {
        nn::oe::DisplayVersion version;
        nn::oe::GetDisplayVersion(&version);

        m_DisplayVersion = version.value;
    }

    // (SIGLO-77505)試遊台の判別
    m_InStoreDemoMode = nn::oe::IsInStoreDemoModeEnabled() ? "True" : "False";

    // (SIGLO-79014)起動パラメータ取得
    m_IsExistedParam = nn::oe::TryPopLaunchParameter(&m_RealParamSize, m_ParamBuf, sizeof(m_ParamBuf));

#else // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // Generi 版は適当な文字列を入れておく
    m_DesiredLanguage = "TestLanguage(test)";
    m_DisplayVersion = "99.999";
    m_InStoreDemoMode = "Generic";

    // 起動パラメータ
    strcpy(m_ParamBuf, "Generic can not use this function. This parameter is provisional data.");
    m_RealParamSize = strlen(m_ParamBuf);
    m_IsExistedParam = true;
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )

    m_DisplayTopLineNum = 0;

    g_LaunchParamMaxLine = static_cast<int>(ceil(static_cast<double>(m_RealParamSize) / LaunchParamLineByteSize));

    g_LaunchParamMaxTopLine = g_LaunchParamMaxLine - LaunchParamDisplayLineCnt;
    if (g_LaunchParamMaxTopLine < 0)
    {
        g_LaunchParamMaxTopLine = 0;
    }
}

void OperationEnvScene::SetOperationModeValue(nn::oe::OperationMode inMode) NN_NOEXCEPT
{
    for (auto& operationPair : OperationPairList)
    {
        if (operationPair.mode == inMode)
        {
            m_OperationMode = operationPair.display;
            break;
        }
    }
}

void OperationEnvScene::SetPerformanceModeValue(nn::oe::PerformanceMode inMode) NN_NOEXCEPT
{
    for (auto& performPair : PerformancePairList)
    {
        if (performPair.mode == inMode)
        {
            m_PerformanceMode = performPair.display;
            break;
        }
    }
}

void OperationEnvScene::SetFocusStateValue(nn::oe::FocusState inState) NN_NOEXCEPT
{
    for (auto& focusPair : FocusStatePairList)
    {
        if (focusPair.state == inState)
        {
            m_FocusState = focusPair.display;
            //NN_LOG("[Trace] SetFocusStateValue : %s\n", m_FocusState.c_str());
            break;
        }
    }
}

void OperationEnvScene::OperateFocusHandling(nn::oe::FocusHandlingMode inSetMode) NN_NOEXCEPT
{
    // FocusHandlingMode を設定しつつ、設定した状態の文字列を変換しておく
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    nn::oe::SetFocusHandlingMode(inSetMode);
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    m_CurrentFocusHandlingModeValue = inSetMode;

    for (auto& hanlingPair : FocusHandlingModePairList)
    {
        if (hanlingPair.mode == inSetMode)
        {
            m_FocusHandlingMode = hanlingPair.display;
            break;
        }
    }
}

void OperationEnvScene::SwitchFocusHandlingMode() NN_NOEXCEPT
{
    // 現在の状態を見て、状態を切り替える
    // ひとまず Notify ⇔ Suspend を相互に切り替える形にする
    nn::oe::FocusHandlingMode setMode;
    switch (m_CurrentFocusHandlingModeValue)
    {
    case nn::oe::FocusHandlingMode_Suspend:
    case nn::oe::FocusHandlingMode_SuspendAndNotify:
    case nn::oe::FocusHandlingMode_InFocusOnly:
        setMode = nn::oe::FocusHandlingMode_Notify;
        break;
    case nn::oe::FocusHandlingMode_Notify:
        setMode = nn::oe::FocusHandlingMode_Suspend;
        break;
    default:
        // あり得ないと思うが定義値以外の値が設定された場合は Suspend を設定しておく
        setMode = nn::oe::FocusHandlingMode_Suspend;
        break;
    }

    this->OperateFocusHandling(setMode);
}

void OperationEnvScene::InternalHandleNPad() NN_NOEXCEPT
{
    if (m_State == State_None)
    {
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Y::Mask))
        {
            m_State = State_SwitchingFocusHandling;
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::X::Mask))
        {
            m_State = State_ExitInStoreDemoReady;
        }
        else if (m_IsExistedParam == true &&
            HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask))
        {
            m_State = State_ShowLaunchParameter;
        }
    }
    else if (m_State == State_ExitInStoreDemoReady)
    {
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask))
        {
            m_State = State_ExitInStoreDemo;
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask))
        {
            m_State = State_None;
        }
    }
    else if (m_State == State_ShowLaunchParameter)
    {
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Up::Mask) ||
            HasHidControllerAnyButtons(nn::hid::NpadButton::StickLUp::Mask))
        {
            ScrollLaunchParamTopLine(&m_DisplayTopLineNum, -1);
        }
        else if (
            HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Down::Mask) ||
            HasHidControllerAnyButtons(nn::hid::NpadButton::StickLDown::Mask))
        {
            ScrollLaunchParamTopLine(&m_DisplayTopLineNum, +1);
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::X::Mask))
        {
            m_State = State_None;
        }
    }
}

void OperationEnvScene::InternalHandleTouchScreen() NN_NOEXCEPT
{
    if (m_State == State_None)
    {
        if (m_FocusHandlingSwitchRange.range.IsInRange(m_PreviousTouch) == true)
        {
            m_State = State_SwitchingFocusHandling;
        }
        else if (m_ExitInStoreDemoRange.range.IsInRange(m_PreviousTouch) == true)
        {
            m_State = State_ExitInStoreDemoReady;
        }
        else if (m_ShowLaunchParamRange.range.IsInRange(m_PreviousTouch) == true)
        {
            m_State = State_ShowLaunchParameter;
        }
    }
    else if (m_State == State_ExitInStoreDemoReady)
    {
        if (m_ConfirmOkRange.range.IsInRange(m_PreviousTouch) == true)
        {
            m_State = State_ExitInStoreDemo;
        }
        else if (m_ConfirmCancelRange.range.IsInRange(m_PreviousTouch) == true)
        {
            m_State = State_None;
        }
    }
    else if (m_State == State_ShowLaunchParameter)
    {
        if (m_ScrollUpLaunchParamRange.range.IsInRange(m_PreviousTouch) == true)
        {
            ScrollLaunchParamTopLine(&m_DisplayTopLineNum, -1);
        }
        else if (m_ScrollDownLaunchParamRange.range.IsInRange(m_PreviousTouch) == true)
        {
            ScrollLaunchParamTopLine(&m_DisplayTopLineNum, +1);
        }
        else if (m_HideLaunchParamRange.range.IsInRange(m_PreviousTouch) == true)
        {
            m_State = State_None;
        }
    }
}

void OperationEnvScene::InternalProcess() NN_NOEXCEPT
{
    if (m_State == State_SwitchingFocusHandling)
    {
        this->SwitchFocusHandlingMode();
        m_State = State_None;
    }
    else if (m_State == State_ExitInStoreDemo)
    {
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        nn::oe::ExitApplicationAndGoBackToInStoreDemoMenu();
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        m_State = State_None;
    }
}

void OperationEnvScene::InternalDrawDebugText(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    writer->SetTextColor(White);
    writer->SetCursor(40.0f, 75.0f);
    writer->Print("[OperationEnvironment]");

    if (m_State == State_ExitInStoreDemoReady)
    {
        // 警告画面の描画
        this->DrawWarningMessage(writer, &m_WarningView);

        // 以降の処理は書き出さない
        return;
    }
    else if (m_State == State_ShowLaunchParameter)
    {
        this->ShowLauchParam(writer);
        // 以降の処理は書き出さない
        return;
    }

    const float LineSize = 33.0f;
    int ypos = 0;

    {
        writer->SetCursor(50.0f, 110.0f + (LineSize * (ypos)));
        writer->Print("- Display Version : %s", m_DisplayVersion.c_str());
        ++ypos;
    }
    {
        writer->SetCursor(50.0f, 110.0f + (LineSize * (ypos)));
        writer->Print("- Desired Language : %s", m_DesiredLanguage.c_str());
        ++ypos;
    }
    {
        writer->SetCursor(50.0f, 110.0f + (LineSize * (ypos)));
        writer->Print("- Operation Mode : %s", m_OperationMode.c_str());
        ++ypos;
    }
    {
        writer->SetCursor(50.0f, 110.0f + (LineSize * (ypos)));
        writer->Print("- Performance Mode : %s", m_PerformanceMode.c_str());
        ++ypos;
    }
    {
        writer->SetCursor(50.0f, 110.0f + (LineSize * (ypos)));
        writer->Print("- FocusHandling Mode : %s", m_FocusHandlingMode.c_str());
        ++ypos;
    }
    {
        writer->SetCursor(50.0f, 110.0f + (LineSize * (ypos)));
        writer->Print("- In Store Demo Mode : %s", m_InStoreDemoMode.c_str());
        ++ypos;
    }
    {
        writer->SetCursor(50.0f, 110.0f + (LineSize * (ypos)));
        writer->Print("- Launch Parameter : ");
        ++ypos;

        if (m_IsExistedParam == true)
        {
            writer->Print("%d [Byte]", m_RealParamSize);

            m_ShowLaunchParamRange.pos = { 500.0f, writer->GetCursorY() };
            this->WriteTouchRange(writer, &m_ShowLaunchParamRange, true);
        }
        else
        {
            writer->Print("Not Existed");
        }
    }

    if (m_State == State_None)
    {
        writer->SetScale(2.0f, 2.0f);
        this->WriteTouchRange(writer, &m_FocusHandlingSwitchRange);
        this->WriteTouchRange(writer, &m_ExitInStoreDemoRange);
    }
}

void OperationEnvScene::ShowLauchParam(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    // 全体位置調整
    const Position BaseOffset = { 0.0f, 160.0f };
    // 個別位置調整
    const auto AddressPosX = BaseOffset.x + 40.0f;
    const auto HexParamPosX = AddressPosX + 112.0f;
    const auto AsciiParamPosX = HexParamPosX + 840.0f;
    const auto LineHeight = 32.0f;

    auto displayLineCnt = g_LaunchParamMaxLine;
    if (g_LaunchParamMaxLine > LaunchParamDisplayLineCnt)
    {
        displayLineCnt = LaunchParamDisplayLineCnt;

        // スクロールボタンの表示判別
        writer->SetScale(2.0f, 2.0f);
        if (m_DisplayTopLineNum > 0)
        {
            this->WriteTouchRange(writer, &m_ScrollUpLaunchParamRange);
        }
        if (m_DisplayTopLineNum < g_LaunchParamMaxTopLine)
        {
            this->WriteTouchRange(writer, &m_ScrollDownLaunchParamRange);
        }
    }

    writer->SetTextColor(White);
    writer->SetScale(1.5f, 1.5f);
    writer->SetCursor(500.0f, 120.0f);
    writer->Print("[Launch Parameter]");

    this->WriteTouchRange(writer, &m_HideLaunchParamRange);

    // 強制等幅描画
    writer->SetFixedWidth(LaunchParamCharWidth);
    writer->SetFixedWidthEnabled(true);
    for (int i = 0; i < displayLineCnt; i++)
    {
        // Address
        writer->SetTextColor(Gray);
        writer->SetCursor(AddressPosX, BaseOffset.y + LineHeight * i);
        auto lineFirstCharIdx = LaunchParamLineByteSize * (m_DisplayTopLineNum + i);
        writer->Print("%04x :", lineFirstCharIdx);

        writer->SetTextColor(White);
        auto remainingCharCnt = static_cast<int>(m_RealParamSize) - lineFirstCharIdx;
        auto displayCharCnt = remainingCharCnt > LaunchParamLineByteSize ?
            LaunchParamLineByteSize : remainingCharCnt;
        for (int j = 0; j < displayCharCnt; j++)
        {
            // Hex
            writer->SetCursorX(
                HexParamPosX + j * LaunchParamCharWidth * LaunchParamHexCharCnt
            );
            writer->Print("%02x", m_ParamBuf[lineFirstCharIdx + j]);

            // Ascii
            auto asciiChar = m_ParamBuf[lineFirstCharIdx + j];
            if (asciiChar <= 0x1f || asciiChar >= 0x7f)
            {
                asciiChar = '.';
            }
            writer->SetCursorX(AsciiParamPosX + j * LaunchParamCharWidth);
            writer->Print("%c", asciiChar);
        }
    }
    writer->SetFixedWidthEnabled(false);
}
