﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex> // for std::lock_guard<>

#include <nn/nn_Assert.h>
#include <nn/crypto/crypto_Sha1Generator.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>

#include "TestAppSimple_Hash.h"
#include "TestAppSimple_FsUtilities.h"

namespace {
    const size_t ThreadStackSize = 8192;              // スレッド操作スレッドのスタックサイズ
    NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[ThreadStackSize];       // RomHash 画面用のスレッドスタック
    NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStackForAoc[ThreadStackSize]; // AocInfo 画面用のスレッドスタック
}

Hash::Hash(const char* inFilePath) NN_NOEXCEPT
    : m_FilePath(inFilePath), m_IsProcessing(false), m_IsStop(true),
    m_IsHashGetSuccess(false), m_ProgressPercent(0), m_Mutex(true)
{
}

Hash::~Hash() NN_NOEXCEPT
{
    // 念のためスレッドの終了処理を呼んでおく
    // スレッドが動いていなければ何もしないはず
    this->StopSha1ProcessAsync();
}

int Hash::GetSha1(char* pOutStr, size_t inGetStrSize) NN_NOEXCEPT
{
    if (this->IsProcessing() == true)
    {
        // 呼ばれることはないと思うが、スレッド処理が動いている時はすぐに処理を返す
        return -1;
    }

    nn::Result result;
    fsutil::File file;

    result = file.Open(m_FilePath, nn::fs::OpenMode_Read);
    if (result.IsFailure())
    {
        return -1;
    }

    auto fileSize = file.GetSize();
    if (result.IsFailure() || fileSize < 0)
    {
        return -1;
    }

    const size_t workBufSize = 256 * 1024;
    void* workBuf = this->AllocateWorkHeap(workBufSize);

    nn::crypto::Sha1Generator sha1;
    sha1.Initialize();

    auto restSize = fileSize;
    size_t outSize = 0;
    size_t readSize = 0;
    int64_t offset = 0;
    do
    {
        // ファイル読み込み
        readSize = (restSize < workBufSize) ? static_cast<size_t>(restSize) : workBufSize;
        result = file.Read(&outSize, offset, workBuf, static_cast<size_t>(readSize));
        if (result.IsFailure())
        {
            // ファイル読み込み処理で失敗
            break;
        }
        restSize -= outSize;
        offset += outSize;

        // ハッシュの更新
        if (outSize > 0)
        {
            sha1.Update(workBuf, outSize);
        }
    } while (restSize > 0 && outSize > 0);

    file.Close();
    this->FreeWorkHeap(workBuf);

    if (result.IsFailure())
    {
        // ファイル読み込み処理で失敗していたら抜ける
        return -1;
    }

    // ハッシュ値(SHA1のチェックサム)を計算
    uint8_t hashVal[sha1.HashSize] = { 0 };
    sha1.GetHash(hashVal, sha1.HashSize);

    char tmpStr[Sha1MaxStrSize + 1] = { 0 };
    // ハッシュ値160bit(20byte)を16進文字列表記した40文字に加工する。
    for (int i = 0; i < sha1.HashSize; ++i)
    {
        nn::util::SNPrintf(tmpStr + (2 * i), 3, "%02X", hashVal[i]);
    }

    // そもそも最大文字数以上は変換できないため念のため上限を設定しておく
    int strSize = static_cast<int>(inGetStrSize);
    if (strSize > (Sha1MaxStrSize + 1))
    {
        strSize = (Sha1MaxStrSize + 1);
    }
    // 念のため変換元にヌル文字を設定しておく
    tmpStr[strSize - 1] = '\0';
    nn::util::Strlcpy(pOutStr, tmpStr, strSize);

    return 0;
}

void* Hash::AllocateWorkHeap(size_t inHeapSize) NN_NOEXCEPT
{
    return new uint8_t[inHeapSize];
}

void Hash::FreeWorkHeap(void* inPtr) NN_NOEXCEPT
{
    delete[] reinterpret_cast<uint8_t*>(inPtr);
}


// 非同期用メソッド
int Hash::StartSha1ProcessAsync() NN_NOEXCEPT
{
    // スレッドの作成
    auto result = nn::os::CreateThread(&m_Thread, ProcessGetSha1Static, this, g_ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority);
    NN_ASSERT(result.IsSuccess(), "Cannot create Thread");
    if (result.IsFailure())
    {
        // 作成に失敗した場合はすぐに抜ける
        return -1;
    }

    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        m_IsProcessing = true;
        m_IsStop = false;
        m_IsHashGetSuccess = false;
        m_ProgressPercent = 0;

        m_IsAocFileAccess = false;
    }

    // スレッドの実行を開始する
    nn::os::StartThread(&m_Thread);
    return 0;
}

int Hash::StartSha1ProcessAocAsync(const std::string& inMountName, int inIndex) NN_NOEXCEPT
{
    // スレッドの作成
    auto result = nn::os::CreateThread(&m_Thread, ProcessGetSha1Static, this, g_ThreadStackForAoc, ThreadStackSize, nn::os::DefaultThreadPriority);
    NN_ASSERT(result.IsSuccess(), "Cannot create Thread");
    if (result.IsFailure())
    {
        // 作成に失敗した場合はすぐに抜ける
        return -1;
    }

    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        m_IsProcessing = true;
        m_IsStop = false;
        m_IsHashGetSuccess = false;
        m_ProgressPercent = 0;

        m_IsAocFileAccess = true;
        m_AocMountName = inMountName;
        m_AocIndex = inIndex;
    }

    // スレッドの実行を開始する
    nn::os::StartThread(&m_Thread);
    return 0;
}

void Hash::StopSha1ProcessAsync() NN_NOEXCEPT
{
    if (m_IsStop == false)
    {
        m_IsStop = true;

        // スレッドが終了するのを待つ
        nn::os::WaitThread(&m_Thread);
        // スレッドを破棄する
        nn::os::DestroyThread(&m_Thread);
    }
}

bool Hash::IsProcessing() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    return m_IsProcessing;
}

int Hash::GetProgressPercent() NN_NOEXCEPT
{
    return m_ProgressPercent;
}

int Hash::GetSha1Data(char* pOutStr, size_t inGetStrSize) NN_NOEXCEPT
{
    if (this->IsProcessing() == true || m_IsHashGetSuccess == false)
    {
        return -1;
    }

    // そもそも最大文字数以上は変換できないため念のため上限を設定しておく
    int strSize = static_cast<int>(inGetStrSize);
    if (strSize > (Sha1MaxStrSize + 1))
    {
        strSize = (Sha1MaxStrSize + 1);
    }
    // 念のため変換元にヌル文字を設定しておく
    m_Sha1Data[strSize - 1] = '\0';
    nn::util::Strlcpy(pOutStr, m_Sha1Data, strSize);

    return 0;
}

void Hash::ProcessGetSha1Static(void* inPtr) NN_NOEXCEPT
{
    Hash* ptr = reinterpret_cast<Hash*>(inPtr);
    if (ptr != nullptr)
    {
        if (ptr->m_IsAocFileAccess == true)
        {
            fsutil::ScopedMountAoc sma(ptr->m_AocMountName, ptr->m_AocIndex);
            if (sma.GetLastResult().IsSuccess())
            {
                ptr->ProcessGetSha1();
            }
        }
        else
        {
            ptr->ProcessGetSha1();
        }
    }
}

void Hash::ProcessGetSha1() NN_NOEXCEPT
{
    nn::Result result;
    fsutil::File file;

    // スコープを抜けると設定フラグが反転するクラス定義
    class ScopedFlagHolder
    {
    public:
        ScopedFlagHolder(bool& inFlag, nn::os::Mutex& inMtx)
            : m_TargetFlagPtr(&inFlag), m_MtxPtr(&inMtx) {}
        ~ScopedFlagHolder()
        {
            std::lock_guard<nn::os::Mutex> lock(*m_MtxPtr);
            *m_TargetFlagPtr = !(*m_TargetFlagPtr);
        }
    private:
        bool* m_TargetFlagPtr;
        nn::os::Mutex* m_MtxPtr;
    };
    // m_IsProcessing のフラグは、この関数から抜けるタイミングで反転させる
    ScopedFlagHolder processingFlag(m_IsProcessing, m_Mutex);

    result = file.Open(m_FilePath, nn::fs::OpenMode_Read);
    if (result.IsFailure())
    {
        return;
    }

    auto fileSize = file.GetSize();
    if (result.IsFailure() || fileSize < 0)
    {
        return;
    }

    const size_t workBufSize = 256 * 1024;
    void* workBuf = this->AllocateWorkHeap(workBufSize);

    nn::crypto::Sha1Generator sha1;
    sha1.Initialize();

    auto restSize = fileSize;
    size_t outSize = 0;
    size_t readSize = 0;
    int64_t offset = 0;
    do
    {
        if (m_IsStop == true)
        {
            break;
        }

        // ファイル読み込み
        readSize = (restSize < workBufSize) ? static_cast<size_t>(restSize) : workBufSize;
        result = file.Read(&outSize, offset, workBuf, static_cast<size_t>(readSize));
        if (result.IsFailure())
        {
            // ファイル読み込み処理で失敗
            break;
        }
        restSize -= outSize;
        offset += outSize;

        // プログレス計算
        {
            m_ProgressPercent = static_cast<int>((offset * 100) / fileSize);
        }

        // ハッシュの更新
        if (outSize > 0)
        {
            sha1.Update(workBuf, outSize);
        }
    } while (restSize > 0 && outSize > 0);

    file.Close();
    this->FreeWorkHeap(workBuf);

    if (result.IsFailure() || m_IsStop == true)
    {
        // ファイル読み込み処理で失敗していたら抜ける
        return;
    }

    // ハッシュ値(SHA1のチェックサム)を計算
    uint8_t hashVal[sha1.HashSize] = { 0 };
    sha1.GetHash(hashVal, sha1.HashSize);

    // ハッシュ値160bit(20byte)を16進文字列表記した40文字に加工する。
    for (int i = 0; i < sha1.HashSize; ++i)
    {
        nn::util::SNPrintf(m_Sha1Data + (2 * i), 3, "%02X", hashVal[i]);
    }
    // 念のためヌル文字を設定しておく
    m_Sha1Data[Sha1MaxStrSize] = '\0';

    m_IsHashGetSuccess = true;
}
