﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>

#include "TestAppSimple_AccountManager.h"
#include "TestAppSimple_EcServiceConfig.h"
#include "TestAppSimple_EcServiceAccount.h"
#include "TestAppSimple_EcServiceCatalog.h"
#include "TestAppSimple_RapidJson.h"

#if defined( TESTAPP_ENABLE_EC_ACCESS )
#include <nn/oe.h>
#include <nn/ec.h>
#include <nn/ec/ec_ShopServiceAccessor.h>
#endif // defined( TESTAPP_ENABLE_EC_ACCESS )

namespace {

#if !defined( TESTAPP_ENABLE_EC_ACCESS )
    // ConsumableApi 検証用レスポンス
    const char ConsumableApiEmptyResponse[] =
    "{"
        "\"lang\":\"ja\","
        "\"consumables\":null"
    "}";

    // ConsumableApi 検証用レスポンス
    const char ConsumableApiDummyResponse[] =
    "{"
        "\"lang\" : \"ja\","
        "\"consumables\": ["
            "{"
                "\"consumable_id\": \"123456789abcdef0\","
                "\"name\": \"consumables_name_1\","
                "\"description\": \"consumables_description_1\","
                "\"disclaimer\": \"consumables_disclaimer_1\""
            "},"
            "{"
                "\"consumable_id\": \"123456789abcdef1\","
                "\"name\": \"consumables_name_2\","
                "\"description\": \"consumables_description_2\","
                "\"disclaimer\": \"consumables_disclaimer_2\""
            "},"
            "{"
                "\"consumable_id\": \"123456789abcdef2\","
                "\"name\": \"consumables_name_3\","
                "\"description\": \"consumables_description_3\","
                "\"disclaimer\": \"consumables_disclaimer_3\""
            "}"
        "]"
    "}";

    // ConsumableItemApi 検証用レスポンス
    const char ConsumableItemApiEmptyResponse[] =
    "{"
        "\"lang\":\"ja\","
        "\"length\":null,"
        "\"offset\":null,"
        "\"total\":null,"
        "\"consumable_items\":null"
    "}";

    // ConsumableItemApi 検証用レスポンス
    const char ConsumableItemApiDummyResponse[] =
    "{"
        "\"lang\": \"ja\","
        "\"length\": 3,"
        "\"offset\": 0,"
        "\"total\": 3,"
        "\"consumable_items\" : ["
            "{"
                "\"item_id\" : \"123456789abcdefa\","
                "\"name\" : \"consumable_items_name_1\","
                "\"ns_uid\" : 9223372036854775807,"
                "\"price\": {"
                    "\"sales_status\": \"onsale\","
                    "\"regular_price\": {"
                        "\"currency\": \"USD\","
                        "\"formatted_value\": \"$9.99\","
                        "\"raw_value\": \"9.99\""
                    "},"
                    "\"discount_price\": {"
                        "\"currency\": \"USD\","
                        "\"formatted_value\": \"$7.77\","
                        "\"raw_value\": \"7.77\","
                        "\"start_datetime\": \"2022-02-02T00:00:00-08:00\","
                        "\"end_datetime\": \"2022-04-01T23:59:59-08:00\""
                    "}"
                "}"
            "},"
            "{"
                "\"item_id\" : \"123456789abcdefb\","
                "\"name\" : \"consumable_items_name_2\","
                "\"ns_uid\" : 9223372036854775806,"
                "\"price\": {"
                    "\"sales_status\": \"sales_termination\","
                    "\"regular_price\": {"
                        "\"currency\": \"USD\","
                        "\"formatted_value\": \"$8.88\","
                        "\"raw_value\": \"8.88\""
                    "}"
                "}"
            "},"
            "{"
                "\"item_id\" : \"123456789abcdefc\","
                "\"name\" : \"consumable_items_name_3\","
                "\"ns_uid\" : 9223372036854775805,"
                "\"price\": {"
                    "\"sales_status\": \"unreleased\","
                    "\"regular_price\": {"
                        "\"currency\": \"USD\","
                        "\"formatted_value\": \"$6.66\","
                        "\"raw_value\": \"6.66\""
                    "},"
                    "\"discount_price\": {"
                        "\"currency\": \"USD\","
                        "\"formatted_value\": \"$3.33\","
                        "\"raw_value\": \"3.33\","
                        "\"start_datetime\": \"\","
                        "\"end_datetime\": \"\""
                    "}"
                "}"
            "}"
        "]"
    "}";

    void GetResponse(std::unique_ptr<char[]>& json, size_t& size, const char* response) NN_NOEXCEPT
    {
        size = std::strlen(response);
        int count = static_cast<int>(size) + 1;
        std::unique_ptr<char[]> buf(new char[count]);
        NN_ABORT_UNLESS(buf);

        NN_ABORT_UNLESS(nn::util::Strlcpy(buf.get(), response, count) < count);
        json = std::move(buf);
    }

    void GetConsumableApiResponse(std::unique_ptr<char[]>& json, size_t& size) NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(int, s_Index);
        GetResponse(json, size, (s_Index++ % 2) ? ConsumableApiEmptyResponse : ConsumableApiDummyResponse);
    }

    void GetConsumableItemApiResponse(std::unique_ptr<char[]>& json, size_t& size) NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(int, s_Index);
        GetResponse(json, size, (s_Index++ % 2) ? ConsumableItemApiEmptyResponse : ConsumableItemApiDummyResponse);
    }
#endif // !defined( TESTAPP_ENABLE_EC_ACCESS )

    // パーサーの文字列用バッファサイズ
    const size_t StringBufferSize = 1024;

    // Consumable API レスポンスのパーサー
    class EcConsumableResponseAdaptor
    {
    public:
        static const int NodeDepthMax = 16;
        static const int PathLengthMax = 256;
        typedef nn::http::json::JsonPath<NodeDepthMax, PathLengthMax, false> JsonPathType;

        EcConsumableResponseAdaptor() NN_NOEXCEPT
        {
        };

        void NotifyObjectBegin(const JsonPathType& path) NN_NOEXCEPT
        {
            if (path.Match("$"))
            {
                m_Response.reset(new EcConsumableResponse());
            }
            else if (path.Match(GetNextConsumablePath()))
            {
                m_Response->consumables.push_back(std::unique_ptr<EcConsumable>(new EcConsumable()));
            }
        }

        void Update(const JsonPathType& path, const char* value, int length) NN_NOEXCEPT
        {
            if (path.Match("$.lang"))
            {
                m_Response->language.assign(value, value + length);
            }
            else if (path.Match(GetCurrentConsumablePath("consumable_id")))
            {
                GetCurrentConsumable()->consumableId.assign(value, value + length);
            }
            else if (path.Match(GetCurrentConsumablePath("name")))
            {
                GetCurrentConsumable()->name.assign(value, value + length);
            }
            else if (path.Match(GetCurrentConsumablePath("description")))
            {
                GetCurrentConsumable()->description.assign(value, value + length);
            }
            else if (path.Match(GetCurrentConsumablePath("disclaimer")))
            {
                GetCurrentConsumable()->disclaimer.assign(value, value + length);
            }
        }

        void Update(const JsonPathType&, std::nullptr_t) NN_NOEXCEPT {}
        void Update(const JsonPathType&, bool) NN_NOEXCEPT {}
        void Update(const JsonPathType&, double) NN_NOEXCEPT {}
        void Update(const JsonPathType&, int64_t) NN_NOEXCEPT {}
        void Update(const JsonPathType&, uint64_t) NN_NOEXCEPT {}
        void NotifyObjectEnd(const JsonPathType&) NN_NOEXCEPT {}

        bool GetResponse(std::unique_ptr<EcConsumableResponse>& response) NN_NOEXCEPT
        {
            if (m_Response)
            {
                response = std::move(m_Response);
                return true;
            }
            return false;
        }

    private:
        char m_Path[PathLengthMax];
        std::unique_ptr<EcConsumableResponse> m_Response;

        const char* GetNextConsumablePath() NN_NOEXCEPT
        {
            int index = static_cast<int>(m_Response->consumables.size());
            NN_ABORT_UNLESS(nn::util::SNPrintf(m_Path, sizeof(m_Path), "$.consumables[%d]", index) < sizeof(m_Path));
            return m_Path;
        }

        const char* GetCurrentConsumablePath(const char* key) NN_NOEXCEPT
        {
            int index = static_cast<int>(m_Response->consumables.size()) - 1; // 空の場合は不正なパスになるがワザと
            NN_ABORT_UNLESS(nn::util::SNPrintf(m_Path, sizeof(m_Path), "$.consumables[%d].%s", index, key) < sizeof(m_Path));
            return m_Path;
        }

        EcConsumable* GetCurrentConsumable() NN_NOEXCEPT
        {
            auto empty = !m_Response || m_Response->consumables.empty();
            NN_ASSERT(!empty);
            return !empty ? m_Response->consumables.back().get() : nullptr;
        }
    };

    // ConsumableItem API レスポンスのパーサー
    class EcConsumableItemResponseAdaptor
    {
    public:
        static const int NodeDepthMax = 16;
        static const int PathLengthMax = 256;
        typedef nn::http::json::JsonPath<NodeDepthMax, PathLengthMax, false> JsonPathType;

        EcConsumableItemResponseAdaptor() NN_NOEXCEPT
        {
        };

        void NotifyObjectBegin(const JsonPathType& path) NN_NOEXCEPT
        {
            if (path.Match("$"))
            {
                m_Response.reset(new EcConsumableItemResponse());
            }
            else if (path.Match(GetNextConsumableItemPath()))
            {
                m_Response->consumableItems.push_back(std::unique_ptr<EcConsumableItem>(new EcConsumableItem()));
            }
            else if (path.Match(GetCurrentConsumableItemPath("price")))
            {
                GetCurrentConsumableItem()->price.reset(new EcConsumableItemPrice());
            }
            else if (path.Match(GetCurrentConsumableItemPath("price.regular_price")))
            {
                GetCurrentConsumableItem()->price->regularPrice.reset(new EcConsumableItemRegularPrice());
            }
            else if (path.Match(GetCurrentConsumableItemPath("price.discount_price")))
            {
                GetCurrentConsumableItem()->price->discountPrice.reset(new EcConsumableItemDiscountPrice());
            }
        }

        void Update(const JsonPathType& path, int64_t value) NN_NOEXCEPT
        {
            if (path.Match("$.length"))
            {
                m_Response->length = value;
            }
            else if (path.Match("$.offset"))
            {
                m_Response->offset = value;
            }
            else if (path.Match("$.total"))
            {
                m_Response->total = value;
            }
            else if (path.Match(GetCurrentConsumableItemPath("ns_uid")))
            {
                GetCurrentConsumableItem()->nsUid = value;
            }
        }

        void Update(const JsonPathType& path, const char* value, int length) NN_NOEXCEPT
        {
            if (path.Match("$.lang"))
            {
                m_Response->language.assign(value, value + length);
            }
            else if (path.Match(GetCurrentConsumableItemPath("item_id")))
            {
                GetCurrentConsumableItem()->itemId.assign(value, value + length);
            }
            else if (path.Match(GetCurrentConsumableItemPath("name")))
            {
                GetCurrentConsumableItem()->name.assign(value, value + length);
            }
            else if (path.Match(GetCurrentConsumableItemPath("price.sales_status")))
            {
                GetCurrentConsumableItem()->price->salesStatus.assign(value, value + length);
            }
            else if (path.Match(GetCurrentConsumableItemPath("price.regular_price.currency")))
            {
                GetCurrentConsumableItem()->price->regularPrice->currency.assign(value, value + length);
            }
            else if (path.Match(GetCurrentConsumableItemPath("price.regular_price.formatted_value")))
            {
                GetCurrentConsumableItem()->price->regularPrice->formattedValue.assign(value, value + length);
            }
            else if (path.Match(GetCurrentConsumableItemPath("price.regular_price.raw_value")))
            {
                GetCurrentConsumableItem()->price->regularPrice->rawValue.assign(value, value + length);
            }
            else if (path.Match(GetCurrentConsumableItemPath("price.discount_price.currency")))
            {
                GetCurrentConsumableItem()->price->discountPrice->currency.assign(value, value + length);
            }
            else if (path.Match(GetCurrentConsumableItemPath("price.discount_price.formatted_value")))
            {
                GetCurrentConsumableItem()->price->discountPrice->formattedValue.assign(value, value + length);
            }
            else if (path.Match(GetCurrentConsumableItemPath("price.discount_price.raw_value")))
            {
                GetCurrentConsumableItem()->price->discountPrice->rawValue.assign(value, value + length);
            }
            else if (path.Match(GetCurrentConsumableItemPath("price.discount_price.start_datetime")))
            {
                GetCurrentConsumableItem()->price->discountPrice->startDatetime.assign(value, value + length);
            }
            else if (path.Match(GetCurrentConsumableItemPath("price.discount_price.end_datetime")))
            {
                GetCurrentConsumableItem()->price->discountPrice->endDatetime.assign(value, value + length);
            }
        }

        void Update(const JsonPathType&, std::nullptr_t) NN_NOEXCEPT {}
        void Update(const JsonPathType&, bool) NN_NOEXCEPT {}
        void Update(const JsonPathType&, double) NN_NOEXCEPT {}
        void Update(const JsonPathType&, uint64_t) NN_NOEXCEPT {}
        void NotifyObjectEnd(const JsonPathType&) NN_NOEXCEPT {}

        bool GetResponse(std::unique_ptr<EcConsumableItemResponse>& response) NN_NOEXCEPT
        {
            if (m_Response)
            {
                response = std::move(m_Response);
                return true;
            }
            return false;
        }

    private:
        char m_Path[PathLengthMax];
        std::unique_ptr<EcConsumableItemResponse> m_Response;

        const char* GetNextConsumableItemPath() NN_NOEXCEPT
        {
            int index = static_cast<int>(m_Response->consumableItems.size());
            NN_ABORT_UNLESS(nn::util::SNPrintf(m_Path, sizeof(m_Path), "$.consumable_items[%d]", index) < sizeof(m_Path));
            return m_Path;
        }

        const char* GetCurrentConsumableItemPath(const char* key) NN_NOEXCEPT
        {
            int index = static_cast<int>(m_Response->consumableItems.size()) - 1; // 空の場合は不正なパスになるがワザと
            NN_ABORT_UNLESS(nn::util::SNPrintf(m_Path, sizeof(m_Path), "$.consumable_items[%d].%s", index, key) < sizeof(m_Path));
            return m_Path;
        }

        EcConsumableItem* GetCurrentConsumableItem() NN_NOEXCEPT
        {
            auto empty = !m_Response || m_Response->consumableItems.empty();
            NN_ASSERT(!empty);
            return !empty ? m_Response->consumableItems.back().get() : nullptr;
        }
    };

} // namespace <unnamed>


EcCataloger::EcCataloger() NN_NOEXCEPT
{
    m_Request = Request_None;
    m_pUserSelector = nullptr;
    m_Uid = nn::account::InvalidUid;
}

EcCataloger::~EcCataloger() NN_NOEXCEPT
{
}

void EcCataloger::PrepareToInquireList(UserSelector* pUserSelector) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pUserSelector);

    Prepare(Request_InquireList, nn::account::InvalidUid, pUserSelector, "");
}

void EcCataloger::PrepareToInquireList(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_ASSERT_NOT_EQUAL(uid, nn::account::InvalidUid);

    Prepare(Request_InquireList, uid, nullptr, "");
}

void EcCataloger::PrepareToInquireItem(const nn::account::Uid& uid, const EcCatalog& catalog) NN_NOEXCEPT
{
    NN_ASSERT_NOT_EQUAL(uid, nn::account::InvalidUid);
    NN_ASSERT(catalog.IsConsumableSelected());

    auto consumable = catalog.GetSelectedConsumable();
    NN_ASSERT_NOT_NULL(consumable);
    Prepare(Request_InquireItem, uid, nullptr, consumable->consumableId.c_str());
}

void EcCataloger::Prepare(Request request, const nn::account::Uid& uid, UserSelector* pUserSelector, const char* pConsumableId) NN_NOEXCEPT
{
    NN_ASSERT_NOT_EQUAL(request, Request_None);
    NN_ASSERT_NOT_NULL(pConsumableId);

    SetStateToPrepared();
    m_Request = request;
    m_pUserSelector = pUserSelector;
    m_Uid = uid;
    m_ConsumableId = pConsumableId;
    m_ErrorCode = nn::err::ErrorCode::GetInvalidErrorCode();
    m_ListReponse.reset(nullptr);
    m_ItemReponse.reset(nullptr);
}

const nn::err::ErrorCode& EcCataloger::GetErrorCode() const NN_NOEXCEPT
{
    NN_ASSERT(IsExecuted());
    return m_ErrorCode;
}

bool EcCataloger::UpdateCatalog(EcCatalog* catalog) NN_NOEXCEPT
{
    NN_ASSERT(IsExecuted());
    auto updated = false;
    if (m_LastResult.IsSuccess())
    {
        switch (m_Request)
        {
        case Request_InquireList:
            updated = UpdateCatalogList(catalog);
            break;

        case Request_InquireItem:
            updated = UpdateCatalogItem(catalog);
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }
    }
    return updated;
}

bool EcCataloger::UpdateCatalogList(EcCatalog* catalog) NN_NOEXCEPT
{
    if (m_ListReponse)
    {
        catalog->list = std::move(m_ListReponse);
        catalog->map.clear();
        catalog->selectedConsumableIndex = -1;
        return true;
    }
    return false;
}

bool EcCataloger::UpdateCatalogItem(EcCatalog* catalog) NN_NOEXCEPT
{
    if (m_ItemReponse)
    {
        auto it = catalog->map.find(m_ConsumableId);
        if (it != catalog->map.end())
        {
            it->second = std::move(m_ItemReponse);
        }
        else
        {
            catalog->map.emplace(m_ConsumableId, std::move(m_ItemReponse));
        }
        return true;
    }
    return false;
}

bool EcCataloger::Exeute() NN_NOEXCEPT
{
    if (m_pUserSelector != nullptr)
    {
        auto uid = m_pUserSelector->GetUid();
        NN_ASSERT_NOT_EQUAL(uid, nn::account::InvalidUid);
        m_Uid = uid;
    }
    switch (m_Request)
    {
    case Request_InquireList:
        m_LastResult = InquireList();
        break;

    case Request_InquireItem:
        m_LastResult = InquireItem();
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }
    return m_LastResult.IsSuccess();
}

nn::Result EcCataloger::InquireList() NN_NOEXCEPT
{
    std::unique_ptr<char[]> json;
    size_t size;
#if defined( TESTAPP_ENABLE_EC_ACCESS )
    char url[128];
    auto language = nn::oe::GetDesiredLanguage();
    const char* BasePath = "/v1/applications/{applicationId}/consumables?country={country}&lang=%s";
    NN_ABORT_UNLESS(nn::util::SNPrintf(url, sizeof(url), BasePath, language.string) < sizeof(url));
    NN_RESULT_DO(Inquire(json, size, url));
#else // defined( TESTAPP_ENABLE_EC_ACCESS )
    GetConsumableApiResponse(json, size);
#endif // defined( TESTAPP_ENABLE_EC_ACCESS )
    TESTAPP_EC_DEBUG_LOG("Response(%zu) =>\n%s\n\n", size, json.get());

    // レスポンスパース
    std::unique_ptr<char[]> buff(new char[StringBufferSize]);
    NN_ABORT_UNLESS(buff);

    json::MemoriedStreamForRapidJson stream(json.get(), size);
    stream.SetStringBuffer(buff.get(), StringBufferSize);

    EcConsumableResponseAdaptor adaptor;
    NN_RESULT_DO(json::ImportJsonByRapidJson(&adaptor, stream));

    adaptor.GetResponse(m_ListReponse);
    NN_ASSERT(m_ListReponse);
    NN_RESULT_SUCCESS;
}

nn::Result EcCataloger::InquireItem() NN_NOEXCEPT
{
    std::unique_ptr<char[]> json;
    size_t size;
#if defined( TESTAPP_ENABLE_EC_ACCESS )
    char url[128];
    auto language = nn::oe::GetDesiredLanguage();
    const char* BasePath = "/v1/consumables/%s/items?country={country}&lang=%s";
    NN_ABORT_UNLESS(nn::util::SNPrintf(url, sizeof(url), BasePath, m_ConsumableId.c_str(), language.string) < sizeof(url));
    NN_RESULT_DO(Inquire(json, size, url));
#else // defined( TESTAPP_ENABLE_EC_ACCESS )
    GetConsumableItemApiResponse(json, size);
#endif // defined( TESTAPP_ENABLE_EC_ACCESS )
    TESTAPP_EC_DEBUG_LOG("Response(%zu) =>\n%s\n\n", size, json.get());

    // レスポンスパース
    std::unique_ptr<char[]> buff(new char[StringBufferSize]);
    NN_ABORT_UNLESS(buff);

    json::MemoriedStreamForRapidJson stream(json.get(), size);
    stream.SetStringBuffer(buff.get(), StringBufferSize);

    EcConsumableItemResponseAdaptor adaptor;
    NN_RESULT_DO(json::ImportJsonByRapidJson(&adaptor, stream));

    adaptor.GetResponse(m_ItemReponse);
    NN_ASSERT(m_ItemReponse);
    NN_RESULT_SUCCESS;
}

nn::Result EcCataloger::Inquire(std::unique_ptr<char[]>& json, size_t& size, const char* url) NN_NOEXCEPT
{
#if defined( TESTAPP_ENABLE_EC_ACCESS )
    nn::ec::ShopServiceAccessor accessor;
    NN_RESULT_DO(accessor.Initialize(nn::ec::ShopService(nn::ec::ShopService::Type_Catalog)));

    nn::ec::ShopServiceAccessor::AsyncResponse response;
    NN_RESULT_DO(accessor.Request(&response, m_Uid, nn::ec::ShopService::Method_Get, url));

    NN_RESULT_TRY(response.GetSize(&size))
        NN_RESULT_CATCH(nn::ec::ResultShowErrorCodeRequired)
        {
            m_ErrorCode = response.GetErrorCode();
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    std::unique_ptr<char[]> buf(new char[size + 1]);
    NN_ABORT_UNLESS(buf);

    NN_RESULT_TRY(response.Get(buf.get(), size))
        NN_RESULT_CATCH(nn::ec::ResultShowErrorCodeRequired)
        {
            m_ErrorCode = response.GetErrorCode();
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY
    buf[size] = '\0';

    json = std::move(buf);
#else // defined( TESTAPP_ENABLE_EC_ACCESS )
    NN_UNUSED(json);
    NN_UNUSED(size);
    NN_UNUSED(url);
#endif // defined( TESTAPP_ENABLE_EC_ACCESS )
    NN_RESULT_SUCCESS;
}

EcConsumableItemShower::EcConsumableItemShower() NN_NOEXCEPT
{
    m_Request = Request_None;
    m_Uid = nn::account::InvalidUid;
}

EcConsumableItemShower::~EcConsumableItemShower() NN_NOEXCEPT
{
}

void EcConsumableItemShower::PrepareToConsumableItemDetail(const nn::ApplicationId& applicationId, const nn::account::Uid& uid, const EcCatalog& catalog) NN_NOEXCEPT
{
    NN_ASSERT_NOT_EQUAL(applicationId, nn::ApplicationId::GetInvalidId());
    NN_ASSERT_NOT_EQUAL(uid, nn::account::InvalidUid);
    NN_ASSERT(catalog.IsConsumableItemSelected());

    SetStateToPrepared();
    m_Request = Request_ConsumableItemDetail;
    m_ApplicationId = applicationId;
    m_Uid = uid;

    auto consumable = catalog.GetSelectedConsumable();
    NN_ASSERT_NOT_NULL(consumable);
    m_ConsumableId = consumable->consumableId;

    auto consumableItem = catalog.GetSelectedConsumableItem();
    NN_ASSERT_NOT_NULL(consumableItem);
    m_NsUid = consumableItem->nsUid;
}

bool EcConsumableItemShower::Exeute() NN_NOEXCEPT
{
    switch (m_Request)
    {
    case Request_ConsumableItemDetail:
        m_LastResult = ShowShopConsumableItemDetail();
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }
    return m_LastResult.IsSuccess();
}

nn::Result EcConsumableItemShower::ShowShopConsumableItemDetail() NN_NOEXCEPT
{
#if defined( TESTAPP_ENABLE_EC_ACCESS )
    AccountManager::UserInfo info;
    const auto opened = AccountManager::GetInstance().GetOpenedUserInfo(m_Uid, &info);
    nn::account::UserHandle handle;
    if (opened)
    {
        handle = info.handle;
    }
    else
    {
        NN_RESULT_DO(nn::account::OpenUser(&handle, m_Uid));
    }
    NN_UTIL_SCOPE_EXIT
    {
        if (!opened)
        {
            nn::account::CloseUser(handle);
        }
    };

    nn::ec::ConsumableId consumableId;
    nn::ec::NsUid nsuid;
    NN_ABORT_UNLESS(nn::util::Strlcpy(consumableId.value, m_ConsumableId.c_str(), sizeof(consumableId.value)) < sizeof(consumableId.value));
    nsuid.value = static_cast<nn::Bit64>(m_NsUid);

    nn::ec::ShowShopConsumableItemDetail(m_ApplicationId, consumableId, nsuid, handle);
#endif // defined( TESTAPP_ENABLE_EC_ACCESS )
    NN_RESULT_SUCCESS;
}
