﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "TestAppSimple_EcOcsiScene.h"
#include "TestAppSimple_EcServerAccessor.h"

EcOcsiScene::EcOcsiScene() NN_NOEXCEPT
    : m_State(State_None)
    , m_PageCount(0)
    , m_PageIndex(0)
    , m_LineCount(0)
    , m_LineIndex(0)
{
}

EcOcsiScene::~EcOcsiScene() NN_NOEXCEPT
{
}

void EcOcsiScene::InternalSetup() NN_NOEXCEPT
{
    // EcService 共通
    Position line;
    SetEcServiceHeader(&line, "EcOcsi");

    // EcOcsi 固有
    m_UpdateLable.base = line;
    m_UpdateLable.prefix = "Rights";
    m_UpdateLable.range.labelStr = "B: Update";
    m_CosumeLable.base = line;
    m_CosumeLable.base.x += 280.0f;
    m_CosumeLable.prefix = "";
    m_CosumeLable.range.labelStr = "X: Cosume";
    line.y += NewLine; // Rights [B:Update] [X:Cosume]
    line.y += (NewLine * LineCountMax); // Rights....
    line.y += NewLine; // RIGHT/LEFT/UP/DOWN: Move   A: Check

    m_PrevPageLabel.base = line;
    m_PrevPageLabel.prefix = "";
    m_PrevPageLabel.range.labelStr = "<-: BackPage";
    m_NextPageLabel.base = line;
    m_NextPageLabel.base.x += 1000.0f;
    m_NextPageLabel.prefix = "";
    m_NextPageLabel.range.labelStr = "->: NextPage";
}

void EcOcsiScene::InternalHandleNPad() NN_NOEXCEPT
{
    if (m_State != State_None)
    {
        // 処理中
    }
    else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickL::Mask))
    {
        m_State = State_PurchaseRights;
    }
    else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask))
    {
        m_State = State_CheckRight;
    }
    else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask))
    {
        m_State = State_InquireRights;
    }
    else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::X::Mask))
    {
        m_State = State_ConsumeRights;
    }
    else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Y::Mask))
    {
        m_State = State_SelectUser;
    }
    else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickR::Mask))
    {
        m_State = State_ShowError;
    }
    else if ((HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Right::Mask)
         || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLRight::Mask)))
    {
        MovePageRight();
    }
    else if ((HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Left::Mask)
        || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLLeft::Mask)))
    {
        MovePageLeft();
    }
    else if ((HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Down::Mask)
        || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLDown::Mask)))
    {
        MoveLineDown();
    }
    else if ((HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Up::Mask)
        || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLUp::Mask)))
    {
        MoveLineUp();
    }
}

void EcOcsiScene::InternalHandleTouchScreen() NN_NOEXCEPT
{
    auto touch = m_PreviousTouch;
    if (m_State != State_None)
    {
        // 処理中
    }
    else if (m_AccountLabel.range.IsInRange(touch))
    {
        m_State = State_SelectUser;
    }
    else if (m_UpdateLable.range.IsInRange(touch))
    {
        m_State = State_InquireRights;
    }
    else if (m_CosumeLable.range.IsInRange(touch))
    {
        m_State = State_ConsumeRights;
    }
    else if (m_NextPageLabel.range.IsInRange(touch))
    {
        MovePageRight();
    }
    else if (m_PrevPageLabel.range.IsInRange(touch))
    {
        MovePageLeft();
    }
}

void EcOcsiScene::InternalProcess() NN_NOEXCEPT
{
    switch (m_State)
    {
    case State_None:
        break;
    case State_SelectUser:
        EcServerAccessor::GetInstance().Execute(EcServerAccessor::Command_InquireWithSelectUser);
        break;
    case State_InquireRights:
        EcServerAccessor::GetInstance().Execute(EcServerAccessor::Command_InquireOcsiRights);
        break;
    case State_ConsumeRights:
        EcServerAccessor::GetInstance().Execute(EcServerAccessor::Command_ConsumeOcsiRights);
        break;
    case State_PurchaseRights:
        EcServerAccessor::GetInstance().Execute(EcServerAccessor::Command_PurchaseOcsiRights);
        break;
    case State_CheckRight:
        EcServerAccessor::GetInstance().SwitchOcsiRight(GetRightIndex());
        break;
    case State_ShowError:
        EcServerAccessor::GetInstance().ShowErrorViewer();
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
    m_State = State_None;
}

void EcOcsiScene::InternalDrawDebugText(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    Position line = BaseLine;
    DrawEcServiceHeader(&line, writer);

    // 操作
    auto& list = EcServerAccessor::GetInstance().GetOcsiRights();
    if (EcServerAccessor::GetInstance().IsUserValid())
    {
        WriteLabel(writer, &m_UpdateLable);
    }
    if (list.GetSelectedCount() > 0)
    {
        WriteLabel(writer, &m_CosumeLable);
    }
    line.y += NewLine;

    // 権利一覧
    for (int i = 0; i < m_LineCount; i++)
    {
        auto index = i + (m_PageIndex * LineCountMax);
        auto& info = list.infos[index];
        auto item = info.rightData.GetConsumableServiceItemId();

        // カーソル
        if (i == m_LineIndex)
        {
            writer->SetCursor(line.x + 10.0f, line.y);
            writer->Print("*");
        }

        // 消費済みなら以降の出力色を変更
        if (info.isConsumed)
        {
            writer->SetTextColor(Yellow);
        }

        // 行番号
        Position pos(line.x + Indent, line.y);
        writer->SetFixedWidthEnabled(true);
        writer->SetFixedWidth(16.0f);
        writer->SetCursor(pos.x, pos.y);
        writer->Print("%2d.", (index + 1));
        writer->SetFixedWidthEnabled(false);

        // チェックボックス
        writer->SetCursor(pos.x + 50.0f, pos.y);
        writer->Print("[");
        if (info.isSelected)
        {
            writer->SetCursor(pos.x + 68.0f, pos.y);
            writer->Print("+");
        }
        writer->SetCursor(pos.x + 95.0f, pos.y);
        writer->Print("]");

        // アイテム名（※最大16文字）
        writer->SetCursor(pos.x + 120.0f, pos.y);
        writer->Print(item.value);

        // 消費状態
        if (info.isConsumed)
        {
            writer->SetCursor(pos.x + 800.0f, pos.y);
            writer->Print("Consumed");
        }

        writer->SetTextColor(White);
        line.y += NewLine;
    }
    line.y += NewLine * (LineCountMax - m_LineCount);

    // 操作
    if (m_LineCount > 0)
    {
        writer->SetCursor(line.x, line.y);
        writer->Print("RIGHT/LEFT/UP/DOWN: Move   A: Check");
    }
    line.y += NewLine;
    if (m_PageIndex > 0)
    {
        WriteLabel(writer, &m_PrevPageLabel);
    }
    if ((m_PageIndex + 1) < m_PageCount)
    {
        WriteLabel(writer, &m_NextPageLabel);
    }
    line.y += NewLine;
} // NOLINT (readability/fn_size)

void EcOcsiScene::UpdateContents() NN_NOEXCEPT
{
    auto& list = EcServerAccessor::GetInstance().GetOcsiRights();
    if (list.count > 0)
    {
        m_PageCount = static_cast<int>((list.count - 1) / LineCountMax) + 1;
        m_PageIndex = 0;
        m_LineCount = std::min((list.count - (m_PageIndex * LineCountMax)), LineCountMax);
        m_LineIndex = 0;
    }
    else
    {
        m_PageCount = 0;
        m_PageIndex = 0;
        m_LineCount = 0;
        m_LineIndex = 0;
    }
}

void EcOcsiScene::MovePageRight() NN_NOEXCEPT
{
    auto& list = EcServerAccessor::GetInstance().GetOcsiRights();
    m_PageIndex = ((m_PageIndex + 1) < m_PageCount) ? (m_PageIndex + 1) : std::max((m_PageCount - 1), 0);
    m_LineCount = std::min((list.count - (m_PageIndex * LineCountMax)), LineCountMax);
    m_LineIndex = std::min(std::max(m_LineIndex, 0), (m_LineCount - 1));
}

void EcOcsiScene::MovePageLeft() NN_NOEXCEPT
{
    auto& list = EcServerAccessor::GetInstance().GetOcsiRights();
    m_PageIndex = (m_PageIndex > 0) ? (m_PageIndex - 1) : 0;
    m_LineCount = std::min((list.count - (m_PageIndex * LineCountMax)), LineCountMax);
    m_LineIndex = std::min(std::max(m_LineIndex, 0), (m_LineCount - 1));
}

void EcOcsiScene::MoveLineDown() NN_NOEXCEPT
{
    m_LineIndex = ((m_LineIndex + 1) < m_LineCount) ? (m_LineIndex + 1) : 0;
}

void EcOcsiScene::MoveLineUp() NN_NOEXCEPT
{
    m_LineIndex = (m_LineIndex > 0) ? (m_LineIndex - 1) : (m_LineCount - 1);
}

int EcOcsiScene::GetRightIndex() const NN_NOEXCEPT
{
    int index = -1;
    if ((m_PageIndex < m_PageCount) && (m_LineIndex < m_LineCount))
    {
        index = m_LineIndex + (m_PageIndex * LineCountMax);
    }
    return index;
}
