﻿using System;
using System.IO;
using System.Reflection;
using System.Text.RegularExpressions;

namespace DecodeStackTrace
{
    internal class EnvironmentInfo
    {
        internal static string SdkRootPath
        {
            get
            {
                return sdkRootPath.Value;
            }
        }

        private static string ToolChainsPath
        {
            get
            {
                return toolChainsPath.Value;
            }
        }

        private static string ClangToolsPath
        {
            get
            {
                return clangToolsPath.Value;
            }
        }

        internal static string ObjdumpPath
        {
            get
            {
                return objdumpPath.Value;
            }
        }

        internal static string Addr2LinePath
        {
            get
            {
                return addr2linePath.Value;
            }
        }

        internal static string NmPath
        {
            get
            {
                return nmPath.Value;
            }
        }

        private static Lazy<string> sdkRootPath = new Lazy<string>(GetSdkRootPath);

        private static Lazy<string> toolChainsPath = new Lazy<string>(GetToolChainsPath);

        private static Lazy<string> clangToolsPath = new Lazy<string>(GetClangToolsPath);

        private static Lazy<string> objdumpPath = new Lazy<string>(GetObjdumpPath);

        private static Lazy<string> addr2linePath = new Lazy<string>(GetAddr2LinePath);

        private static Lazy<string> nmPath = new Lazy<string>(GetNmPath);

        private static string GetSdkRootPath()
        {
            var currentDirectory = Path.GetDirectoryName(Assembly.GetEntryAssembly().Location);
            var driveRootDirectory = Directory.GetDirectoryRoot(currentDirectory);
            do
            {
                if (File.Exists(Path.Combine(currentDirectory, "NintendoSdkRootMark")))
                {
                    return currentDirectory;
                }
                currentDirectory = Path.GetDirectoryName(currentDirectory);
            } while (currentDirectory != driveRootDirectory);
            throw new DirectoryNotFoundException();
        }

        private static string GetToolChainsPath()
        {
            var narlFilePath = Path.Combine(
                SdkRootPath, "Common", "Build", "Nact", "Environment", "TargetToolChainsInfo.autogen.narl");
            if (!File.Exists(narlFilePath))
            {
                throw new FileNotFoundException($"{narlFilePath} not found.");
            }

            var narlString = File.ReadAllText(narlFilePath);
            var m = Regex.Match(narlString, @"@public\s+Rynda\s+=\s+Constants\s+{\s+@public\s+Version\s+=\s+""(?<version>[^""]+)"";\s+};", RegexOptions.Multiline);
            if (!m.Success)
            {
                throw new InvalidDataException($"Failed to parse {narlFilePath}.");
            }
            var toolChainsRoot = GetToolChainsRoot();
            var path = Path.Combine(toolChainsRoot, m.Groups["version"].Value);
            if (!Directory.Exists(path))
            {
                throw new DirectoryNotFoundException($"{path} not found.");
            }
            return path;
        }

        private static string GetToolChainsRoot()
        {
            var path = GetToolChainsRootImpl();
            if (!Directory.Exists(path))
            {
                throw new DirectoryNotFoundException($"{path} not found.");
            }
            return path;
        }

        private static string GetToolChainsRootImpl()
        {
            var ryndaRoot = Environment.GetEnvironmentVariable("RYNDA_ROOT");
            if (ryndaRoot != null)
            {
                return ryndaRoot;
            }

            var sigloToolChainsRoot = Environment.GetEnvironmentVariable("SIGLO_TOOLCHAINS_ROOT");
            if (sigloToolChainsRoot != null)
            {
                return sigloToolChainsRoot;
            }

            return Path.Combine(SdkRootPath, "ToolChains");
        }

        private static string GetClangToolsPath()
        {
            var path = Path.Combine(ToolChainsPath, "nx", "aarch64", "bin");
            if (!Directory.Exists(path))
            {
                throw new DirectoryNotFoundException($"{path} not found.");
            }
            return path;
        }

        private static string GetObjdumpPath()
        {
            var path = Path.Combine(ClangToolsPath, "aarch64-nintendo-nx-elf-objdump.exe");
            if (!File.Exists(path))
            {
                throw new FileNotFoundException($"{path} not found.", path);
            }
            return path;
        }

        private static string GetAddr2LinePath()
        {
            var path = Path.Combine(ClangToolsPath, "aarch64-nintendo-nx-elf-addr2line.exe");
            if (!File.Exists(path))
            {
                throw new FileNotFoundException($"{path} not found.", path);
            }
            return path;
        }

        private static string GetNmPath()
        {
            var path = Path.Combine(ClangToolsPath, "aarch64-nintendo-nx-elf-nm.exe");
            if (!File.Exists(path))
            {
                throw new FileNotFoundException($"{path} not found.", path);
            }
            return path;
        }
    }
}
