﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Runtime.Serialization.Json;
using System.Threading;
using System.Threading.Tasks;
using System.Xml.Serialization;

namespace PCToolTester
{
    using JUnit;
    using Resources;

    public class Tester
    {
        private const string ErrorFormat = "{0}: Actual: {1} - Expected: {2}";
        private string basePath;
        private readonly ConstantDictionary constants = new ConstantDictionary();
        private readonly HashSet<string> processedPrepareEmptyDirectories = new HashSet<string>();

        public Tester()
        {
            this.ParallelExecuteCount = 1;
        }

        public int ParallelExecuteCount { get; set; }
        public bool Silent { get; set; }

        public bool Test(string inputFilePath, string outputFilePath, Constant[] defines = null)
        {
            this.basePath = Path.GetDirectoryName(inputFilePath);

            var stream = new FileStream(inputFilePath, FileMode.Open, FileAccess.Read);
            var serializer = new DataContractJsonSerializer(typeof(TestSet));
            var result = serializer.ReadObject(stream) as TestSet;

            if (result.Constants != null)
            {
                this.constants.Add(result.Constants);
            }

            var testsuite = this.ExecuteTests(result.Tests, defines);

            using (TextWriter writer = new StreamWriter(outputFilePath))
            {
                var xmlNamespace = new XmlSerializerNamespaces();
                xmlNamespace.Add(string.Empty, string.Empty);

                var xmlSerializer = new XmlSerializer(typeof(testsuite));
                xmlSerializer.Serialize(writer, testsuite, xmlNamespace);
            }

            return int.Parse(testsuite.errors) == 0 && int.Parse(testsuite.failures) == 0;
        }

        /// <summary>
        /// テストを実行します。
        /// </summary>
        /// <param name="tests"></param>
        /// <param name="defines"></param>
        /// <returns></returns>
        private testsuite ExecuteTests(IEnumerable<Test> tests, Constant[] defines = null)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            var testcases = new List<testcase>();
            var tasks = new List<Task<TestResult>>();
            var token = new CancellationToken();

            var workerThreadCount = 0;
            var completionPortThreadCount = 0;
            ThreadPool.GetMaxThreads(out workerThreadCount, out completionPortThreadCount);
            if (this.ParallelExecuteCount > workerThreadCount)
            {
                ThreadPool.SetMaxThreads(this.ParallelExecuteCount, completionPortThreadCount);
            }

            var lockObject = new object();
            var processingTestCount = 0;
            var processedTestCount = 0;
            var waitingTest = new Queue<Test>(tests);
            var displayProgressMessage = false;

            while (processedTestCount < tests.Count())
            {
                while (processingTestCount < ParallelExecuteCount && waitingTest.Count > 0)
                {
                    var test = waitingTest.Dequeue();
                    var task = Task.Run(() => TaskMain(test, defines), token);

                    task.ContinueWith(t =>
                        {
                            lock (lockObject)
                            {
                                processingTestCount--;
                                processedTestCount++;
                                displayProgressMessage = true;
                            }
                        });

                    tasks.Add(task);
                    lock (lockObject)
                    {
                        processingTestCount++;
                        displayProgressMessage = true;
                    }
                }

                lock (lockObject)
                {
                    if (displayProgressMessage == true)
                    {
                        this.ProgressMessage(waitingTest.Count, processingTestCount, processedTestCount, tests.Count());
                        displayProgressMessage = false;
                    }
                }

                Thread.Sleep(10);
            }

            this.ProgressMessage(waitingTest.Count, processingTestCount, processedTestCount, tests.Count());

            // 結果を収集します。
            var errorCount = 0;
            var failureCount = 0;

            foreach (var task in tasks)
            {
                var testResult = task.Result;

                if (!this.Silent)
                {
                    testResult.OutputResult(Console.Out);
                }

                var testcase = new testcase();
                testcase.name = testResult.Name;
                testcase.time = string.Format("{0:f3}", testResult.ElapsedSecounds);
                testcases.Add(testcase);

                switch (testResult.Result)
                {
                    case TestResult.Results.Done:
                        break;

                    case TestResult.Results.Error:
                        var error = new error();
                        //error.type = "error type";
                        error.message = testResult.Message;
                        error.Text = this.ToXmlText(testResult.ExtraMessages);
                        testcase.Items = new[] { error };
                        testcase.ItemsState = new[] { ItemStates.error };
                        errorCount++;
                        break;

                    case TestResult.Results.Failure:
                        var failure = new failure();
                        //failure.type = "failure type";
                        failure.message = testResult.Message;
                        failure.Text = this.ToXmlText(testResult.ExtraMessages);
                        testcase.Items = new[] { failure };
                        testcase.ItemsState = new[] { ItemStates.failure };
                        failureCount++;
                        break;
                }
            }

            stopwatch.Stop();
            var totalTime = (double)stopwatch.ElapsedTicks / (double)Stopwatch.Frequency;

            var testsuite = new testsuite();
            testsuite.tests = tasks.Count().ToString();          // テストの総件数
            testsuite.time = string.Format("{0:f3}", totalTime); // テストの総時間
            testsuite.failures = failureCount.ToString();        // failureの件数
            testsuite.errors = errorCount.ToString();            // errorの件数
            testsuite.Items = testcases.ToArray();
            testsuite.ItemsKind = Enumerable.Repeat(ItemKinds.testcase, tasks.Count()).ToArray();

            return testsuite;
        }

        private void ProgressMessage(int waiting, int working, int completed, int totalCount)
        {
            if (this.Silent == false)
            {
                Console.WriteLine(MessageResource.Message_Progress, waiting, working, completed, totalCount);
            }
        }

        private string[] ToXmlText(IEnumerable<string> values)
        {
            if (values == null)
            {
                return null;
            }

            var list = new List<string>();
            var index = 0;
            var last = values.Count() - 1;
            foreach (var value in values)
            {
                if (index != last)
                {
                    list.Add(value + "\n");
                }
                else
                {
                    list.Add(value);
                }
                index++;
            }
            return list.ToArray();
        }

        /// <summary>
        /// タスクのメイン処理です。
        /// </summary>
        /// <param name="test"></param>
        /// <param name="defines"></param>
        /// <returns></returns>
        private TestResult TaskMain(Test test, Constant[] defines = null)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            var testResult = new TestResult();
            testResult.Name = test.Name;

            try
            {
                var testConstants = this.constants;

                // テスト毎に定義された定数が優先されます。
                if (test.Constants != null)
                {
                    testConstants = new ConstantDictionary(this.constants);
                    testConstants.Add(test.Constants);
                }

                // コマンドラインで定義された定数が優先されます。
                if (defines != null)
                {
                    testConstants = new ConstantDictionary(testConstants);
                    testConstants.Add(defines);
                }

                if (test.PrepareEmptyDirectories != null)
                {
                    foreach (var path in test.PrepareEmptyDirectories)
                    {
                        this.PrepareEmptyDirectory(path, testConstants);
                    }
                }

                var startInfo = new ProcessStartInfo();
                startInfo.FileName = PathUtility.GetFullFilePath(this.basePath, testConstants.ExpandConstants(test.ExecuteFilePath));
                startInfo.Arguments = testConstants.ExpandConstants(test.Arguments);
                startInfo.UseShellExecute = false;
                startInfo.CreateNoWindow = true;
                startInfo.RedirectStandardError = true;
                startInfo.WorkingDirectory = this.basePath;

                Process process = null;
                try
                {
                    process = Process.Start(startInfo);
                }
                catch (Exception ex)
                {
                    var message = string.Format(MessageResource.Message_Error_FailedExecuteFile, startInfo.FileName);
                    this.HandleException(testResult, ex, message);
                    return testResult;
                }

                var standardErrors = new List<string>();
                process.ErrorDataReceived += (s, e) =>
                    {
                        if (e.Data != null)
                        {
                            standardErrors.Add(e.Data);
                        }
                    };
                process.BeginErrorReadLine();

                while (process.HasExited == false)
                {
                    Thread.Sleep(10);
                }

                var compareFile = true;
                if (test.Result != null)
                {
                    int exitCode = int.Parse(test.Result);
                    if (exitCode != process.ExitCode)
                    {
                        standardErrors.Add(string.Format(MessageResource.Message_Error_UnmatchExitCode, process.ExitCode, exitCode));
                        testResult.Result = TestResult.Results.Error;
                        compareFile = false;
                    }
                }
                else
                {
                    if (process.ExitCode != 0)
                    {
                        standardErrors.Add(string.Format(MessageResource.Message_ReturnValue, process.ExitCode));
                        testResult.Result = TestResult.Results.Failure;
                        compareFile = false;
                    }
                }

                if (test.Outputs != null && compareFile == true)
                {
                    var errorMessages = this.CompareTest(test, testConstants);
                    if (errorMessages.Count() > 0)
                    {
                        standardErrors.AddRange(errorMessages);
                        testResult.Result = TestResult.Results.Error;
                    }
                }

                if (standardErrors.Count() > 0)
                {
                    testResult.ExtraMessages = standardErrors;
                }
            }
            catch (Exception ex)
            {
                this.HandleException(testResult, ex);
            }
            finally
            {
                stopwatch.Stop();
                testResult.ElapsedTicks = stopwatch.ElapsedTicks;
            }

            return testResult;
        }

        private void PrepareEmptyDirectory(string path, ConstantDictionary testConstants)
        {
            var expandedPath = PathUtility.GetFullFilePath(this.basePath, testConstants.ExpandConstants(path));
            var lowerCasePath = expandedPath.ToLower();

            // 同じパスが重複して指定されていないかチェックします。
            lock (this.processedPrepareEmptyDirectories)
            {
                if (this.processedPrepareEmptyDirectories.Contains(lowerCasePath))
                {
                    throw new ApplicationException(
                        string.Format(
                            MessageResource.Message_Error_PrepareEmptyDirectories_Duplicate,
                            path,
                            expandedPath));
                }

                this.processedPrepareEmptyDirectories.Add(lowerCasePath);
            }

            // 安全のため、パスに２階層以上のディレクトリ指定されているか確認します。
            // (ドライブレター + トップディレクトリ + その他 = 3)
            var pathComponents = expandedPath.Split(new[] { Path.DirectorySeparatorChar }, 3);
            if (pathComponents.Length != 3)
            {
                throw new ApplicationException(
                    string.Format(
                        MessageResource.Message_Error_PrepareEmptyDirectories_TooFewPathComponents,
                        path,
                        expandedPath));
            }

            if (Directory.Exists(expandedPath))
            {
                // 既に存在するときは内容を空にします。
                try
                {
                    var dirInfo = new DirectoryInfo(expandedPath);
                    foreach (var file in dirInfo.GetFiles())
                    {
                        file.Delete();
                    }
                    foreach (var dir in dirInfo.GetDirectories())
                    {
                        dir.Delete(recursive: true);
                    }
                }
                catch (Exception ex)
                {
                    var message = string.Format(
                        MessageResource.Message_Error_PrepareEmptyDirectories_FailedPrepare,
                        path,
                        expandedPath);
                    throw new ApplicationException(message, ex);
                }
            }
            else
            {
                // 無いときは空のディレクトリを作成します。
                // 安全のため、指定パスのトップディレクトリが実際に存在する場合にのみ許可します。
                if (!PathUtility.IsAnyAnscestorDirectoryExist(expandedPath))
                {
                    throw new ApplicationException(
                        string.Format(
                            MessageResource.Message_Error_PrepareEmptyDirectories_NotFoundTop,
                            path,
                            expandedPath));
                }

                try
                {
                    Directory.CreateDirectory(expandedPath);
                }
                catch (Exception ex)
                {
                    var message = string.Format(
                        MessageResource.Message_Error_PrepareEmptyDirectories_FailedPrepare,
                        path,
                        expandedPath);
                    throw new ApplicationException(message, ex);
                }
            }
        }

        /// <summary>
        /// 比較を行います。
        /// </summary>
        /// <param name="test"></param>
        /// <returns></returns>
        private IEnumerable<string> CompareTest(Test test, ConstantDictionary testConstants)
        {
            var errorMessages = new List<string>();

            foreach (var output in test.Outputs)
            {
                string actualFullPath = PathUtility.GetFullFilePath(this.basePath, testConstants.ExpandConstants(output.Actual));
                string expectedFullPath = PathUtility.GetFullFilePath(this.basePath, testConstants.ExpandConstants(output.Expected));

                var isDirectoryActual = Directory.Exists(actualFullPath);
                var isDirectoryExpected = Directory.Exists(expectedFullPath);

                if (isDirectoryActual == true || isDirectoryExpected == true)
                {
                    if (isDirectoryActual != isDirectoryExpected)
                    {
                        // フォルダとファイルの比較をしようとしています。
                        errorMessages.Add(string.Format(ErrorFormat, MessageResource.Message_Error_NotCompareFileAndDirectory, output.Actual, output.Expected));
                    }
                    else
                    {
                        var listActual = new DirectoryInfo(actualFullPath).GetFiles("*.*", SearchOption.AllDirectories);
                        var listExpected = new DirectoryInfo(expectedFullPath).GetFiles("*.*", SearchOption.AllDirectories);

                        if (listActual.SequenceEqual(listExpected, new FileInfoNameComparer()) == false)
                        {
                            // フォルダ内のファイルが一致しません。
                            errorMessages.Add(string.Format(ErrorFormat, MessageResource.Message_Error_UnmatchFileCount, output.Actual, output.Expected));

                            var onlyActual = this.ExistsOnlyFilePaths(listActual, listExpected);
                            if (onlyActual.Count() > 0)
                            {
                                errorMessages.Add(MessageResource.Message_Error_ExistsActual);
                                errorMessages.AddRange(onlyActual);
                            }

                            var onlyExpectecd = this.ExistsOnlyFilePaths(listExpected, listActual);
                            if (onlyExpectecd.Count() > 0)
                            {
                                errorMessages.Add(MessageResource.Message_Error_ExistsExpected);
                                errorMessages.AddRange(onlyExpectecd);
                            }
                        }
                        else
                        {
                            for (int index = 0; index < listActual.Count(); index++)
                            {
                                var actual = listActual[index].FullName;
                                var expected = listExpected[index].FullName;

                                if (this.CompareFile(actual, expected) == false)
                                {
                                    errorMessages.Add(string.Format(ErrorFormat, MessageResource.Message_Error_NotMatchBinary, actual, expected));
                                }
                            }
                        }
                    }
                }
                else
                {
                    if (this.CompareFile(actualFullPath, expectedFullPath) == false)
                    {
                        errorMessages.Add(string.Format(ErrorFormat, MessageResource.Message_Error_NotMatchBinary, output.Actual, output.Expected));
                    }
                }
            }

            return errorMessages;
        }

        /// <summary>
        /// targetと比較して sourceにしか存在しないファイルを取得します。
        /// </summary>
        /// <param name="source"></param>
        /// <param name="target"></param>
        /// <returns></returns>
        private IEnumerable<string> ExistsOnlyFilePaths(IEnumerable<FileInfo> source, IEnumerable<FileInfo> target)
        {
            var hashSet = new HashSet<string>(target.Select(i => i.Name));
            return source
                .Where(i => hashSet.Contains(i.Name) == false)
                .Select(i => i.FullName);
        }

        /// <summary>
        /// ファイルをバイナリ比較します。
        /// </summary>
        /// <param name="filePathA"></param>
        /// <param name="filePathB"></param>
        /// <returns></returns>
        private bool CompareFile(string filePathA, string filePathB)
        {
            if (filePathA == filePathB)
            {
                return true;
            }

            FileStream streamA = null;
            FileStream streamB = null;
            try
            {
                streamA = new FileStream(filePathA, FileMode.Open);
                streamB = new FileStream(filePathB, FileMode.Open);

                if (streamA.Length != streamB.Length)
                {
                    return false;
                }

                var bufferSize = 1024 * 100;
                var bufferA = new byte[bufferSize];
                var bufferB = new byte[bufferSize];

                while (true)
                {
                    var countA = streamA.Read(bufferA, 0, bufferSize);
                    var countB = streamB.Read(bufferB, 0, bufferSize);
                    if (countA <= 0 && countB <= 0)
                    {
                        break;
                    }

                    if (countA != countB)
                    {
                        return false;
                    }

                    for (int index = 0; index < countA; index++)
                    {
                        if (bufferA[index] != bufferB[index])
                        {
                            return false;
                        }
                    }
                }

                return true;
            }
            finally
            {
                streamA.Close();
                streamB.Close();
            }
        }

        private void HandleException(TestResult testResult, Exception ex, string message = null)
        {
            var errorMessages = new List<string>();
            if (message != null)
            {
                errorMessages.Add(message);
            }
            errorMessages.Add(ex.Message);
            if (ex.InnerException != null)
            {
                errorMessages.Add("InnerException: " + ex.InnerException.ToString());
            }
            errorMessages.AddRange(ex.StackTrace.Replace(Environment.NewLine, "\n").Split('\n'));
            testResult.ExtraMessages = errorMessages;
            testResult.Result = TestResult.Results.Failure;
        }

        private class FileInfoNameComparer : EqualityComparer<FileInfo>
        {
            public override bool Equals(FileInfo a, FileInfo b)
            {
                return a.Name == b.Name;
            }

            public override int GetHashCode(FileInfo obj)
            {
                if (obj == null)
                {
                    throw new ArgumentNullException("obj");
                }
                return obj.GetHashCode();
            }
        }

        private class TestResult
        {
            public enum Results
            {
                Done,
                Error,
                Failure,
            }

            public TestResult()
            {
                Result = Results.Done;
            }

            public string Name { get; set; }
            public Results Result { get; set; }
            public long ElapsedTicks { get; set; }
            public string Message { get; set; }
            public IEnumerable<string> ExtraMessages { get; set; }

            public double ElapsedSecounds
            {
                get
                {
                    return (double)this.ElapsedTicks / (double)Stopwatch.Frequency;
                }
            }

            public void OutputResult(TextWriter textWriter)
            {
                textWriter.WriteLine($"Test: {this.Name} ({this.Result})");

                IEnumerable<string> messages = new[] { this.Message };

                if (this.ExtraMessages != null)
                {
                    messages = messages.Concat(this.ExtraMessages);
                }

                foreach (var msg in messages.Where(it => !string.IsNullOrWhiteSpace(it)))
                {
                    Console.WriteLine($"  {msg}");
                }
            }
        }
    }
}
