﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using Nintendo.Foundation.IO;
using PCToolTester.Resources;
using System;
using System.Diagnostics;
using System.Linq;

namespace PCToolTester
{

    internal class Program
    {
        private const int SuccessCode = 0;
        private const int ErrorCode = 1;

        public static int Main(string[] args)
        {
            try
            {
                var commandLineParameters = ParseCommand(args);
                if (commandLineParameters == null)
                {
                    return ErrorCode;
                }

                var tester = new Tester();
                if (commandLineParameters.Jobs != null)
                {
                    tester.ParallelExecuteCount = (int)commandLineParameters.Jobs;
                }
                tester.Silent = commandLineParameters.Silent;

                var filePath = PathUtility.GetFullFilePath(Environment.CurrentDirectory, commandLineParameters.FilePath);
                var output = PathUtility.GetFullFilePath(Environment.CurrentDirectory, commandLineParameters.Output);

                return tester.Test(filePath, output, commandLineParameters.Defines) ? SuccessCode : ErrorCode;
            }
            catch (Exception ex)
            {
                Console.Error.WriteLine(ex.Message);
                return ErrorCode;
            }
        }

        /// <summary>
        /// コマンドラインの解析を行います。
        /// </summary>
        /// <param name="args"></param>
        /// <returns></returns>
        private static CommandLineParameters ParseCommand(string[] args)
        {
            CommandLineParameters commandLineParameters = null;
            try
            {
                var commandLineParserSettings = new CommandLineParserSettings()
                {
                    ApplicationDescription = MessageResource.CommandLine_ApplicationDescription,
                };

                if (new CommandLineParser(commandLineParserSettings).ParseArgs(args, out commandLineParameters) == false)
                {
                    return null;
                }
            }
            catch
            {
                return null;
            }

            if (commandLineParameters.Defines != null)
            {
                var duplicatedDefine = commandLineParameters.Defines
                    .GroupBy(it => it.Name)
                    .FirstOrDefault(it => it.Take(2).Count() > 1);

                if (duplicatedDefine != null)
                {
                    // 同じ名前の定数値を２回以上定義することはできません('{0}')。
                    throw new ArgumentException(string.Format(MessageResource.Message_Error_CommandLine_DuplicateDefineName, duplicatedDefine.Key));
                }
            }

            return commandLineParameters;
        }

        /// <summary>
        /// コマンドラインのパラメータを格納するクラスです。
        /// </summary>
        private class CommandLineParameters
        {
            public CommandLineParameters()
            {
                this.FilePath = string.Empty;
                this.Output = string.Empty;
            }

            [CommandLineValue(0, IsRequired = true, ValueName = "input-filepath", Description = nameof(MessageResource.CommandLine_InputDescription), DescriptionConverterName = nameof(GetDescription))]
            public string FilePath { get; set; }

            [CommandLineOption('o', "output", IsRequired = true, ValueName = "output-filepath", Description = nameof(MessageResource.CommandLine_OutputDescription), DescriptionConverterName = nameof(GetDescription))]
            public string Output { get; set; }

            [CommandLineOption('j', "jobs", ValueName = "jobs", Description = nameof(MessageResource.CommandLine_JobsDescription), DescriptionConverterName = nameof(GetDescription))]
            public int? Jobs { get; set; }

            [CommandLineOption('s', "silent", ValueName = "silent", Description = nameof(MessageResource.CommandLine_SilentDescription), DescriptionConverterName = nameof(GetDescription))]
            public bool Silent { get; set; }

            [CommandLineOption('d', "define", ValueName = "name=value", Description = nameof(MessageResource.CommandLine_DefineDescription), DescriptionConverterName = nameof(GetDescription), ConverterName = nameof(ConvertDefine))]
            public Constant[] Defines { get; set; }

            public static Constant ConvertDefine(string value, string optionName)
            {
                string[] nameValuePair = value.Split(new[] { '=' }, 2);

                if (nameValuePair.Length != 2)
                {
                    // 定数値が name=value の形式になっていません ('{0}')。
                    throw new ArgumentException(string.Format(MessageResource.Message_Error_CommandLine_InvalidDefineFormat, value));
                }

                var name = nameValuePair[0];
                var constValue = nameValuePair[1];

                if (string.IsNullOrEmpty(name) || name != name.Trim())
                {
                    // 定数値の名前に空白文字は使えません('{0}')。
                    throw new ArgumentException(string.Format(MessageResource.Message_Error_CommandLine_InvalidDefineName, value));
                }

                return new Constant()
                {
                    Name = name,
                    Value = constValue,
                };
            }

            public static string GetDescription(string description, string valueName)
            {
                string result = MessageResource.ResourceManager.GetString(description, Resources.MessageResource.Culture);
                Debug.Assert(result != null, "Command line option Description is null.");
                return result;
            }
        }
    }
}
