﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.ToolFoundation;
using Nintendo.ToolFoundation.Audio;
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy.Extensions;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Threading;
using System.Threading.Tasks;
using static NintendoWare.Spy.WaveformSpyModel;

namespace NintendoWare.Spy.Tests
{
    [TestClass]
    public class LoudnessTests
    {
        private enum ChannelOrder
        {
            ITU,
            EBU,
        }

        private class Reference
        {
            public ChannelOrder ChannelOrder { get; }

            public string FilePath { get; }

            public float Expected { get; }

            public int SampleRate { get; }

            public Reference(ChannelOrder order, string path, float result, int sampleRate)
            {
                this.ChannelOrder = order;
                this.FilePath = path;
                this.Expected = result;
                this.SampleRate = sampleRate;
            }

            public static Reference ITU(string path, float result)
            {
                return new Reference(ChannelOrder.ITU, Path.Combine("Resources/ITU-R_BS.2217-2", path), result, 48000);
            }
        }

        // TODO: SIGLO-52536 WaveformSpyModel.LoudnessAccumulator を使ってテストできるようにする
        /// <summary>
        /// <see cref="LoudnessCalculator"/> を呼び出すためのアダプターです。
        /// </summary>
        private class LoudnessAccumulator
        {
            private class WaveformRegion
            {
                public short[][] Samples { get; }
                public int From { get; }
                public int To { get; }

                public WaveformRegion(short[][] samples, int from, int to)
                {
                    this.Samples = samples;
                    this.From = from;
                    this.To = to;
                }
            }

            private readonly LoudnessCalculator _loudnessCalculator = new LoudnessCalculator();
            private readonly Loudness.LoudnessContext[] _loudnessContext = new Loudness.LoudnessContext[Loudness.ChannelCount].Populate(_ => new Loudness.LoudnessContext());
            private readonly double[] _channelValue = new double[Loudness.ChannelCount];
            private readonly float[] _sampleSquareSum = new float[Loudness.ChannelCount];
            private readonly float[] _samplePeak = new float[Loudness.ChannelCount];
            private readonly float[] _sampleTruePeak = new float[Loudness.ChannelCount];
            private readonly List<WaveformRegion> _waveformRegions = new List<WaveformRegion>();
            private long _sampleIndex;
            private int _accumulatedSampleCount;
            private Loudness.Parameter _parameter;

            public LoudnessAccumulator()
            {
                Ensure.Operation.True(Loudness.ChannelCount <= ChannelCountMax);
            }

            public void SetParameter(Loudness.Parameter parameter)
            {
                _parameter = parameter;
                _loudnessCalculator.SetParameter(parameter);
            }

            public IEnumerable<LoudnessInfo> Accumulate(short[][] samples, int offset, int count)
            {
                int sampleIndex = 0;
                while (sampleIndex < count)
                {
                    // ラウドネスの計算に足りないサンプル数だけ累積
                    int sampleCount = Math.Min(count - sampleIndex, _parameter.StepSampleCount - _accumulatedSampleCount);
                    Assertion.Operation.True(sampleCount > 0);

                    _waveformRegions.Add(new WaveformRegion(samples, sampleIndex, sampleIndex + sampleCount));
                    sampleIndex += sampleCount;
                    _accumulatedSampleCount += sampleCount;

                    // ラウドネスの計算にサンプル数が足りないときはここで終了
                    if (_accumulatedSampleCount < _parameter.StepSampleCount)
                    {
                        break;
                    }

                    // ラウドネスを計算
                    {
                        this.AccumulateSample();

                        double totalMeanSquare = 0;
                        float[] sampleSquareSumArray = new float[Loudness.ChannelCount];
                        float[] samplePeakArray = new float[Loudness.ChannelCount];
                        float[] sampleTruePeakArray = new float[Loudness.ChannelCount];
                        for (int channelIndex = 0; channelIndex < Loudness.ChannelCount; channelIndex++)
                        {
                            totalMeanSquare += _channelValue[channelIndex] * Loudness.ChannelWeightArray[channelIndex];
                            sampleSquareSumArray[channelIndex] = _sampleSquareSum[channelIndex];
                            samplePeakArray[channelIndex] = _samplePeak[channelIndex];
                            sampleTruePeakArray[channelIndex] = _sampleTruePeak[channelIndex];
                        }

                        _loudnessCalculator.AddStep((float)totalMeanSquare, sampleSquareSumArray, samplePeakArray, sampleTruePeakArray);
                    }

                    _sampleIndex += _parameter.StepSampleCount;

                    var loudnessInfo = ReadLoudnessInfo();

                    // 計算の完了したサンプルのインデックスを記録します。
                    loudnessInfo.SampleIndex = _sampleIndex;

                    ResetValue();

                    yield return loudnessInfo;
                }
            }

            private void ResetValue()
            {
                for (int i = 0; i < ChannelCountMax; ++i)
                {
                    _channelValue[i] = 0;
                    _sampleSquareSum[i] = 0;
                    _samplePeak[i] = 0;
                    _sampleTruePeak[i] = 0;
                }

                _accumulatedSampleCount = 0;
            }

            private void AccumulateSample()
            {
                Parallel.For(0, ChannelCountMax, channelIndex =>
                {
                    var samplePeak = float.MinValue;
                    var sampleTruePeak = float.MinValue;
                    var sampleSquareSum = 0.0f;
                    var channelValue = 0.0;
                    var loudnessContext = _loudnessContext[channelIndex];

                    foreach (var region in _waveformRegions)
                    {
                        short[] sample = region.Samples[channelIndex];

                        for (int i = region.From; i < region.To; i++)
                        {
                            float sampleValue = sample[i] / 32768.0f;

                            samplePeak = Math.Max(samplePeak, Math.Abs(sampleValue));
                            sampleTruePeak = Math.Max(sampleTruePeak, Loudness.CalcTruePeak(loudnessContext, sampleValue));
                            sampleSquareSum += sampleValue * sampleValue;
                            float kOut = Loudness.CalcKWeightingFilter(loudnessContext, sampleValue, _parameter);
                            channelValue += kOut * kOut;
                        }
                    }

                    _samplePeak[channelIndex] = Math.Max(_samplePeak[channelIndex], samplePeak);
                    _sampleTruePeak[channelIndex] = Math.Max(_sampleTruePeak[channelIndex], sampleTruePeak);
                    _sampleSquareSum[channelIndex] += sampleSquareSum;
                    _channelValue[channelIndex] += channelValue;
                });

                _waveformRegions.Clear();
            }

            private LoudnessInfo ReadLoudnessInfo()
            {
                var loudness = new LoudnessInfo();

                loudness.MomentaryLoudnessValue = _loudnessCalculator.MomentaryLoudnessValue;
                loudness.ShortTermLoudnessValue = _loudnessCalculator.ShortTermLoudnessValue;
                loudness.AbsGatedLoudnessValue = _loudnessCalculator.AbsGatedLoudnessValue;
                loudness.RelGatedLoudnessValue = _loudnessCalculator.RelGatedLoudnessValue;

                for (int channelIndex = 0; channelIndex < Loudness.ChannelCount; channelIndex++)
                {
                    var channelInfo = loudness.Channels[channelIndex];
                    channelInfo.PeakValue = _loudnessCalculator.PeakValue[channelIndex];
                    channelInfo.TruePeakValue = _loudnessCalculator.TruePeakValue[channelIndex];
                    channelInfo.RmsValue = _loudnessCalculator.RmsValue[channelIndex];
                }

                return loudness;
            }
        }

        [TestMethod]
        public void TestTruePeak()
        {
            var truePeakCalculator = new Loudness.TruePeakCalculator();

            float[] input =
            {
                1.0f,
                1.0f,
                -1.0f,
                -1.0f,
                1.0f,
                1.0f,
                -1.0f,
                -1.0f,
                1.0f,
                1.0f,
                -1.0f,
                -1.0f,
            };

            float truePeak = 0;

            for (int i = 0; i < input.Length; ++i)
            {
                truePeak = Math.Max(truePeak, truePeakCalculator.Calculate(input[i]));
            }

            Assert.IsTrue(truePeak > 1.4f);
        }

        [TestMethod]
        public void TestLoudness()
        {
            var list = new Reference[]
            {
                // ITU-R BS.2217 ファレンス波形の 0:00:05 から5秒間を切り出したもの。
                // (テスト時間短縮のため、計算結果が変化していないかだけ確認します)
                new Reference(ChannelOrder.ITU, "Resources/raw/1770-2_Conf_6ch_VinCntr-23LKFS_5sec.wav", -28.09190f, 48000),
                new Reference(ChannelOrder.ITU, "Resources/raw/1770-2_Conf_6ch_VinCntr-23LKFS_5sec_32kHz.wav", -28.09190f, 32000),

                // ITU-R BS.2217 Compliance material for Recommendation ITU-R BS.1770
                // http://www.itu.int/pub/R-REP-BS.2217
                // (波形データを Resources/ITU-R_BS.2217-2 に追加すればテストされます)
                Reference.ITU("1770-2_Comp_RelGateTest.wav", -10),
                Reference.ITU("1770-2_Comp_AbsGateTest.wav", -69.5f),
                Reference.ITU("1770-2_Comp_24LKFS_25Hz_2ch.wav", -24),
                Reference.ITU("1770-2_Comp_24LKFS_100Hz_2ch.wav", -24),
                Reference.ITU("1770-2_Comp_24LKFS_500Hz_2ch.wav", -24),
                Reference.ITU("1770-2_Comp_24LKFS_1000Hz_2ch.wav", -24),
                Reference.ITU("1770-2_Comp_24LKFS_2000Hz_2ch.wav", -24),
                Reference.ITU("1770-2_Comp_24LKFS_10000Hz_2ch.wav", -24),
                Reference.ITU("1770-2_Comp_23LKFS_25Hz_2ch.wav", -23),
                Reference.ITU("1770-2_Comp_23LKFS_100Hz_2ch.wav", -23),
                Reference.ITU("1770-2_Comp_23LKFS_500Hz_2ch.wav", -23),
                Reference.ITU("1770-2_Comp_23LKFS_1000Hz_2ch.wav", -23),
                Reference.ITU("1770-2_Comp_23LKFS_2000Hz_2ch.wav", -23),
                Reference.ITU("1770-2_Comp_23LKFS_10000Hz_2ch.wav", -23),
                Reference.ITU("1770-2_Comp_18LKFS_FrequencySweep.wav", -18),
                Reference.ITU("1770-2_Comp_24LKFS_SummingTest.wav", -24),
                Reference.ITU("1770-2_Comp_23LKFS_SummingTest.wav", -23),
                Reference.ITU("1770-2_Comp_24LKFS_ChannelCheckLeft.wav", -24),
                Reference.ITU("1770-2_Comp_24LKFS_ChannelCheckRight.wav", -24),
                Reference.ITU("1770-2_Comp_24LKFS_ChannelCheckCentre.wav", -24),
                Reference.ITU("1770-2_Comp_24LKFS_ChannelCheckLFE.wav", float.NegativeInfinity),
                Reference.ITU("1770-2_Comp_24LKFS_ChannelCheckLs.wav", -24),
                Reference.ITU("1770-2_Comp_24LKFS_ChannelCheckRs.wav", -24),
                Reference.ITU("1770-2_Comp_23LKFS_ChannelCheckLeft.wav", -23),
                Reference.ITU("1770-2_Comp_23LKFS_ChannelCheckRight.wav", -23),
                Reference.ITU("1770-2_Comp_23LKFS_ChannelCheckCentre.wav", -23),
                Reference.ITU("1770-2_Comp_23LKFS_ChannelCheckLFE.wav", float.NegativeInfinity),
                Reference.ITU("1770-2_Comp_23LKFS_ChannelCheckLs.wav", -23),
                Reference.ITU("1770-2_Comp_23LKFS_ChannelCheckRs.wav", -23),
                Reference.ITU("1770-2_Conf_6ch_VinCntr-24LKFS.wav", -24),
                Reference.ITU("1770-2_Conf_6ch_VinL+R-24LKFS.wav", -24),
                Reference.ITU("1770-2_Conf_6ch_VinL-R-C-24LKFS.wav", -24),
                Reference.ITU("1770-2_Conf_Stereo_VinL+R-24LKFS.wav", -24),
                Reference.ITU("1770-2_Conf_Mono_Voice+Music-24LKFS.wav", -24),
                Reference.ITU("1770-2_Conf_6ch_VinCntr-23LKFS.wav", -23),
                Reference.ITU("1770-2_Conf_6ch_VinL+R-23LKFS.wav", -23),
                Reference.ITU("1770-2_Conf_6ch_VinL-R-C-23LKFS.wav", -23),
                Reference.ITU("1770-2_Conf_Stereo_VinL+R-23LKFS.wav", -23),
                Reference.ITU("1770-2_Conf_Mono_Voice+Music-23LKFS.wav", -23),

                // EBU Loudness Test Set
                // https://tech.ebu.ch/publications/ebu_loudness_test_set
            };

            int successCount = 0;
            Parallel.ForEach(list, it =>
            {
                if (this.TestLoudness(it))
                {
                    Interlocked.Increment(ref successCount);
                }
            });

            Assert.IsTrue(successCount > 0);
        }

        private bool TestLoudness(Reference reference)
        {
            const int SampleCount = 100;

            // リソースに登録してある波形ファイルのみテストします。
            if (!File.Exists(reference.FilePath))
            {
                return false;
            }

            var waveFile = WaveFileReader.ReadWaveFile(reference.FilePath);

            List<Tuple<int, int>> mappings = new List<Tuple<int, int>>();

            switch (waveFile.Data.ChannelCount)
            {
                case 1:
                    mappings.Add(Tuple.Create(0, 0)); // FL
                    break;

                case 2:
                    mappings.Add(Tuple.Create(0, 0)); // FL
                    mappings.Add(Tuple.Create(1, 1)); // FR
                    break;

                case 5:
                    mappings.Add(Tuple.Create(0, 0)); // FL
                    mappings.Add(Tuple.Create(1, 1)); // FR
                    mappings.Add(Tuple.Create(2, 4)); // FC
                    mappings.Add(Tuple.Create(3, 2)); // RL
                    mappings.Add(Tuple.Create(4, 3)); // RR
                    break;

                case 6:
                    switch (reference.ChannelOrder)
                    {
                        case ChannelOrder.ITU:
                            mappings.Add(Tuple.Create(0, 0)); // FL
                            mappings.Add(Tuple.Create(1, 1)); // FR
                            mappings.Add(Tuple.Create(2, 4)); // FC
                            mappings.Add(Tuple.Create(3, 5)); // LFE
                            mappings.Add(Tuple.Create(4, 2)); // RL
                            mappings.Add(Tuple.Create(5, 3)); // RR
                            break;

                        case ChannelOrder.EBU:
                            mappings.Add(Tuple.Create(0, 0)); // FL
                            mappings.Add(Tuple.Create(1, 1)); // FR
                            mappings.Add(Tuple.Create(2, 4)); // FC
                            mappings.Add(Tuple.Create(3, 2)); // RL
                            mappings.Add(Tuple.Create(4, 3)); // RR
                            mappings.Add(Tuple.Create(5, 5)); // LFE
                            break;
                    }
                    break;
            }

            if (mappings.IsEmpty())
            {
                throw new InvalidOperationException("unexpected wav file.");
            }

            var accumulator = new LoudnessAccumulator();

            switch (reference.SampleRate)
            {
                case 48000:
                    accumulator.SetParameter(Loudness.Parameter48kHz);
                    break;

                case 32000:
                    accumulator.SetParameter(Loudness.Parameter32kHz);
                    break;

                default:
                    throw new InvalidOperationException("unexpected sample rate.");
            }

            LoudnessInfo loudnessInfo = null;
            for (long index = 0; index < waveFile.Data.SampleCount; index += SampleCount)
            {
                short[][] samples = new short[Loudness.ChannelCount][].Populate(_ => new short[SampleCount]);
                int sampleCount = (int)Math.Min(SampleCount, waveFile.Data.SampleCount - index);

                for (int i = 0; i < sampleCount; i++)
                {
                    foreach (var mapping in mappings)
                    {
                        samples[mapping.Item2][i] = (short)waveFile.Data.Samples[mapping.Item1][index + i];
                    }
                }

                loudnessInfo = accumulator.Accumulate(samples, 0, sampleCount).LastOrDefault() ?? loudnessInfo;
            }

            try
            {
                Assert.IsNotNull(loudnessInfo);
                Assert.IsTrue(loudnessInfo.RelGatedLoudnessValue == reference.Expected
                    || MathNtf.NearlyEquals(loudnessInfo.RelGatedLoudnessValue, reference.Expected, 0.1f));

                Console.WriteLine($"Success: actual {loudnessInfo.RelGatedLoudnessValue} expected {reference.Expected} {reference.FilePath}");
                return true;
            }
            catch
            {
                Console.WriteLine($"Failed: actual {loudnessInfo?.RelGatedLoudnessValue} expected {reference.Expected} {reference.FilePath}");
                throw;
            }
        }
    }
}
