﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nnt.h>
#include <nn/os.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/npns.h>
#include <nn/npns/npns_ApiSystem.h>
#include <nn/account.h>
#include <nn/account/account_ResultForAdministrators.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/nifm.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nn/time.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>
#include <nn/result/result_HandlingUtility.h>
#include "../../../Programs/Eris/Sources/Processes/npns/npns_Config.h"
#include <nnt/npnsUtil.h>

namespace nn { namespace npns {
extern nn::Result RequestChangeStateForceTimed(nn::npns::State targetState, nn::TimeSpan timeout) NN_NOEXCEPT;
extern nn::Result RequestChangeStateForceAsync(nn::npns::State targetState) NN_NOEXCEPT;
}}

namespace {

using namespace nn;
using namespace nnt::npns::util;

class NpnsPseudoHalfAwakeTest : public nnt::npns::util::TestBase
{
protected:
    NpnsPseudoHalfAwakeTest()
    {
    }
    static void SetUpTestCase()
    {
        // 以下の理由により必ず TestBase::SetUpTestCase() より先に
        // nn::account::InitializeForAdministrator() でアカウントを初期化する.
        //  - Windows 上の動作では、npns::Initialize で別権限の account 初期化が行われる
        //  - 権限の異なる account::Initialize を遅れて行っても利用可能 API が増えない
        nn::account::InitializeForAdministrator();

        TestBase::SetUpTestCase();

        nn::Result result;

        char jid[nn::npns::JidLength];
        result = nn::npns::GetJid(jid, sizeof(jid));
        if (result.IsSuccess())
        {
            nn::npns::DestroyJid();
        }
        result = nn::npns::CreateJid();
        NNT_ASSERT_RESULT_SUCCESS(result);
    }
};

class ScopedTimeLogger
{
public:
    NN_IMPLICIT ScopedTimeLogger(const char* log):
        m_Log(log)
    {
        NN_LOG("%s Start %s\n", NpnsPseudoHalfAwakeTest::GetDateTimeStr(), log);
        m_Start = nn::os::GetSystemTick();
    }

    ~ScopedTimeLogger()
    {
        auto elapsedMilliSecond = (nn::os::GetSystemTick() - m_Start).ToTimeSpan().GetMilliSeconds();
        NN_LOG("%s %s / %lld ms\n",  NpnsPseudoHalfAwakeTest::GetDateTimeStr(), m_Log, elapsedMilliSecond);
    }

private:
    nn::os::Tick m_Start;
    const char* m_Log;
};

nn::TimeSpan GetRandomTimeSpan(int64_t minMilliSeconds, int64_t maxMilliSeconds)
{
    NN_ASSERT_LESS(minMilliSeconds, maxMilliSeconds);

    int64_t random;
    nn::os::GenerateRandomBytes(&random, sizeof(random));

    if(random < 0)
    {
        random *= -1;
    }

    int64_t ret = (random % (maxMilliSeconds - minMilliSeconds)) + minMilliSeconds;
    return nn::TimeSpan::FromMilliSeconds(ret);
}

void SetTestModeEnabled(bool isEnabled)
{
    nn::settings::fwdbg::SetSettingsItemValue("npns", "test_mode", &isEnabled, sizeof(isEnabled));
}

} // namespace

// 半起床繰り返し
TEST_F(NpnsPseudoHalfAwakeTest, NormalHalfAwake)
{
    // sleep in 想定
    {
        ScopedTimeLogger logger("Suspend");
        NNT_EXPECT_RESULT_SUCCESS(nn::npns::Suspend());
        EXPECT_TRUE(PollState(nn::npns::State_Suspend, nn::TimeSpan::FromSeconds(10)));
    }

    for (int i = 0; i < 10; ++i)
    {
        NN_LOG("\nTry : %d\n", i);

        {
            // インフラ接続確立のState_Onlineまで遷移(ネットワーク環境によってばらつくので60秒)
            ScopedTimeLogger logger("State_Online(timeout:60sec)");
            NNT_EXPECT_RESULT_SUCCESS(nn::npns::RequestChangeStateForceAsync(nn::npns::State_Online));
            EXPECT_TRUE(PollState(nn::npns::State_Online, nn::TimeSpan::FromSeconds(60)));
        }

        // half awake 開始想定
        {
            if(i % 2 == 0)
            {
                ScopedTimeLogger logger("ConnectedOnHalfAwake(random timeout)");
                const auto timeout = GetRandomTimeSpan(100, 3000); // 適当なタイミングで抜けるよう散らす
                auto result = nn::npns::RequestChangeStateForceTimed(nn::npns::State_ConnectedOnHalfAwake, timeout);
                NN_LOG("Request State_ConnectedOnHalfAwake. result(0x%08x) timeout:%lld ms\n", result.GetInnerValueForDebug(), timeout.GetMilliSeconds());
            }
            else
            {
                ScopedTimeLogger logger("ConnectedOnHalfAwake(timeout:10sec)");
                const auto timeout = nn::TimeSpan::FromSeconds(10);
                NNT_EXPECT_RESULT_SUCCESS(nn::npns::RequestChangeStateForceTimed(nn::npns::State_ConnectedOnHalfAwake, timeout));
            }
        }

        // half awake 終了想定 (npns::Suspend は npns::Resume と回数合わせないと効かないので使わない)
        {
            ScopedTimeLogger logger("Suspend");
            NNT_EXPECT_RESULT_SUCCESS(nn::npns::RequestChangeStateForceAsync(nn::npns::State_Suspend));
            EXPECT_TRUE(PollState(nn::npns::State_Suspend, nn::TimeSpan::FromSeconds(10)));
        }
    }

    // full awake 想定
    {
        // インフラ接続確立のState_Onlineまで遷移(ネットワーク環境によってばらつくので60秒)
        ScopedTimeLogger logger("State_Online(timeout:60sec)");
        NNT_EXPECT_RESULT_SUCCESS(nn::npns::RequestChangeStateForceAsync(nn::npns::State_Online));
        EXPECT_TRUE(PollState(nn::npns::State_Online, nn::TimeSpan::FromSeconds(60)));
    }
    {
        ScopedTimeLogger logger("Connected");
        NNT_EXPECT_RESULT_SUCCESS(nn::npns::Resume());
        EXPECT_TRUE(PollState(nn::npns::State_Connected, nn::TimeSpan::FromSeconds(10)));
    }
}

// 半起床中に全起床
TEST_F(NpnsPseudoHalfAwakeTest, InterruptFullAwake)
{
    for (int i = 0; i < 10; ++i)
    {
        NN_LOG("\nTry : %d\n", i);

        // sleep in 想定
        {
            ScopedTimeLogger logger("Suspend");
            NNT_EXPECT_RESULT_SUCCESS(nn::npns::Suspend());
            EXPECT_TRUE(PollState(nn::npns::State_Suspend, nn::TimeSpan::FromSeconds(10)));
        }

        // half awake 開始想定
        {
            {
                // インフラ接続確立のState_Onlineまで遷移(ネットワーク環境によってばらつくので60秒)
                ScopedTimeLogger logger("State_Online(timeout:60sec)");
                NNT_EXPECT_RESULT_SUCCESS(nn::npns::RequestChangeStateForceAsync(nn::npns::State_Online));
                EXPECT_TRUE(PollState(nn::npns::State_Online, nn::TimeSpan::FromSeconds(60)));
            }

            if(i % 2 == 0)
            {
                ScopedTimeLogger logger("ConnectedOnHalfAwake(random timeout)");
                const auto timeout = GetRandomTimeSpan(100, 3000); // 適当なタイミングで抜けるよう散らす
                auto result = nn::npns::RequestChangeStateForceTimed(nn::npns::State_ConnectedOnHalfAwake, timeout);
                NN_LOG("Request State_ConnectedOnHalfAwake. result(0x%08x) timeout:%lld ms\n", result.GetInnerValueForDebug(), timeout.GetMilliSeconds());
            }
            else
            {
                ScopedTimeLogger logger("ConnectedOnHalfAwake(timeout:10sec)");
                const auto timeout = nn::TimeSpan::FromSeconds(10);
                NNT_EXPECT_RESULT_SUCCESS(nn::npns::RequestChangeStateForceTimed(nn::npns::State_ConnectedOnHalfAwake, timeout));
            }
        }

        // half awake 中に full awake 想定
        {
            // インフラ接続確立のState_Onlineまで遷移(ネットワーク環境によってばらつくので60秒)
            ScopedTimeLogger logger("State_Online(timeout:60sec)");
            NNT_EXPECT_RESULT_SUCCESS(nn::npns::RequestChangeStateForceAsync(nn::npns::State_Online));
            EXPECT_TRUE(PollState(nn::npns::State_Online, nn::TimeSpan::FromSeconds(60)));
        }
        {
            ScopedTimeLogger logger("Connected");
            NNT_EXPECT_RESULT_SUCCESS(nn::npns::Resume());
            EXPECT_TRUE(PollState(nn::npns::State_Connected, nn::TimeSpan::FromSeconds(10)));
        }
    }
}

extern "C" void nnMain()
{
    int result = 1;
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    {
#ifndef NN_BUILD_CONFIG_OS_WIN
        // テストモードの有効化
        SetTestModeEnabled(true);
        NN_UTIL_SCOPE_EXIT{ SetTestModeEnabled(false); };
#endif

        result = RUN_ALL_TESTS();
    }

    // Windows ではグローバル変数のデストラクタ関係でうまく終了できないので Exit しない
#ifndef NN_BUILD_CONFIG_OS_WIN
    nnt::Exit(result);
#endif
}
