﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nnt.h>
#include <nn/nn_Log.h>
#include <nn/npns.h>
#include <nn/npns/npns_ApiSystem.h>
#include <nn/account.h>
#include <nn/account/account_ResultForAdministrators.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>
#include <nn/result/result_HandlingUtility.h>
#include "../../../Programs/Eris/Sources/Processes/npns/npns_Config.h"
#include <nnt/npnsUtil.h>

#define FULL

namespace {

using namespace nn;
using namespace nnt::npns::util;

class NpnsBasicTest : public nnt::npns::util::TestBase
{
protected:
    NpnsBasicTest()
    {
    }
    static void SetUpTestCase()
    {
        // 以下の理由により必ず TestBase::SetUpTestCase() より先に
        // nn::account::InitializeForAdministrator() でアカウントを初期化する.
        //  - Windows 上の動作では、npns::Initialize で別権限の account 初期化が行われる
        //  - 権限の異なる account::Initialize を遅れて行っても利用可能 API が増えない
        nn::account::InitializeForAdministrator();

        TestBase::SetUpTestCase();

#ifdef FULL
        nn::Result result;

        char jid[nn::npns::JidLength];
        result = nn::npns::GetJid(jid, sizeof(jid));
        if (result.IsSuccess())
        {
            nn::npns::DestroyJid();
        }
        result = nn::npns::CreateJid();
        NNT_ASSERT_RESULT_SUCCESS(result);
#endif
    }
};

}

#ifdef FULL
TEST_F(NpnsBasicTest, SuspendResume)
{
    nn::Result result;

    for (int i = 0; i < 5; ++i)
    {
        result = nn::npns::Suspend();
        NNT_ASSERT_RESULT_SUCCESS(result);

        nn::npns::State state = nn::npns::GetState();
        GTEST_ASSERT_EQ(state, nn::npns::State_Suspend);

        result = nn::npns::Resume();
        NNT_ASSERT_RESULT_SUCCESS(result);

        auto start = nn::os::GetSystemTick();
        bool bResult = PollState(nn::npns::State_Connected, nn::TimeSpan::FromSeconds(10));
        auto elapsedMilliSeconds = (nn::os::GetSystemTick() - start).ToTimeSpan().GetMilliSeconds();
        GTEST_ASSERT_EQ(bResult, true) <<
            "try count:" << i <<
            ", current state:" <<  static_cast<int>(nn::npns::GetState()) <<
            ", elapsed:" << elapsedMilliSeconds << "[ms]";
        NN_UNUSED(bResult);
        NN_LOG("Transition to State_Connected elapsed:%lld ms, try count:%d\n", elapsedMilliSeconds, i);
    }
}
#endif

#ifdef FULL
TEST_F(NpnsBasicTest, NotificationToken)
{
    nn::Result result;
    nn::ApplicationId appId ={ 0x0100000000000000 };
    result = nn::npns::ListenTo(appId);
    NNT_ASSERT_RESULT_SUCCESS(result);

    nn::npns::NotificationToken token;
    nn::account::Uid uid ={ { 0x0123456789abcdefULL, 0xfedcba9876543210ULL } };
    result = nn::npns::CreateToken(&token, uid, appId);
    NNT_ASSERT_RESULT_SUCCESS(result);

    nn::os::SystemEvent event;
    result = nn::npns::GetReceiveEvent(event);
    NNT_ASSERT_RESULT_SUCCESS(result);

    PublishNotification(token, "notify by token");
#ifdef SEND_AUTO
    bool bSignaled = event.TimedWait(nn::TimeSpan::FromSeconds(10));
    GTEST_ASSERT_EQ(bSignaled, true); NN_UNUSED(bSignaled);
#else
    event.Wait();
#endif

    nn::npns::NotificationData nd;
    result = nn::npns::Receive(&nd);
    NNT_ASSERT_RESULT_SUCCESS(result);
    GTEST_ASSERT_EQ(strcmp(nd.GetPayload(), "notify by token"), 0);

    NN_LOG("appId = %016llx\n", nd.GetApplicationId());
    NN_LOG("payload = %.*s\n", nd.GetPayloadSize(), nd.GetPayload());

    result  = nn::npns::DestroyToken(uid, appId);
    NNT_ASSERT_RESULT_SUCCESS(result);

#ifdef SEND_AUTO
    PublishNotification(token, "notify by token");
    bSignaled = event.TimedWait(nn::TimeSpan::FromSeconds(33));
    GTEST_ASSERT_EQ(bSignaled, false); NN_UNUSED(bSignaled);
#endif
}
#endif

#ifdef NN_BUILD_CONFIG_OS_HORIZON
#ifdef FULL
TEST_F(NpnsBasicTest, NotificationTokenForUser)
{
    nn::Result result;
    result = nn::npns::ListenToMyApplicationId(); // appId 指定なし
    NNT_ASSERT_RESULT_SUCCESS(result);

    nn::npns::NotificationToken token;
    nn::account::Uid uid ={ { 0x0123456789abcdefULL, 0xfedcba9876543210ULL } };
    result = nn::npns::CreateToken(&token, uid); // appId 指定なし
    NNT_ASSERT_RESULT_SUCCESS(result);

    nn::os::SystemEvent event;
    result = nn::npns::GetReceiveEvent(event);
    NNT_ASSERT_RESULT_SUCCESS(result);

    PublishNotification(token, "[NotificationTokenForUser]notify by token");
#ifdef SEND_AUTO
    bool bSignaled = event.TimedWait(nn::TimeSpan::FromSeconds(10));
    GTEST_ASSERT_EQ(bSignaled, true); NN_UNUSED(bSignaled);
#else
    event.Wait();
#endif

    nn::npns::NotificationData nd;
    result = nn::npns::Receive(&nd);
    NNT_ASSERT_RESULT_SUCCESS(result);
    GTEST_ASSERT_EQ(strcmp(nd.GetPayload(), "[NotificationTokenForUser]notify by token"), 0);

    NN_LOG("appId = %016llx\n", nd.GetApplicationId());
    NN_LOG("payload = %.*s\n", nd.GetPayloadSize(), nd.GetPayload());

    result  = nn::npns::DestroyToken(uid); // appId 指定なし
    NNT_ASSERT_RESULT_SUCCESS(result);

#ifdef SEND_AUTO
    PublishNotification(token, "notify by token");
    bSignaled = event.TimedWait(nn::TimeSpan::FromSeconds(33));
    GTEST_ASSERT_EQ(bSignaled, false); NN_UNUSED(bSignaled);
#endif
}
#endif
#endif // NN_BUILD_CONFIG_OS_HORIZON

#ifdef FULL
TEST_F(NpnsBasicTest, TopicError)
{
    Result result;
    result = nn::npns::SubscribeTopic("nx_topic_not_exist");
    NNT_ASSERT_RESULT_FAILURE(nn::npns::ResultTopicNotFound, result);

    result = nn::npns::UnsubscribeTopic("nx_topic_not_exist");
    NNT_ASSERT_RESULT_FAILURE(nn::npns::ResultTopicNotFound, result);

    bool bResult;
    result = nn::npns::QueryIsTopicExist(&bResult, "nx_topic_not_exist");
    NNT_ASSERT_RESULT_SUCCESS(result);
    GTEST_ASSERT_EQ(bResult, false);


    char topic[64];
    const char* prefix = "data";
    nn::ApplicationId appId ={ 0x0100000000000129 };
#if 1
    CreateTopic(topic, sizeof(topic), appId, prefix);
#else
    nn::util::SNPrintf(topic, sizeof(topic), "nx_%s_%016llx", prefix, appId.value);
#endif
    result = nn::npns::SubscribeTopic(topic);
    NNT_ASSERT_RESULT_SUCCESS(result);

    result = nn::npns::SubscribeTopic(topic);
    NNT_ASSERT_RESULT_SUCCESS(result);

    result = nn::npns::QueryIsTopicExist(&bResult, topic);
    NNT_ASSERT_RESULT_SUCCESS(result);
    GTEST_ASSERT_EQ(bResult, true);

    result = nn::npns::UnsubscribeTopic(topic);
    NNT_ASSERT_RESULT_SUCCESS(result);

    result = nn::npns::UnsubscribeTopic(topic);
    NNT_ASSERT_RESULT_FAILURE(nn::npns::ResultTopicNotSubscribed, result);

    result = nn::npns::QueryIsTopicExist(&bResult, topic);
    NNT_ASSERT_RESULT_SUCCESS(result);
    GTEST_ASSERT_EQ(bResult, true);
}
#endif

#ifdef FULL
TEST_F(NpnsBasicTest, TopicReceive)
{
    nn::ApplicationId appId ={ 0x0100000000000127 };

    Result result;

    nn::os::SystemEvent event;
    result = nn::npns::GetReceiveEvent(event);
    NNT_ASSERT_RESULT_SUCCESS(result);

    char topic[64];
    const char* prefix = "data";
    CreateTopic(topic, sizeof(topic), appId, prefix);
    result = nn::npns::SubscribeTopic(topic);
    NNT_ASSERT_RESULT_SUCCESS(result);

    result = nn::npns::ListenTo(appId);
    NNT_ASSERT_RESULT_SUCCESS(result);

    PublishNotificationByTopic(topic, "notify by topic");
#ifdef SEND_AUTO
    bool bSignaled = event.TimedWait(nn::TimeSpan::FromSeconds(100));
    GTEST_ASSERT_EQ(bSignaled, true); NN_UNUSED(bSignaled);
#else
    event.Wait();
#endif

    nn::npns::NotificationData nd;
    result = nn::npns::Receive(&nd);
    NNT_ASSERT_RESULT_SUCCESS(result);

    NN_LOG("payload = %.*s\n", nd.GetPayloadSize(), nd.GetPayload());
    GTEST_ASSERT_EQ(strcmp(nd.GetPayload(), "notify by topic"), 0);

    result = nn::npns::UnsubscribeTopic(topic);
    NNT_ASSERT_RESULT_SUCCESS(result);

    // Unsubscribe 前に受信していたものが残っていれば以降のテストに影響しないようにすべて Receive() する
    do
    {
        result = nn::npns::Receive(&nd);
        if (result.IsSuccess())
        {
            NN_LOG("payload = %.*s\n", nd.GetPayloadSize(), nd.GetPayload());
        }
    } while (result.IsSuccess());

//    DestroyTopic(appId);
}
#endif

#ifdef FULL
TEST_F(NpnsBasicTest, DelayListen)
{
    nn::Result result;
    nn::ApplicationId appId ={ 0x0100000000000125 };

    nn::npns::NotificationToken token;
    nn::account::Uid uid ={ { 0x0123456789abcdefULL, 0xfedcba9876543210ULL } };
    result = nn::npns::CreateToken(&token, uid, appId);
    NNT_ASSERT_RESULT_SUCCESS(result);

    nn::os::SystemEvent event;
    result = nn::npns::GetReceiveEvent(event);
    NNT_ASSERT_RESULT_SUCCESS(result);

    nn::npns::NotificationData nd;
    result = nn::npns::Receive(&nd);
    NNT_ASSERT_RESULT_FAILURE(npns::ResultNotReceived, result);

    EXPECT_FALSE(event.TryWait());

    PublishNotification(token, "for listen-delay");

    // 通知が届いた後に ListenTo するパターンをテスト
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10)); // このSleepの間に通知が来ることを期待

    result = nn::npns::Receive(&nd);
    NNT_ASSERT_RESULT_FAILURE(npns::ResultNotReceived, result);

    // ここではまだ event は非シグナル
    EXPECT_FALSE(event.TryWait());

    result = nn::npns::ListenTo(appId);
    NNT_ASSERT_RESULT_SUCCESS(result);

    // ListenTo と同時に event はシグナルしているはず
#ifdef SEND_AUTO
    bool bSignaled = event.TryWait();
    GTEST_ASSERT_EQ(bSignaled, true); NN_UNUSED(bSignaled);
#else
    event.Wait();
#endif

    bool bFound = false;
    while(nn::npns::Receive(&nd).IsSuccess())
    {
        // 一度bFound==trueになればずっとtrueのままにしておく. 受信したpayloadは念のためすべてログしておく.
        if(!bFound)
        {
            bFound = strcmp(nd.GetPayload(), "for listen-delay") == 0;
        }
        NN_LOG("payload = %.*s\n", nd.GetPayloadSize(), nd.GetPayload());
    }
    EXPECT_TRUE(bFound) << "Not receive \"for listen-delay\"";
}
#endif

// ListenUndelivered を呼ぶので最後
#ifdef FULL
TEST_F(NpnsBasicTest, ListenUndelivered)
{
    nn::Result result;
    nn::ApplicationId appId ={ 0x0100000000000123 };

    nn::npns::NotificationToken token;
    nn::account::Uid uid ={ { 0x0123456789abcdefULL, 0xfedcba9876543210ULL } };
    result = nn::npns::CreateToken(&token, uid, appId);
    NNT_ASSERT_RESULT_SUCCESS(result);

    nn::os::SystemEvent event;
    result = nn::npns::GetReceiveEvent(event);
    NNT_ASSERT_RESULT_SUCCESS(result);

    nn::npns::NotificationData nd;
    result = nn::npns::Receive(&nd);
    NNT_ASSERT_RESULT_FAILURE(npns::ResultNotReceived, result);

    PublishNotification(token, "for listen-all");

    // 通知が届いた後に ListenUndelivered するパターンをテスト
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    result = nn::npns::Receive(&nd);
    NNT_ASSERT_RESULT_FAILURE(npns::ResultNotReceived, result);

    result = nn::npns::ListenUndelivered();
    NNT_ASSERT_RESULT_SUCCESS(result);

#ifdef SEND_AUTO
    bool bSignaled = event.TimedWait(nn::TimeSpan::FromSeconds(10));
    GTEST_ASSERT_EQ(bSignaled, true); NN_UNUSED(bSignaled);
#else
    event.Wait();
#endif

    bool bFound = false;
    while(nn::npns::Receive(&nd).IsSuccess())
    {
        // 一度bFound==trueになればずっとtrueのままにしておく. 受信したpayloadは念のためすべてログしておく.
        if(!bFound)
        {
            bFound = strcmp(nd.GetPayload(), "for listen-all") == 0;
        }
        NN_LOG("payload = %.*s\n", nd.GetPayloadSize(), nd.GetPayload());
    }
    EXPECT_TRUE(bFound) << "Not receive \"for listen-all\"";
}
#endif

#ifdef FULL
TEST_F(NpnsBasicTest, JidManagement)
{
    nn::Result result;
    char jid[nn::npns::JidLength] = {};
    char password[nn::npns::PasswordLength] = {};

    result = nn::npns::DetachJid(jid, sizeof(jid), password, sizeof(password));
    if (result.IsSuccess())
    {
        GTEST_ASSERT_NE(std::strcmp(jid, ""), 0);
        GTEST_ASSERT_NE(std::strcmp(password, ""), 0);
        GTEST_ASSERT_NE(std::strcmp(jid, password), 0);
    }

    result = nn::npns::GetJid(jid, sizeof(jid));
    NNT_ASSERT_RESULT_FAILURE(npns::ResultAccountNotAvailable, result);

    result = nn::npns::CreateJid();
    NNT_ASSERT_RESULT_SUCCESS(result);

    result = nn::npns::DestroyJid();
    NNT_ASSERT_RESULT_SUCCESS(result);

    result = nn::npns::CreateJid();
    NNT_ASSERT_RESULT_SUCCESS(result);

    char jid1[nn::npns::UsernameLength];
    result = nn::npns::GetJid(jid1, sizeof(jid1));
    NNT_ASSERT_RESULT_SUCCESS(result);

    GTEST_ASSERT_NE(std::strstr(jid1, "npns.srv.nintendo.net"), nullptr);

    result = nn::npns::CreateJid();
    NNT_ASSERT_RESULT_SUCCESS(result);

    char jid2[nn::npns::UsernameLength];
    result = nn::npns::GetJid(jid2, sizeof(jid2));
    NNT_ASSERT_RESULT_SUCCESS(result);

    GTEST_ASSERT_EQ(std::strcmp(jid1, jid2), 0);

    result = nn::npns::Suspend();
    NNT_ASSERT_RESULT_SUCCESS(result);

    nn::npns::State state = nn::npns::GetState();
    GTEST_ASSERT_EQ(state, nn::npns::State_Suspend);

    result = nn::npns::DestroyJid();
    NNT_ASSERT_RESULT_SUCCESS(result);

    result = nn::npns::GetJid(jid2, sizeof(jid2));
    NNT_ASSERT_RESULT_FAILURE(npns::ResultAccountNotAvailable, result);

    result = nn::npns::Resume();
    NNT_ASSERT_RESULT_SUCCESS(result);

    for (int i = 0; i < 10; ++i)
    {
        auto state = nn::npns::GetState();
        if (state >= nn::npns::State_ReadyToConnect
            && state <= nn::npns::State_Connected)
        {
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    result = nn::npns::GetJid(jid2, sizeof(jid2));
    NNT_ASSERT_RESULT_SUCCESS(result);

    GTEST_ASSERT_NE(std::strcmp(jid1, jid2), 0);
}
#endif

nn::Result SetDefaultProfile(nn::account::ProfileEditor& editor) NN_NOEXCEPT
{
    static const uint8_t DefaultUseData[nn::account::UserDataBytesMax] = {
#if defined(NN_BUILD_CONFIG_ENDIAN_LITTLE)
        0x01, 0x00, 0x00, 0x00, // version, mainImageType, padding(2),
        0x01, 0x00, 0x00, 0x00, // charId[LSB], charId>>8, charId>>16, charId[MSB]
        0x01, 0x00, 0x00, 0x01, // bgId[LSB], bgId>>8, bgId>>16, bgId[MSB]
#elif defined(NN_BUILD_CONFIG_ENDIAN_BIG)
        0x01, 0x00, 0x00, 0x00, // version, mainImageType, padding(2),
        0x00, 0x00, 0x00, 0x01, // charId[MSB], charId>>16, charId>>8, charId[LSB]
        0x01, 0x00, 0x00, 0x01, // bgId[MSB], bgId>>16, bgId>>8, bgId[LSB]
#else
#error "Specified endian type is not supported."
#endif
    };

    nn::account::Nickname name = {"User"};
    editor.SetNickname(name);
    editor.SetUserData(reinterpret_cast<const char*>(DefaultUseData), sizeof(DefaultUseData));

    NN_RESULT_SUCCESS;
}


nn::Result SetupAccount(nn::account::Uid* pUid)
{
    nn::account::ProfileEditor editor;
    NN_RESULT_DO(nn::account::BeginUserRegistration(pUid));

    NN_RESULT_DO(nn::account::GetProfileEditor(&editor, *pUid));
    SetDefaultProfile(editor);
    NN_RESULT_DO(nn::account::CompleteUserRegistration(*pUid));

    nn::account::NetworkServiceAccountAdministrator admin;
    NN_RESULT_DO(nn::account::GetNetworkServiceAccountAdministrator(&admin, *pUid));

    nn::account::AsyncContext task;
    NN_RESULT_DO(admin.RegisterAsync(&task));
    NN_UTIL_SCOPE_EXIT
    {
        task.Cancel();
        bool done;
        NN_ABORT_UNLESS_RESULT_SUCCESS(task.HasDone(&done));
        while (!done)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
            NN_ABORT_UNLESS_RESULT_SUCCESS(task.HasDone(&done));
        }
    };

    nn::os::SystemEvent e;
    NN_RESULT_DO(task.GetSystemEvent(&e));
    e.Wait();
    NN_RESULT_DO(task.GetResult());

    return ResultSuccess();
}

nn::Result UnregisterAccount(const nn::account::Uid& uid)
{
    nn::account::NetworkServiceAccountAdministrator admin;
    NN_RESULT_DO(nn::account::GetNetworkServiceAccountAdministrator(&admin, uid));

    nn::account::AsyncContext task;
    NN_RESULT_DO(admin.UnregisterAsync(&task));
    NN_UTIL_SCOPE_EXIT
    {
        task.Cancel();
        bool done;
        NN_ABORT_UNLESS_RESULT_SUCCESS(task.HasDone(&done));
        while (!done)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
            NN_ABORT_UNLESS_RESULT_SUCCESS(task.HasDone(&done));
        }
    };

    nn::os::SystemEvent e;
    NN_RESULT_DO(task.GetSystemEvent(&e));
    e.Wait();
    NN_RESULT_DO(task.GetResult());

    return ResultSuccess();
}

#ifdef FULL
TEST_F(NpnsBasicTest, PushChannel)
{
    Result result;

    nn::account::Uid uid;

    result = SetupAccount(&uid);
    if (result <= nn::account::ResultUserRegistryFull())
    {
        int ret;
        result = nn::account::ListAllUsers(&ret, &uid, 1);
        if (result.IsSuccess())
        {
            result = UnregisterAccount(uid);
            NNT_ASSERT_RESULT_SUCCESS(result);
            result = nn::account::DeleteUser(uid);
        }
        NNT_ASSERT_RESULT_SUCCESS(result);
        result = SetupAccount(&uid);
    }
    NNT_ASSERT_RESULT_SUCCESS(result);

    NN_UTIL_SCOPE_EXIT
    {
        UnregisterAccount(uid);
        nn::account::DeleteUser(uid);
    };

    char jid[nn::npns::JidLength];
    result = nn::npns::GetJid(jid, sizeof(jid));
    if (result <= nn::npns::ResultAccountNotAvailable())
    {
        result = nn::npns::CreateJid();
    }
    NNT_ASSERT_RESULT_SUCCESS(result);

    result = nn::npns::UploadTokenToBaaS(uid);
    NNT_ASSERT_RESULT_SUCCESS(result);

    result = nn::npns::DestroyTokenForBaaS(uid);
    NNT_ASSERT_RESULT_SUCCESS(result);

}
#endif

#ifdef FULL
TEST_F(NpnsBasicTest, ManyNotificationData)
{
    nn::Result result;
    nn::ApplicationId appId ={ 0x0100000000000126 };

    nn::npns::NotificationToken token;
    nn::account::Uid uid ={ { 0x0123456789abcdefULL, 0xfedcba9876543210ULL } };
    result = nn::npns::CreateToken(&token, uid, appId);
    NNT_ASSERT_RESULT_SUCCESS(result);

    nn::os::SystemEvent event;
    result = nn::npns::GetReceiveEvent(event);
    NNT_ASSERT_RESULT_SUCCESS(result);

    result = nn::npns::ListenTo(appId);
    NNT_ASSERT_RESULT_SUCCESS(result);

    // 内部のNotificationQueueのバッファは通知16個分. 溢れるように17個の通知を投げる.
    for(int i = 0 ; i < 17 ; i++)
    {
        char buffer[128];
        nn::util::SNPrintf(buffer, sizeof(buffer), "notification %02d", i);
        PublishNotification(token, buffer);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

    }

#ifdef SEND_AUTO
    bool bSignaled = event.TimedWait(nn::TimeSpan::FromSeconds(10));
    GTEST_ASSERT_EQ(bSignaled, true); NN_UNUSED(bSignaled);
#else
    event.Wait();
#endif

    // 一番目の通知が押し出されて消えているはず
    nn::npns::NotificationData nd;
    for(int i = 0 ; i < 16 ; i++)
    {
        char buffer[128];
        nn::util::SNPrintf(buffer, sizeof(buffer), "notification %02d", i + 1);

        NNT_ASSERT_RESULT_SUCCESS(nn::npns::Receive(&nd));
        EXPECT_STREQ(buffer, nd.GetPayload());
    }
}
#endif

#ifdef FULL
TEST_F(NpnsBasicTest, RefreshResources)
{
    Result result;

    // 既存アカウントすべて消す
    {
        nn::account::Uid uids[nn::account::UserCountMax];
        int ret;
        NNT_ASSERT_RESULT_SUCCESS(nn::account::ListAllUsers(&ret, uids, nn::account::UserCountMax));
        for(int i = 0 ; i < ret ; i++)
        {
            NNT_ASSERT_RESULT_SUCCESS(UnregisterAccount(uids[i]));
            NNT_ASSERT_RESULT_SUCCESS(nn::account::DeleteUser(uids[i]));
        }
    }

    // UserCountMax人のaccountを用意
    nn::account::Uid uids[nn::account::UserCountMax];
    {
        for(int i = 0 ; i < nn::account::UserCountMax ; i++)
        {
            NNT_ASSERT_RESULT_SUCCESS(SetupAccount(&uids[i]));
        }
    }

    NN_UTIL_SCOPE_EXIT
    {
        for(int i = 0 ; i < nn::account::UserCountMax ; i++)
        {
            UnregisterAccount(uids[i]);
            nn::account::DeleteUser(uids[i]);
        }
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::npns::RefreshResources(uids, nn::account::UserCountMax));

    // 複数回叩いても問題なし
    NNT_ASSERT_RESULT_SUCCESS(nn::npns::RefreshResources(uids, nn::account::UserCountMax));
}
#endif

extern "C" void nnMain()
{
    int result = 1;
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    result = RUN_ALL_TESTS();

    // Windows ではグローバル変数のデストラクタ関係でうまく終了できないので Exit しない
#ifndef NN_BUILD_CONFIG_OS_WIN
    nnt::Exit(result);
#endif
}
