﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/audioctrl.h>
#include <nn/audioctrl/audioctrl_AudioControllerDeviceControllerShimForProduction.h>
#include <nn/audioctrl/audioctrl_PlayReport.h>

TEST(TargetVolumeControl, Success)
{
    nn::audioctrl::AudioTarget targets[] = { nn::audioctrl::AudioTarget_Speaker, nn::audioctrl::AudioTarget_Headphone, nn::audioctrl::AudioTarget_UsbOutputDevice };
    for(auto audioTarget : targets)
    {
        {
            auto volume = nn::audioctrl::GetTargetVolumeMin();
            nn::audioctrl::SetTargetVolume(audioTarget, volume);
            EXPECT_TRUE(nn::audioctrl::GetTargetVolume(audioTarget) == volume);
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

        {
            auto volume = nn::audioctrl::GetTargetVolumeMax();
            nn::audioctrl::SetTargetVolume(audioTarget, volume);
            EXPECT_TRUE(nn::audioctrl::GetTargetVolume(audioTarget) == volume);
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

        {
            auto isMute = false;
            nn::audioctrl::SetTargetMute(audioTarget, isMute);
            EXPECT_TRUE(nn::audioctrl::IsTargetMute(audioTarget) == isMute);
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

        {
            auto isMute = true;
            nn::audioctrl::SetTargetMute(audioTarget, isMute);
            EXPECT_TRUE(nn::audioctrl::IsTargetMute(audioTarget) == isMute);
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
    }
}

TEST(GetTargetVolume, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audioctrl::GetTargetVolume(nn::audioctrl::AudioTarget_Count), "");
}

TEST(SetTargetVolume, Precondition)
{
    nn::audioctrl::AudioTarget targets[] = { nn::audioctrl::AudioTarget_Speaker, nn::audioctrl::AudioTarget_Headphone, nn::audioctrl::AudioTarget_UsbOutputDevice };
    for(auto audioTarget : targets)
    {
        EXPECT_DEATH_IF_SUPPORTED(nn::audioctrl::SetTargetVolume(audioTarget, nn::audioctrl::GetTargetVolumeMin() - 1), "");
        EXPECT_DEATH_IF_SUPPORTED(nn::audioctrl::SetTargetVolume(audioTarget, nn::audioctrl::GetTargetVolumeMax() + 1), "");
    }

    EXPECT_DEATH_IF_SUPPORTED(nn::audioctrl::SetTargetVolume(static_cast<nn::audioctrl::AudioTarget>(-1), nn::audioctrl::GetTargetVolumeMin()), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audioctrl::SetTargetVolume(nn::audioctrl::AudioTarget_Count, nn::audioctrl::GetTargetVolumeMin()), "");
}

TEST(IsTargetMute, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audioctrl::IsTargetMute(nn::audioctrl::AudioTarget_Count), "");
}

TEST(SetTargetMute, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audioctrl::SetTargetMute(static_cast<nn::audioctrl::AudioTarget>(-1), false), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audioctrl::SetTargetMute(nn::audioctrl::AudioTarget_Count, false), "");
}

TEST(IsTargetConnected, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audioctrl::IsTargetConnected(nn::audioctrl::AudioTarget_Count), "");
}

TEST(IsTargetConnected, Success)
{
    nn::audioctrl::AudioTarget targets[] = { nn::audioctrl::AudioTarget_Speaker, nn::audioctrl::AudioTarget_Headphone, nn::audioctrl::AudioTarget_UsbOutputDevice };
    for(auto audioTarget : targets)
    {
        const auto IsConnected = nn::audioctrl::IsTargetConnected(audioTarget);
        NN_LOG("audioTarget(%d) IsConnected(%d)\n", audioTarget, IsConnected);
    }
}

TEST(DefaultTarget, Success)
{
    nn::audioctrl::AudioTarget targets[] = { nn::audioctrl::AudioTarget_Speaker, nn::audioctrl::AudioTarget_Tv, };
    for(auto audioTarget : targets)
    {
        nn::audioctrl::SetDefaultTarget(audioTarget, nn::TimeSpan::FromMilliSeconds(10), nn::TimeSpan::FromMilliSeconds(10));
        EXPECT_EQ(nn::audioctrl::GetDefaultTarget(), audioTarget);
    }
}

TEST(OutputModeSetting, Success)
{
    nn::audioctrl::AudioTarget targets[] = { nn::audioctrl::AudioTarget_Speaker, nn::audioctrl::AudioTarget_Tv, nn::audioctrl::AudioTarget_UsbOutputDevice, };
    for (auto target : targets)
    {
        auto driverMode = nn::audioctrl::GetAudioOutputMode(target);

        nn::audioctrl::AudioOutputMode modes[] = { nn::audioctrl::AudioOutputMode_Pcm1ch, nn::audioctrl::AudioOutputMode_Pcm2ch, nn::audioctrl::AudioOutputMode_Pcm6ch, nn::audioctrl::AudioOutputMode_PcmAuto,} ;
        for (auto mode : modes)
        {
            nn::audioctrl::SetOutputModeSetting(target, mode);
            EXPECT_EQ(nn::audioctrl::GetOutputModeSetting(target), mode);
            EXPECT_EQ(nn::audioctrl::GetAudioOutputMode(target), driverMode);
        }
    }
}

TEST(OutputModeSetting, Precondition)
{
    // invalid target, valid output setting
    {
        nn::audioctrl::AudioTarget targets[] = { nn::audioctrl::AudioTarget_Invalid, nn::audioctrl::AudioTarget_Headphone, nn::audioctrl::AudioTarget_Count, };
        for (auto target : targets)
        {
            nn::audioctrl::AudioOutputMode modes[] = { nn::audioctrl::AudioOutputMode_Pcm1ch, nn::audioctrl::AudioOutputMode_Pcm2ch, nn::audioctrl::AudioOutputMode_Pcm6ch, nn::audioctrl::AudioOutputMode_PcmAuto, };
            for (auto mode : modes)
            {
                EXPECT_DEATH_IF_SUPPORTED(nn::audioctrl::SetOutputModeSetting(target, mode), "");
            }
        }
    }

    // valid target, invalid output setting
    {
        nn::audioctrl::AudioTarget targets[] = { nn::audioctrl::AudioTarget_Speaker, nn::audioctrl::AudioTarget_Tv, nn::audioctrl::AudioTarget_UsbOutputDevice, };
        for (auto target : targets)
        {
            nn::audioctrl::AudioOutputMode modes[] = { nn::audioctrl::AudioOutputMode_Invalid, static_cast<nn::audioctrl::AudioOutputMode>(static_cast<int>(nn::audioctrl::AudioOutputMode_PcmAuto) + 1)};
            for (auto mode : modes)
            {
                EXPECT_DEATH_IF_SUPPORTED(nn::audioctrl::SetOutputModeSetting(target, mode), "");
            }
        }
    }
}

TEST(OutputMode, Success)
{
    nn::audioctrl::AudioTarget targets[] = { nn::audioctrl::AudioTarget_Speaker, nn::audioctrl::AudioTarget_Tv, };
    for (auto target : targets)
    {
        auto settingsMode = nn::audioctrl::GetOutputModeSetting(target);

        nn::audioctrl::AudioOutputMode modes[] = { nn::audioctrl::AudioOutputMode_Pcm1ch, nn::audioctrl::AudioOutputMode_Pcm2ch, nn::audioctrl::AudioOutputMode_Pcm6ch, nn::audioctrl::AudioOutputMode_PcmAuto, };
        for (auto mode : modes)
        {
            nn::audioctrl::SetAudioOutputMode(target, mode);
            EXPECT_EQ(nn::audioctrl::GetAudioOutputMode(target), mode);
            EXPECT_EQ(nn::audioctrl::GetOutputModeSetting(target), settingsMode);
        }
    }
}

TEST(OutputMode, Precondition)
{
    // invalid target, valid output setting
    {
        nn::audioctrl::AudioTarget targets[] = { nn::audioctrl::AudioTarget_Invalid, nn::audioctrl::AudioTarget_Headphone, nn::audioctrl::AudioTarget_Count, };
        for (auto target : targets)
        {
            nn::audioctrl::AudioOutputMode modes[] = { nn::audioctrl::AudioOutputMode_Pcm1ch, nn::audioctrl::AudioOutputMode_Pcm2ch, nn::audioctrl::AudioOutputMode_Pcm6ch, nn::audioctrl::AudioOutputMode_PcmAuto, };
            for (auto mode : modes)
            {
                EXPECT_DEATH_IF_SUPPORTED(nn::audioctrl::SetAudioOutputMode(target, mode), "");
            }
        }
    }

    // valid target, invalid output setting
    {
        nn::audioctrl::AudioTarget targets[] = { nn::audioctrl::AudioTarget_Speaker, nn::audioctrl::AudioTarget_Tv, nn::audioctrl::AudioTarget_UsbOutputDevice };
        for (auto target : targets)
        {
            nn::audioctrl::AudioOutputMode modes[] = { nn::audioctrl::AudioOutputMode_Invalid, static_cast<nn::audioctrl::AudioOutputMode>(static_cast<int>(nn::audioctrl::AudioOutputMode_PcmAuto) + 1)};
            for (auto mode : modes)
            {
                EXPECT_DEATH_IF_SUPPORTED(nn::audioctrl::SetAudioOutputMode(target, mode), "");
            }
        }
    }
}

TEST(ForceMutePolicy, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audioctrl::SetForceMutePolicy(static_cast<nn::audioctrl::ForceMutePolicy>(-1)), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audioctrl::SetForceMutePolicy(static_cast<nn::audioctrl::ForceMutePolicy>(static_cast<int>(nn::audioctrl::ForceMutePolicy_SpeakerMuteOnHeadphoneUnplugged) + 1)), "");
}

TEST(ForceMutePolicy, Success)
{
    nn::audioctrl::ForceMutePolicy policies[] = { nn::audioctrl::ForceMutePolicy_Disable, nn::audioctrl::ForceMutePolicy_SpeakerMuteOnHeadphoneUnplugged };

    for(auto policy : policies)
    {
        nn::audioctrl::SetForceMutePolicy(policy);
        EXPECT_EQ(nn::audioctrl::GetForceMutePolicy(), policy);
    }
}

TEST(SetOutputTargetForProduction, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audioctrl::SetOutputTargetForProduction(static_cast<nn::audioctrl::AudioTarget>(-1)), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audioctrl::SetOutputTargetForProduction(static_cast<nn::audioctrl::AudioTarget>(nn::audioctrl::AudioTarget_Count)), "");
}

TEST(SetOutputTargetForProduction, Success)
{
    nn::audioctrl::AudioTarget targets[] = { nn::audioctrl::AudioTarget_Speaker, nn::audioctrl::AudioTarget_Headphone, nn::audioctrl::AudioTarget_Tv,nn::audioctrl::AudioTarget_Invalid };

    for (auto target : targets)
    {
        nn::audioctrl::SetOutputTargetForProduction(target);
    }
}

TEST(SetInputTargetForceEnabledForProduction, Success)
{
    nn::audioctrl::SetInputTargetForceEnabledForProduction(true);
    nn::audioctrl::SetInputTargetForceEnabledForProduction(false);
}

TEST(SetHeadphoneOutputLevelMode, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audioctrl::SetHeadphoneOutputLevelMode(static_cast<nn::audioctrl::HeadphoneOutputLevelMode>(-1)), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audioctrl::SetHeadphoneOutputLevelMode(static_cast<nn::audioctrl::HeadphoneOutputLevelMode>(static_cast<int>(nn::audioctrl::HeadphoneOutputLevelMode_HighPower) + 1)), "");
}

TEST(SetHeadphoneOutputLevelMode, Success)
{
    const auto speakerVolume = nn::audioctrl::GetTargetVolume(nn::audioctrl::AudioTarget_Speaker);
    const auto headphoneVolume = nn::audioctrl::GetTargetVolume(nn::audioctrl::AudioTarget_Headphone);

    nn::audioctrl::HeadphoneOutputLevelMode modes[] = { nn::audioctrl::HeadphoneOutputLevelMode_Normal, nn::audioctrl::HeadphoneOutputLevelMode_Normal };

    for(auto mode : modes)
    {
        nn::audioctrl::SetHeadphoneOutputLevelMode(mode);
        EXPECT_EQ(mode, nn::audioctrl::GetHeadphoneOutputLevelMode());
    }

    // 再起動以外では本体音量設定値が変化しないことを確認
    EXPECT_EQ(speakerVolume, nn::audioctrl::GetTargetVolume(nn::audioctrl::AudioTarget_Speaker));
    EXPECT_EQ(headphoneVolume, nn::audioctrl::GetTargetVolume(nn::audioctrl::AudioTarget_Headphone));
}

TEST(BindAudioVolumeUpdateEventForPlayReport, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audioctrl::BindAudioVolumeUpdateEventForPlayReport(nullptr), "");
}

// この API を am が使用している環境では確率的にテストが失敗する、
// また PC 環境は PlayReport 関連機能が未実装なのでテストを行わない
#if 0
TEST(BindAudioVolumeUpdateEventForPlayReport, Success)
{
    nn::os::SystemEvent event;

    // NXAddon 4.0.0 時点では 5 秒でタイマーが発動する (AudioControllerServiceImpl 参照) が、
    // 余裕を見て 10 秒間にしておく
    const auto timeout = nn::TimeSpan::FromSeconds(10);

    // バインド直後はシグナル状態になるはず
    nn::audioctrl::BindAudioVolumeUpdateEventForPlayReport(event.GetBase());
    EXPECT_TRUE(event.TryWait());

    const auto volumeA = nn::audioctrl::GetTargetVolumeMin();
    const auto volumeB = nn::audioctrl::GetTargetVolumeMax();
    // ボリューム変更が出来ない環境対策
    if(volumeA != volumeB)
    {
        // ボリューム変更を行うとシグナル状態になるはず
        nn::audioctrl::SetTargetVolume(nn::audioctrl::AudioTarget_Speaker, volumeA);
        nn::audioctrl::SetTargetVolume(nn::audioctrl::AudioTarget_Speaker, volumeB);
        EXPECT_TRUE(event.TimedWait(timeout));
    }

    // ミュート変更を行うとシグナル状態になるはず
    nn::audioctrl::SetTargetMute(nn::audioctrl::AudioTarget_Speaker, true);
    nn::audioctrl::SetTargetMute(nn::audioctrl::AudioTarget_Speaker, false);
    EXPECT_TRUE(event.TimedWait(timeout));
}
#endif

TEST(GetAudioVolumeDataForPlayReport, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audioctrl::GetAudioVolumeDataForPlayReport(nullptr), "");
}

TEST(GetAudioVolumeDataForPlayReport, Success)
{
    const auto SpeakerVolume = nn::audioctrl::GetTargetVolumeMin();
    const auto SpeakerMute = false;
    const auto HeadphoneVolume = nn::audioctrl::GetTargetVolumeMax();
    const auto HeadphoneMute = false;
    const auto HeadphoneOutputLevelMode = nn::audioctrl::HeadphoneOutputLevelMode_HighPower;
#if !defined(NN_BUILD_CONFIG_OS_WIN)
    const auto IsHeadphonePowerLimited = (HeadphoneOutputLevelMode == nn::audioctrl::HeadphoneOutputLevelMode_Normal);
#endif

    nn::audioctrl::SetTargetVolume(nn::audioctrl::AudioTarget_Speaker, SpeakerVolume);
    nn::audioctrl::SetTargetMute(nn::audioctrl::AudioTarget_Speaker, SpeakerMute);
    nn::audioctrl::SetTargetVolume(nn::audioctrl::AudioTarget_Headphone, HeadphoneVolume);
    nn::audioctrl::SetTargetMute(nn::audioctrl::AudioTarget_Headphone, HeadphoneMute);
    nn::audioctrl::SetHeadphoneOutputLevelMode(HeadphoneOutputLevelMode);

    // PlayReport のコミットタイマーが発動するまで待つ
    // 本来は AudioVolumeUpdateEvent で待機するべきだが、am が存在する環境では正しく動かないのでスリープで待機
    // NXAddon 4.0.0 時点では 5 秒でタイマーが発動する (AudioControllerServiceImpl 参照)
#if !defined(NN_BUILD_CONFIG_OS_WIN)
    // 余裕を見て 10 秒スリープ
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
#endif

    nn::audioctrl::PlayReportAudioVolumeData data;
    nn::audioctrl::GetAudioVolumeDataForPlayReport(&data);

    // Win 環境では無効値が返ってくるので値のチェックは行わない
#if !defined(NN_BUILD_CONFIG_OS_WIN)
    EXPECT_EQ(data.speaker.volume, SpeakerVolume);
    EXPECT_EQ(data.speaker.mute, SpeakerMute);
    EXPECT_EQ(data.headphone.volume, HeadphoneVolume);
    EXPECT_EQ(data.headphone.mute, HeadphoneMute);
    EXPECT_EQ(data.isHeadphonePowerLimited, IsHeadphonePowerLimited);
#endif

    // NN_LOG("Speaker(volume: %d mute: %d) Headphone(volume: %d mute: %d) IsHeadphonePowerLimited(%d)\n",
    //         data.speaker.volume, data.speaker.mute,
    //         data.headphone.volume, data.headphone.mute,
    //         data.isHeadphonePowerLimited);
}

TEST(BindAudioOutputDeviceUpdateEventForPlayReport, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audioctrl::BindAudioOutputDeviceUpdateEventForPlayReport(nullptr), "");
}

// この API を am が使用している環境では確率的にテストが失敗するので PC 環境でのみテストを行う
#if defined(NN_BUILD_CONFIG_OS_WIN)
TEST(BindAudioOutputDeviceUpdateEventForPlayReport, Success)
{
    nn::os::SystemEvent event;
    nn::audioctrl::BindAudioOutputDeviceUpdateEventForPlayReport(event.GetBase());
    EXPECT_TRUE(event.TryWait());
}
#endif

TEST(GetAudioOutputTargetForPlayReport, Success)
{
    const auto target = nn::audioctrl::GetAudioOutputTargetForPlayReport();
    EXPECT_TRUE(
            target == nn::audioctrl::PlayReportAudioOutputTarget_Invalid   ||
            target == nn::audioctrl::PlayReportAudioOutputTarget_Headphone ||
            target == nn::audioctrl::PlayReportAudioOutputTarget_Speaker   ||
            target == nn::audioctrl::PlayReportAudioOutputTarget_Tv        ||
            target == nn::audioctrl::PlayReportAudioOutputTarget_UsbOutputDevice);

    // NN_LOG("Invalid(%d) Speaker(%d) Headphone(%d) Tv(%d) UsbOutputDevice(%d)\n",
    //         target ==  nn::audioctrl::PlayReportAudioOutputTarget_Invalid,
    //         target ==  nn::audioctrl::PlayReportAudioOutputTarget_Speaker,
    //         target ==  nn::audioctrl::PlayReportAudioOutputTarget_Headphone,
    //         target ==  nn::audioctrl::PlayReportAudioOutputTarget_Tv
    //         target ==  nn::audioctrl::PlayReportAudioOutputTarget_UsbOutputDevice);
}

TEST(NotifyHeadphoneVolumeWarningDisplayedEvent, Success)
{
    nn::audioctrl::NotifyHeadphoneVolumeWarningDisplayedEvent();
}

TEST(SetSystemOutputMasterVolume, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audioctrl::SetSystemOutputMasterVolume(nn::audioctrl::SystemOutputMasterVolumeMin - 1.f), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audioctrl::SetSystemOutputMasterVolume(nn::audioctrl::SystemOutputMasterVolumeMax + 1.f), "");
}

TEST(SetSystemOutputMasterVolume, Success)
{
    // Note: 実際に音量が意図通り適用されているかはスリープのコメントアウトを外せば手動では確認可能

    nn::audioctrl::SetSystemOutputMasterVolume(nn::audioctrl::SystemOutputMasterVolumeMin);
    EXPECT_EQ(nn::audioctrl::SystemOutputMasterVolumeMin, nn::audioctrl::GetSystemOutputMasterVolume());

    // NN_LOG("SystemOutputMasterVolumeMin");
    // nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000 * 10));

    nn::audioctrl::SetSystemOutputMasterVolume(nn::audioctrl::SystemOutputMasterVolumeMax / 2);
    EXPECT_EQ(nn::audioctrl::SystemOutputMasterVolumeMax / 2, nn::audioctrl::GetSystemOutputMasterVolume());

    // NN_LOG("SystemOutputMasterVolumeMax / 2");
    // nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000 * 10));

    nn::audioctrl::SetSystemOutputMasterVolume(nn::audioctrl::SystemOutputMasterVolumeMax);
    EXPECT_EQ(nn::audioctrl::SystemOutputMasterVolumeMax, nn::audioctrl::GetSystemOutputMasterVolume());

    // NN_LOG("SystemOutputMasterVolumeMax");
    // nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000 * 10));
}

