﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nn_Assert.h>

#include <nn/audio.h>
#include <nnt/audioUtil/testAudio_Util.h>
#include <nnt/audioUtil/testAudio_Constants.h>
#include <nn/fs/fs_Result.h>
#include <nn/nn_Log.h>

#include <memory> // std::unique_ptr
#include <string>

namespace
{

const std::string g_AudioTestName2ch = "testAudio_WaveComparison_SquareWave2ch";
const std::string g_AudioTestName6ch = "testAudio_WaveComparison_SquareWave6ch";
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t g_TestData[1024 * 1024];

char g_WorkBuffer[1024 * 1024];
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WorkBuffer2[1024 * 1024 * 2];

char g_RecorderBuffer[16 * 1024 * 1024];

}

void PrepareSourceVoice( nnt::audio::util::MinimalRenderer* renderer, nn::audio::VoiceType &voice, nn::audio::WaveBuffer &waveBuffer, int channelCount, bool isPitchAndSrcSkipped);
std::size_t GenerateTestWave(void** data, int channel, int channelCount, int sampleRate, int frequency, int sampleCount);
void RunWaveComparisonTest(int channelCount, std::string audioTestFile, bool isManualExecutionModeEnabled, bool isVoicePitchAndSrcSkipped) NN_NOEXCEPT;

std::string GetTargetFileName(const std::string& baseFileName, bool isManualExecutionModeEnabled, bool isVoicePitchAndSrcSkipped) NN_NOEXCEPT
{
    const std::string ExecutionMode = isManualExecutionModeEnabled ? "_Manual" : "_Auto";
    const std::string VoicePitchAndSrc = isVoicePitchAndSrcSkipped ? "_VoicePitchSrcSkipped" : "";
    return nnt::audio::util::GetResultDataDirectory() + "/" + baseFileName + ExecutionMode + VoicePitchAndSrc + ".wav";
}

std::string GetGoldenFileName(const std::string& baseFileName, bool isManualExecutionModeEnabled, bool isVoicePitchAndSrcSkipped) NN_NOEXCEPT
{
    NN_UNUSED(isManualExecutionModeEnabled);
    const std::string VoicePitchAndSrc = isVoicePitchAndSrcSkipped ? "_VoicePitchSrcSkipped" : "";
    return g_RefernerceDataFolder + "/" + baseFileName + VoicePitchAndSrc + ".wav";
}

TEST(WaveComparisonTest_2ch, Success)
{
    for(const auto isManualExecutionModeEnabled : { false, true })
    {
        for(const auto isVoicePitchAndSrcSkipped : { false, true })
        {
            RunWaveComparisonTest(2, g_AudioTestName2ch, isManualExecutionModeEnabled, isVoicePitchAndSrcSkipped);
        }
    }
}

TEST(WaveComparisonTest_6ch, Success)
{
    for(const auto isManualExecutionModeEnabled : { false, true })
    {
        for(const auto isVoicePitchAndSrcSkipped : { false, true })
        {
            RunWaveComparisonTest(6, g_AudioTestName6ch, isManualExecutionModeEnabled, isVoicePitchAndSrcSkipped);
        }
    }
}

void RunWaveComparisonTest(int channelCount, std::string audioTestFile, bool isManualExecutionModeEnabled, bool isVoicePitchAndSrcSkipped) NN_NOEXCEPT
{
    std::string mountPath = "";
    nnt::audio::util::GetMountPath(mountPath);

    // Test environment
    auto params = nnt::audio::util::GetAudioRendererParameterForWaveComparison();

    if(isManualExecutionModeEnabled)
    {
        params.renderingDevice = nn::audio::AudioRendererRenderingDevice_Cpu;
        params.executionMode = nn::audio::AudioRendererExecutionMode_ManualExecution;
    }

    const std::string ExecutionMode = isManualExecutionModeEnabled ? "Manual" : "Auto";
    const std::string TargetFile = GetTargetFileName(audioTestFile, isManualExecutionModeEnabled, isVoicePitchAndSrcSkipped);
    const std::string GoldenFile = GetGoldenFileName(audioTestFile, isManualExecutionModeEnabled, isVoicePitchAndSrcSkipped);
    const nn::TimeSpan CheckDuration = nn::TimeSpan::FromSeconds(1);
    const int32_t RecoderAudioFrameCount = 100;
    const int32_t BitRate = 16;
    const size_t SendReturnBufferSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&params, RecoderAudioFrameCount, channelCount);
    const int ChannelCountMax = 6;

    NN_ASSERT_LESS_EQUAL(channelCount, ChannelCountMax);

    // Prepare Test Components
    /////////////////////////////////////////////////////////////////////////////
    nnt::audio::util::SimpleFsUtility fsUtil;
    nnt::audio::util::MinimalRenderer renderer(g_WorkBuffer, sizeof(g_WorkBuffer));
    nnt::audio::util::Recorder recorder;
    nnt::audio::util::WaveFile testWav;

    //fsUtil->InitializeFileSystem(MountName, MountPath);
    ASSERT_TRUE(fsUtil.InitializeFileSystem(g_MountName.c_str(), mountPath.c_str()).IsSuccess());
    renderer.Initialize(params);

    int64_t estimatedFileSize = nnt::audio::util::WaveFile::GetRequiredFileSize(params.sampleRate, channelCount, BitRate, CheckDuration);
    nn::Result rt = fsUtil.PrepareNewFile(TargetFile.c_str(), estimatedFileSize);
    ASSERT_TRUE(rt.IsSuccess()) << "Failed to create new file.";
    testWav.Open(TargetFile.c_str(), params.sampleRate, static_cast<uint16_t>(channelCount), BitRate);


    // Recorder Setup
    //////////////////////////////////////////////////////////////////////////
    nn::audio::MemoryPoolType pool, sourcePool;
    ASSERT_TRUE(nn::audio::AcquireMemoryPool(renderer.GetConfig(), &pool, g_WorkBuffer2, sizeof(g_WorkBuffer2)));
    ASSERT_TRUE(nn::audio::AcquireMemoryPool(renderer.GetConfig(), &sourcePool, g_TestData, sizeof(g_TestData)));
    RequestAttachMemoryPool(&pool);
    RequestAttachMemoryPool(&sourcePool);
    nn::mem::StandardAllocator allocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));

    // Setup AuxType
    nn::audio::AuxType aux;
    auto bufferSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&params, RecoderAudioFrameCount, channelCount);
    auto sendBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    auto returnBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    NNT_ASSERT_RESULT_SUCCESS(nn::audio::AddAux(renderer.GetConfig(), &aux, renderer.GetFinalMix(), sendBuffer, returnBuffer, bufferSize));

    int8_t inputIndex[ChannelCountMax];
    int8_t outputIndex[ChannelCountMax];

    for(int i = 0; i < channelCount; ++i)
    {
        inputIndex[i] = static_cast<int8_t>(i);
        outputIndex[i] = static_cast<int8_t>(i);
    }

    nn::audio::SetAuxInputOutput(&aux, inputIndex, outputIndex, channelCount);

    // Attach AuxType to Recorder & WaveFile.
    ASSERT_TRUE(recorder.Prepare(&params, &aux, &testWav, SendReturnBufferSize, CheckDuration, g_RecorderBuffer, sizeof(g_RecorderBuffer)).IsSuccess());

    // Prepare Target Voice.
    //////////////////////////////////////////////////////////////////////////
    nn::audio::VoiceType voice;
    nn::audio::WaveBuffer waveBuffer;
    PrepareSourceVoice(&renderer, voice, waveBuffer, channelCount, isVoicePitchAndSrcSkipped);

    // start running
    //////////////////////////////////////////////////////////////////////////
    recorder.Start();
    renderer.Start();

    // keep playing at least for recording period.
    while (recorder.Record())
    {
        if(isManualExecutionModeEnabled)
        {
            renderer.ExecuteRendering();

            // sleep here to give a time that Recorder writes data to a file.
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
        }

        renderer.Wait();
        renderer.Update();
    }

    // finish running
    //////////////////////////////////////////////////////////////////////////
    renderer.Stop();
    recorder.Stop();
    testWav.Close();

    allocator.Free(sendBuffer);
    allocator.Free(returnBuffer);

    nnt::audio::util::SimpleVerify(TargetFile, GoldenFile);

}

std::size_t GenerateTestWave(void* data, int channel, int channelCount, int sampleRate, int frequency, int sampleCount)
{
    // Generating square wave
    const int WaveLength = sampleRate / frequency;
    const int16_t Amplitude = std::numeric_limits<int16_t>::max();
    int count = 0;

    int16_t* p = static_cast<int16_t*>(data);
    for (int i = 0; i < sampleCount; ++i)
    {
        p[i * channelCount + channel] = static_cast<int16_t>(count < (WaveLength / 2) ? Amplitude : - Amplitude);
        if (++count == WaveLength)
        {
            count = 0;
        }
    }
    return sampleCount * sizeof(int16_t);
}

void PrepareSourceVoice( nnt::audio::util::MinimalRenderer* renderer, nn::audio::VoiceType &voice, nn::audio::WaveBuffer &waveBuffer, int channelCount, bool isPitchAndSrcSkipped)
{
    const int sineSampleRate = 48000;
    const int sineFrequency = 440;
    const int sineSampleCount = 48000;
    std::size_t size = 0;

    for(int i = 0; i < channelCount; ++i)
    {
        size += GenerateTestWave(g_TestData, i, channelCount, sineSampleRate, sineFrequency, sineSampleCount);
    }

    const nn::audio::VoiceType::BehaviorOptions behaviorOptions = { false, isPitchAndSrcSkipped };
    nn::audio::AcquireVoiceSlot(renderer->GetConfig(), &voice, sineSampleRate, channelCount, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0, &behaviorOptions);
    nn::audio::SetVoiceDestination(renderer->GetConfig(), &voice, renderer->GetFinalMix());

    waveBuffer.buffer = g_TestData;
    waveBuffer.size = size;
    waveBuffer.startSampleOffset = 0;
    waveBuffer.endSampleOffset = sineSampleCount;
    waveBuffer.loop = 0;
    waveBuffer.isEndOfStream = false;

    nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
    nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
    nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
    nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
    nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);

    for(int i = 0; i < channelCount; ++i)
    {
        nn::audio::SetVoiceMixVolume(&voice, renderer->GetFinalMix(), 0.707f / (i + 1), i, i);
    }

    nn::audio::BiquadFilterParameter filter = { true, { 34, 68, 34 }, { 30889, -14648 } };
    nn::audio::SetVoiceBiquadFilterParameter(&voice, 0, filter);
}


