﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testAudio_ScopedRenderer.h"

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/audio.h>
#include <nnt/audioUtil/testAudio_Util.h>
#include <nnt/audioUtil/testAudio_Constants.h>
#include <nn/fs/fs_Result.h>

#include <memory> // std::unique_ptr

#include "../../Programs/Eris/Sources/Libraries/audio/dsp/audio_EffectCommon.h" // QF_FRACTIONAL_BIT_COUNT
#include <cmath> //pow

namespace
{
    nn::mem::StandardAllocator g_Allocator;
    nn::mem::StandardAllocator g_WaveBufferAllocator;
    nn::mem::StandardAllocator g_EffectBufferAllocator;
    // File system will be initialized by earlier tests (alphabetically ordered)
    bool g_FsInitialized = true;
    int g_Iterations = 5;

    NN_ALIGNAS(4096) char g_WorkBuffer[16 * 1024 * 1024];
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WaveBufferPoolMemory[24 * 1024 * 1024];
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_EffectBufferPoolMemory[12 * 1024 * 1024];

    void* g_MountRomCacheBuffer = nullptr;
    size_t g_MountRomCacheBufferSize = 0;
}

void* Allocate(size_t size)
{
    return g_Allocator.Allocate(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    g_Allocator.Free(p);
}

void InitializeFileSystem()
{
    if( !g_FsInitialized )
    {
        nn::fs::SetAllocator(Allocate, Deallocate);
        g_FsInitialized = true;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&g_MountRomCacheBufferSize));
    g_MountRomCacheBuffer = Allocate(g_MountRomCacheBufferSize);
    NN_ABORT_UNLESS_NOT_NULL(g_MountRomCacheBuffer);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountRom("asset", g_MountRomCacheBuffer, g_MountRomCacheBufferSize));
}

void FinalizeFileSystem()
{
    nn::fs::Unmount("asset");
    Deallocate(g_MountRomCacheBuffer, g_MountRomCacheBufferSize);
    g_MountRomCacheBuffer = nullptr;
    g_MountRomCacheBufferSize = 0;
}

void PrintPerformanceMetrics(const char* funcName, int voiceAvg, int subMixAvg, int finalMixAvg, int sinkAvg, int totalAvg, int64_t rendererUpdateAvg, int totalDumpCount)
{
    NN_LOG("##teamcity[buildStatisticValue key='%sVoiceAvg(usec)' value='%d']\n", funcName, voiceAvg / totalDumpCount);
    NN_LOG("##teamcity[buildStatisticValue key='%sSubMixAvg(usec)' value='%d']\n", funcName, subMixAvg / totalDumpCount);
    NN_LOG("##teamcity[buildStatisticValue key='%sFinalMixAvg(usec)' value='%d']\n", funcName, finalMixAvg / totalDumpCount);
    NN_LOG("##teamcity[buildStatisticValue key='%sSinkAvg(usec)' value='%d']\n", funcName, sinkAvg / totalDumpCount);
    NN_LOG("##teamcity[buildStatisticValue key='%sTotalAvg(usec)' value='%d']\n", funcName, totalAvg / totalDumpCount);
    NN_LOG("##teamcity[buildStatisticValue key='%sRendererUpdateAvg(usec)' value='%lld']\n", funcName, rendererUpdateAvg);
}

void RunPerformanceTest(int delayCount, int reverbCount, int i3dl2ReverbCount, const char* funcName, bool biquad, bool auxI3dl2)
{
    nnt::audio::ScopedRenderer renderer(g_Allocator, g_WaveBufferAllocator, g_EffectBufferAllocator);
    int effectCountAux = 0;
    if( auxI3dl2 )
    {
        effectCountAux = 1;
    }
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = 48000;
    parameter.sampleCount = 48000 / 200;
    parameter.mixBufferCount = 6;
    parameter.voiceCount = 8;
    parameter.subMixCount = 2;
    parameter.sinkCount = 1;
    parameter.effectCount = 1 + delayCount + reverbCount + i3dl2ReverbCount + effectCountAux;
    parameter.performanceFrameCount = 5;

    void* workBuffer;
    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    workBuffer = g_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);

    renderer.RendererSetup(&parameter, 2, 4);

    nn::audio::MemoryPoolType effectBufferPool;
    ASSERT_TRUE(AcquireMemoryPool(renderer.GetConfig(), &effectBufferPool, g_EffectBufferPoolMemory, sizeof(g_EffectBufferPoolMemory)));
    ASSERT_TRUE(RequestAttachMemoryPool(&effectBufferPool));

    const int channelCount = 2;
    if( delayCount > 0 )
    {
        renderer.AddDelays(delayCount, channelCount);
    }
    if( reverbCount > 0 )
    {
        renderer.AddReverbs(reverbCount, channelCount);
    }
    if( i3dl2ReverbCount > 0 )
    {
        renderer.AddI3dl2Reverbs(i3dl2ReverbCount, channelCount);
    }
    if (auxI3dl2)
    {
        renderer.AddAuxEffects(channelCount);
    }

    nn::audio::BufferMixerType mixer;

    NNT_ASSERT_RESULT_SUCCESS(nn::audio::AddBufferMixer(renderer.GetConfig(), &mixer, renderer.GetFinalMix()));
    nn::audio::SetBufferMixerInputOutput(&mixer, renderer.GetSubBus(), renderer.GetMainBus(), channelCount);
    nn::audio::SetBufferMixerVolume(&mixer, 0, 1.0f);
    nn::audio::SetBufferMixerVolume(&mixer, 1, 1.0f);

    renderer.Start();

    nn::audio::MemoryPoolType waveBufferMemoryPool;
    ASSERT_TRUE(AcquireMemoryPool(renderer.GetConfig(), &waveBufferMemoryPool, g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory)));
    ASSERT_TRUE(RequestAttachMemoryPool(&waveBufferMemoryPool));

    InitializeFileSystem();
    renderer.InitializeVoice(biquad);

    NN_LOG("Collecting performance metrics.... \n");
    while( renderer.GetTotalDumpCount() < g_Iterations )
    {
        if( auxI3dl2 )
        {
            renderer.ProcessAuxEffects(channelCount);
        }

        renderer.Wait();
        if( nn::audio::GetReleasedWaveBuffer(renderer.GetVoice()) )
        {
            nn::audio::AppendWaveBuffer(renderer.GetVoice(), renderer.GetWaveBuffer());
        }
        renderer.Update();
        renderer.Wait();
        renderer.UpdatePerformanceBuffer();
    }

    PrintPerformanceMetrics(funcName, renderer.GetVoiceAvg(), renderer.GetSubMixAvg(), renderer.GetFinalMixAvg(), renderer.GetSinkAvg(), renderer.GetTotalAvg(),
        renderer.GetRendererUpdateAvg(), renderer.GetTotalDumpCount());

    if( auxI3dl2 )
    {
        NN_LOG("##teamcity[buildStatisticValue key='%sReadAuxI3dl2Reverb(usec)' value='%llu']\n", funcName, renderer.GetAuxReadAverage());
        NN_LOG("##teamcity[buildStatisticValue key='%sWriteAuxI3dl2Reverb(usec)' value='%llu']\n", funcName, renderer.GetAuxWriteAverage());
        NN_LOG("##teamcity[buildStatisticValue key='%sProcessAuxI3dl2Reverb(usec)' value='%llu']\n", funcName, renderer.GetAuxProcessingAverage());
        NN_LOG("##teamcity[buildStatisticValue key='%sAuxDspAvg(usec)' value='%d']\n", funcName, renderer.GetAuxDspAvg() / renderer.GetTotalDumpCount());
    }

    renderer.Wait();
    renderer.Update();
    renderer.Stop(delayCount, reverbCount, i3dl2ReverbCount);
    renderer.Update();
    g_Allocator.Free(workBuffer);
    renderer.RendererShutdown(delayCount, reverbCount, i3dl2ReverbCount, auxI3dl2);
    FinalizeFileSystem();
}

// ========================================================================================
// Test cases
// ========================================================================================

TEST(PerformanceMetrics, BaseLine)
{
    g_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));
    g_WaveBufferAllocator.Initialize(g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    g_EffectBufferAllocator.Initialize(g_EffectBufferPoolMemory, sizeof(g_EffectBufferPoolMemory));
    const int reverbCount = 0;
    const int delayCount = 0;
    const int i3dl2ReverbCount = 0;
    bool biquadEnabled = false;
    bool auxI3dl2Enabled = false;
    RunPerformanceTest(delayCount, reverbCount, i3dl2ReverbCount, "BaseLine", biquadEnabled, auxI3dl2Enabled);
}

TEST(PerformanceMetrics, Delay)
{
    const int reverbCount = 0;
    const int delayCount = 1;
    const int i3dl2ReverbCount = 0;
    bool biquadEnabled = false;
    bool auxI3dl2Enabled = false;
    RunPerformanceTest(delayCount, reverbCount, i3dl2ReverbCount, "SingleDelay", biquadEnabled, auxI3dl2Enabled);
}

TEST(PerformanceMetrics, Reverb)
{
    const int reverbCount = 1;
    const int delayCount = 0;
    const int i3dl2ReverbCount = 0;
    bool biquadEnabled = false;
    bool auxI3dl2Enabled = false;
    RunPerformanceTest(delayCount, reverbCount, i3dl2ReverbCount, "SingleReverb", biquadEnabled, auxI3dl2Enabled);
}

TEST(PerformanceMetrics, I3dl2Reverb)
{
    const int reverbCount = 0;
    const int delayCount = 0;
    const int i3dl2ReverbCount = 1;
    bool biquadEnabled = false;
    bool auxI3dl2Enabled = false;
    RunPerformanceTest(delayCount, reverbCount, i3dl2ReverbCount, "SingleI3dl2Reverb", biquadEnabled, auxI3dl2Enabled);
}

TEST(PerformanceMetrics, DelayReverb)
{
    const int reverbCount = 1;
    const int delayCount = 1;
    const int i3dl2ReverbCount = 0;
    bool biquadEnabled = false;
    bool auxI3dl2Enabled = false;
    RunPerformanceTest(delayCount, reverbCount, i3dl2ReverbCount, "SingleDelaySingleReverb", biquadEnabled, auxI3dl2Enabled);
}

TEST(PerformanceMetrics, ReverbI3dl2Reverb)
{
    const int reverbCount = 1;
    const int delayCount = 0;
    const int i3dl2ReverbCount = 1;
    bool biquadEnabled = false;
    bool auxI3dl2Enabled = false;
    RunPerformanceTest(delayCount, reverbCount, i3dl2ReverbCount, "SingleReverbSingleI3dl2Reverb", biquadEnabled, auxI3dl2Enabled);
}

TEST(PerformanceMetrics, I3dl2ReverbDelay)
{
    const int reverbCount = 0;
    const int delayCount = 1;
    const int i3dl2ReverbCount = 1;
    bool biquadEnabled = false;
    bool auxI3dl2Enabled = false;
    RunPerformanceTest(delayCount, reverbCount, i3dl2ReverbCount, "SingleI3dl2ReverbSingleDelay", biquadEnabled, auxI3dl2Enabled);
}

TEST(PerformanceMetrics, DoubleDelay)
{
    const int reverbCount = 0;
    const int delayCount = 2;
    const int i3dl2ReverbCount = 0;
    bool biquadEnabled = false;
    bool auxI3dl2Enabled = false;
    RunPerformanceTest(delayCount, reverbCount, i3dl2ReverbCount, "DoubleDelay", biquadEnabled, auxI3dl2Enabled);
}

TEST(PerformanceMetrics, DoubleReverb)
{
    const int reverbCount = 2;
    const int delayCount = 0;
    const int i3dl2ReverbCount = 0;
    bool biquadEnabled = false;
    bool auxI3dl2Enabled = false;
    RunPerformanceTest(delayCount, reverbCount, i3dl2ReverbCount, "DoubleReverb", biquadEnabled, auxI3dl2Enabled);
}

TEST(PerformanceMetrics, DoubleI3dl2Reverb)
{
    const int reverbCount = 0;
    const int delayCount = 0;
    const int i3dl2ReverbCount = 2;
    bool biquadEnabled = false;
    bool auxI3dl2Enabled = false;
    RunPerformanceTest(delayCount, reverbCount, i3dl2ReverbCount, "DoubleI3dl2Reverb", biquadEnabled, auxI3dl2Enabled);
}

TEST(PerformanceMetrics, BiquadBaseLine)
{
    const int reverbCount = 0;
    const int delayCount = 0;
    const int i3dl2ReverbCount = 0;
    bool biquadEnabled = true;
    bool auxI3dl2Enabled = false;
    RunPerformanceTest(delayCount, reverbCount, i3dl2ReverbCount, "BiquadBaseLine", biquadEnabled, auxI3dl2Enabled);
}

TEST(PerformanceMetrics, BiquadDelay)
{
    const int reverbCount = 0;
    const int delayCount = 1;
    const int i3dl2ReverbCount = 0;
    bool biquadEnabled = true;
    bool auxI3dl2Enabled = false;
    RunPerformanceTest(delayCount, reverbCount, i3dl2ReverbCount, "BiquadSingleDelay", biquadEnabled, auxI3dl2Enabled);
}

TEST(PerformanceMetrics, BiquadReverb)
{
    const int reverbCount = 1;
    const int delayCount = 0;
    const int i3dl2ReverbCount = 0;
    bool biquadEnabled = true;
    bool auxI3dl2Enabled = false;
    RunPerformanceTest(delayCount, reverbCount, i3dl2ReverbCount, "BiquadSingleReverb", biquadEnabled, auxI3dl2Enabled);
}

TEST(PerformanceMetrics, BiquadI3dl2Reverb)
{
    const int reverbCount = 0;
    const int delayCount = 0;
    const int i3dl2ReverbCount = 1;
    bool biquadEnabled = true;
    bool auxI3dl2Enabled = false;
    RunPerformanceTest(delayCount, reverbCount, i3dl2ReverbCount, "BiquadSingleI3dl2Reverb", biquadEnabled, auxI3dl2Enabled);
}

TEST(PerformanceMetrics, AuxI3dl2ReverbBaseLine)
{
    const int reverbCount = 0;
    const int delayCount = 0;
    const int i3dl2ReverbCount = 0;
    bool biquadEnabled = false;
    bool auxI3dl2Enabled = true;
    RunPerformanceTest(delayCount, reverbCount, i3dl2ReverbCount, "AuxI3dl2ReverbBaseLine", biquadEnabled, auxI3dl2Enabled);
}

TEST(PerformanceMetrics, AuxI3dl2ReverbDelay)
{
    const int reverbCount = 0;
    const int delayCount = 1;
    const int i3dl2ReverbCount = 0;
    bool biquadEnabled = false;
    bool auxI3dl2Enabled = true;
    RunPerformanceTest(delayCount, reverbCount, i3dl2ReverbCount, "AuxI3dl2ReverbSingleDelay", biquadEnabled, auxI3dl2Enabled);
}

TEST(PerformanceMetrics, AuxI3dl2ReverbReverb)
{
    const int reverbCount = 1;
    const int delayCount = 0;
    const int i3dl2ReverbCount = 0;
    bool biquadEnabled = false;
    bool auxI3dl2Enabled = true;
    RunPerformanceTest(delayCount, reverbCount, i3dl2ReverbCount, "AuxI3dl2ReverbSingleReverb", biquadEnabled, auxI3dl2Enabled);
}

TEST(PerformanceMetrics, AuxI3dl2ReverbI3dl2Reverb)
{
    const int reverbCount = 0;
    const int delayCount = 0;
    const int i3dl2ReverbCount = 1;
    bool biquadEnabled = false;
    bool auxI3dl2Enabled = true;
    RunPerformanceTest(delayCount, reverbCount, i3dl2ReverbCount, "AuxI3dl2ReverbSingleI3dl2Reverb", biquadEnabled, auxI3dl2Enabled);
}

TEST(PerformanceMetrics, BiquadAuxI3dl2Reverb)
{
    const int reverbCount = 0;
    const int delayCount = 0;
    const int i3dl2ReverbCount = 0;
    bool biquadEnabled = true;
    bool auxI3dl2Enabled = true;
    RunPerformanceTest(delayCount, reverbCount, i3dl2ReverbCount, "BiquadAuxI3dl2Reverb", biquadEnabled, auxI3dl2Enabled);
}

TEST(PerformanceMetrics, BiquadAuxI3dl2ReverbDelay)
{
    const int reverbCount = 0;
    const int delayCount = 1;
    const int i3dl2ReverbCount = 0;
    bool biquadEnabled = true;
    bool auxI3dl2Enabled = true;
    RunPerformanceTest(delayCount, reverbCount, i3dl2ReverbCount, "BiquadAuxI3dl2ReverbSingleDelay", biquadEnabled, auxI3dl2Enabled);
}

TEST(PerformanceMetrics, BiquadAuxI3dl2ReverbReverb)
{
    const int reverbCount = 1;
    const int delayCount = 0;
    const int i3dl2ReverbCount = 0;
    bool biquadEnabled = true;
    bool auxI3dl2Enabled = true;
    RunPerformanceTest(delayCount, reverbCount, i3dl2ReverbCount, "BiquadAuxI3dl2ReverbSingleReverb", biquadEnabled, auxI3dl2Enabled);
}

TEST(PerformanceMetrics, BiquadAuxI3dl2ReverbI3dl2Reverb)
{
    const int reverbCount = 0;
    const int delayCount = 0;
    const int i3dl2ReverbCount = 1;
    bool biquadEnabled = true;
    bool auxI3dl2Enabled = true;
    RunPerformanceTest(delayCount, reverbCount, i3dl2ReverbCount, "BiquadAuxI3dl2ReverbSingleI3dl2Reverb", biquadEnabled, auxI3dl2Enabled);
}
