﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testAudio_Effect.h"
#include <nn/nn_Assert.h>

namespace nnt {
namespace audio {

namespace
{
    char g_WorkBuffer[1024 * 1024 * 2];
    char g_WorkBuffer2[1024 * 1024 * 6];
}

EffectScopedRenderer::EffectScopedRenderer(nn::mem::StandardAllocator& allocator)
    : m_Allocator(allocator)
{
};

EffectScopedRenderer::~EffectScopedRenderer()
{
};

void EffectScopedRenderer::RendererSetup(const nn::audio::AudioRendererParameter* param, const int subMixBufferCount, const int finalMixBufferCount)
{
    // gather parameter.
    nn::audio::InitializeAudioRendererParameter(&m_Parameter);
    m_Parameter.sampleRate       = param->sampleRate;
    m_Parameter.sampleCount      = param->sampleCount;
    m_Parameter.mixBufferCount   = param->mixBufferCount;
    m_Parameter.subMixCount      = param->subMixCount;
    m_Parameter.voiceCount       = param->voiceCount;
    m_Parameter.sinkCount        = param->sinkCount;
    m_Parameter.effectCount      = param->effectCount;
    m_Parameter.performanceFrameCount = param->performanceFrameCount;
    m_WorkBufferSize = GetAudioRendererWorkBufferSize(m_Parameter);
    m_pWorkBuffer = m_Allocator.Allocate(m_WorkBufferSize, nn::os::MemoryPageSize);
    NN_ASSERT(m_pWorkBuffer != nullptr);
    // instantiate renderer.
    NN_ASSERT(OpenAudioRenderer(&m_Handle, &m_SystemEvent, m_Parameter, m_pWorkBuffer, m_WorkBufferSize).IsSuccess());

    size_t configBufferSize = GetAudioRendererConfigWorkBufferSize(m_Parameter);
    m_pConfigBuffer = m_Allocator.Allocate(configBufferSize);
    NN_ASSERT(m_pConfigBuffer != nullptr);
    InitializeAudioRendererConfig(&m_Config, m_Parameter, m_pConfigBuffer, configBufferSize);
    NN_ASSERT(nn::audio::AcquireFinalMix(&m_Config, &m_FinalMix, finalMixBufferCount));
    NN_ASSERT(nn::audio::AcquireSubMix(&m_Config, &m_SubMix, param->sampleRate, subMixBufferCount));
};

void EffectScopedRenderer::Start()
{
    NN_ASSERT(nn::audio::RequestUpdateAudioRenderer(m_Handle, &m_Config).IsSuccess());
    NN_ASSERT(nn::audio::StartAudioRenderer(m_Handle).IsSuccess());
}

void EffectScopedRenderer::Update()
{
    NN_ASSERT(nn::audio::RequestUpdateAudioRenderer(m_Handle, &m_Config).IsSuccess());
}

void EffectScopedRenderer::Wait()
{
    m_SystemEvent.Wait();
}

void EffectScopedRenderer::Stop()
{
    nn::audio::StopAudioRenderer(m_Handle);
}

void EffectScopedRenderer::RendererShutdown()
{
    if (nn::audio::AudioRendererState_Started == nn::audio::GetAudioRendererState(m_Handle))
    {
        nn::audio::StopAudioRenderer(m_Handle);
    }
    nn::audio::CloseAudioRenderer(m_Handle);
    m_Allocator.Free(m_pConfigBuffer);
    m_pConfigBuffer = nullptr;
    m_Allocator.Free(m_pWorkBuffer);
    m_pWorkBuffer = nullptr;
};

nn::audio::AudioRendererHandle EffectScopedRenderer::GetHandle()
{
    return m_Handle;
};

nn::audio::AudioRendererConfig* EffectScopedRenderer::GetConfig()
{
    return &m_Config;
};

nn::audio::AudioRendererParameter* EffectScopedRenderer::GetParam()
{
    return &m_Parameter;
};

nn::audio::FinalMixType* EffectScopedRenderer::GetFinalMix()
{
    return &m_FinalMix;
}

nn::audio::SubMixType* EffectScopedRenderer::GetSubMix()
{
    return &m_SubMix;
}

WaveComparisonTestBase::WaveComparisonTestBase(nn::audio::AudioRendererParameter parameter, uint16_t channelCount)
    : m_WaveFileChannelCount(channelCount) // 現状 WaveFile が 2ch 以外は未実装なため、 2 で固定
    , m_WaveFileBitRate(16)     // 現状 BitRate は 16 以外未検証
    , m_RecoderAudioFrameCount(100)  // 少なくすると、Aux のバッファ枯渇によりテストが失敗しうるので、大目に確保する。
    , m_RecodingDuration(0)
    , m_Renderer(g_WorkBuffer, sizeof(g_WorkBuffer))
    , m_Allocator(g_WorkBuffer2, sizeof(g_WorkBuffer2))
    , m_RecorderBuffer(nullptr)
    , m_RecorderBufferSize(0)
{
    m_Parameter = parameter;
}

nn::audio::SubMixType* WaveComparisonTestBase::GetSubMix()
{
    return m_Renderer.GetSubMix();
}

nn::audio::FinalMixType* WaveComparisonTestBase::GetFinalMix()
{
    return m_Renderer.GetFinalMix();
}

nn::audio::AudioRendererConfig* WaveComparisonTestBase::GetConfig()
{
    return m_Renderer.GetConfig();
}

void WaveComparisonTestBase::Start()
{
    m_Recorder.Start();
    m_Renderer.Start();
}
bool WaveComparisonTestBase::Record()
{
    return m_Recorder.Record();
}

void WaveComparisonTestBase::Update()
{
    m_Renderer.Update();
}

void WaveComparisonTestBase::Wait()
{
    m_Renderer.Wait();
}

void WaveComparisonTestBase::Finalize()
{
    m_Renderer.Stop();
    m_Recorder.Stop();
    m_RecordingWav.Close();

    m_Allocator.Free(m_InputWaveFileDataBuffer);
    m_Allocator.Free(m_SendBuffer);
    m_Allocator.Free(m_ReturnBuffer);
    m_Allocator.Free(m_RecorderBuffer);
}


void WaveComparisonTestBase::Initialize()
{
    // setup Fs
    std::string mountPath;
    nnt::audio::util::GetMountPath(mountPath);
    NN_ASSERT(m_FsUtil.InitializeFileSystem(g_MountName.c_str(), mountPath.c_str()).IsSuccess());

    // setup AudioRenderer
    m_Renderer.Initialize(m_Parameter);
}

void WaveComparisonTestBase::PrepareRecording(std::string recordingFileName, nn::TimeSpan recordingDuration, bool trimSilence)
{
    // prepare recording file.
    int64_t estimatedFileSize = nnt::audio::util::WaveFile::GetRequiredFileSize(m_Parameter.sampleRate, m_WaveFileChannelCount, m_WaveFileBitRate, recordingDuration);
    NN_ASSERT(m_FsUtil.PrepareNewFile(recordingFileName.c_str(), estimatedFileSize).IsSuccess());
    NN_ASSERT(m_RecordingWav.Open(recordingFileName.c_str(), m_Parameter.sampleRate, m_WaveFileChannelCount, m_WaveFileBitRate));

    // Setup Aux & Add it to renderer
    auto bufferSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&m_Parameter, m_RecoderAudioFrameCount, m_WaveFileChannelCount);
    m_SendBuffer = static_cast<int32_t*>(m_Allocator.Allocate(nn::util::align_up(bufferSize, nn::audio::MemoryPoolType::SizeGranularity),
                                                              nn::audio::MemoryPoolType::AddressAlignment));
    NN_ASSERT(m_SendBuffer != nullptr);
    m_ReturnBuffer = static_cast<int32_t*>(m_Allocator.Allocate(nn::util::align_up(bufferSize, nn::audio::MemoryPoolType::SizeGranularity),
                                                                nn::audio::MemoryPoolType::AddressAlignment));
    NN_ASSERT(m_ReturnBuffer != nullptr);

    nn::audio::AcquireMemoryPool(m_Renderer.GetConfig(), &m_RecorderPoolSend, m_SendBuffer, nn::util::align_up(bufferSize, nn::audio::MemoryPoolType::SizeGranularity));
    nn::audio::AcquireMemoryPool(m_Renderer.GetConfig(), &m_RecorderPoolReturn, m_ReturnBuffer, nn::util::align_up(bufferSize, nn::audio::MemoryPoolType::SizeGranularity));
    nn::audio::RequestAttachMemoryPool(&m_RecorderPoolSend);
    nn::audio::RequestAttachMemoryPool(&m_RecorderPoolReturn);

    NNT_ASSERT_RESULT_SUCCESS(nn::audio::AddAux(m_Renderer.GetConfig(), &m_RecoderAux, m_Renderer.GetFinalMix(), m_SendBuffer,  m_ReturnBuffer, bufferSize));
    int8_t inputIndex[] = {0, 1, 2, 3, 4, 5};
    int8_t outputIndex[] = {0, 1, 2, 3, 4, 5};
    nn::audio::SetAuxInputOutput(&m_RecoderAux, inputIndex, outputIndex, m_WaveFileChannelCount);

    m_RecorderBufferSize = 1 * 1024 * 1024;
    m_RecorderBuffer = m_Allocator.Allocate(m_RecorderBufferSize);
    NN_ASSERT(m_RecorderBuffer);

    // Attach AuxType to Recorder & WaveFile.
    NN_ASSERT(m_Recorder.Prepare(&m_Parameter, &m_RecoderAux, &m_RecordingWav, bufferSize, recordingDuration, m_RecorderBuffer, m_RecorderBufferSize).IsSuccess());
    m_Recorder.EnableLeadSilenceTrimming(trimSilence);
}

void WaveComparisonTestBase::PrepareSourceVoice(std::string sourceFile)
{

    size_t dataShutterSize = nnt::audio::util::GetWaveSampleSize(sourceFile);
    m_InputWaveFileDataBuffer = m_Allocator.Allocate(nn::util::align_up(dataShutterSize, nn::audio::MemoryPoolType::SizeGranularity),
                                                     nn::audio::MemoryPoolType::AddressAlignment);
    NN_ASSERT(m_InputWaveFileDataBuffer != nullptr);
    nn::audio::AcquireMemoryPool(m_Renderer.GetConfig(), &m_SourcePool, m_InputWaveFileDataBuffer, nn::util::align_up(dataShutterSize, nn::audio::MemoryPoolType::SizeGranularity));
    nn::audio::RequestAttachMemoryPool(&m_SourcePool);

    int sampleRate = 0;
    size_t dataSize = nnt::audio::util::LoadSourceSamples(sourceFile, m_InputWaveFileDataBuffer, dataShutterSize, &sampleRate);

    NN_ASSERT(nn::audio::AcquireVoiceSlot(m_Renderer.GetConfig(),
                                            &m_InputVoice,
                                            sampleRate,
                                            1,
                                            nn::audio::SampleFormat_PcmInt16,
                                            nn::audio::VoiceType::PriorityHighest,
                                            nullptr,
                                            0));

    m_InputWaveBuffer.buffer = m_InputWaveFileDataBuffer;
    m_InputWaveBuffer.size = dataSize;
    m_InputWaveBuffer.startSampleOffset = 0;
    m_InputWaveBuffer.endSampleOffset = static_cast<int32_t>(dataSize / sizeof(int16_t));
    m_InputWaveBuffer.loop = true;
    m_InputWaveBuffer.isEndOfStream = false;
    m_InputWaveBuffer.pContext = nullptr;
    m_InputWaveBuffer.contextSize = 0;
    NN_ASSERT(nn::audio::AppendWaveBuffer(&m_InputVoice, &m_InputWaveBuffer));
    nn::audio::SetVoicePlayState(&m_InputVoice, nn::audio::VoiceType::PlayState_Play);

    nn::audio::SetVoiceDestination(m_Renderer.GetConfig(), &m_InputVoice, m_Renderer.GetSubMix());
    nn::audio::SetVoiceMixVolume(&m_InputVoice, m_Renderer.GetSubMix(), 0.707f / 2, 0, 0);
    nn::audio::SetVoiceMixVolume(&m_InputVoice, m_Renderer.GetSubMix(), 0.707f / 2, 0, 1);
    nn::audio::SetVoiceMixVolume(&m_InputVoice, m_Renderer.GetSubMix(), 0.707f / 2, 0, 2);
    nn::audio::SetVoiceMixVolume(&m_InputVoice, m_Renderer.GetSubMix(), 0.707f / 2, 0, 3);
    nn::audio::SetVoiceMixVolume(&m_InputVoice, m_Renderer.GetSubMix(), 0.707f / 2, 0, 4);
    nn::audio::SetVoiceMixVolume(&m_InputVoice, m_Renderer.GetSubMix(), 0.707f / 2, 0, 5);
}

} // namespace audio
} // namespace nnt
