﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testAudio_Effect.h"

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/audio.h>
#include <nnt/audioUtil/testAudio_Util.h>
#include <nnt/audioUtil/testAudio_Constants.h>
#include <nn/fs/fs_Result.h>

#include <memory> // std::unique_ptr

#include "../../Programs/Eris/Sources/Libraries/audio/dsp/audio_EffectCommon.h" // QF_FRACTIONAL_BIT_COUNT
#include <cmath> //pow

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/audio/audio_AudioRendererApi-os.win32.h>
#endif
namespace
{
    static const int QF_FRACTIONAL_BIT_COUNT = 14;
    const std::string g_AudioTestSourceFileName = "/shutter.wav";
    const std::string g_AudioTestDelayed1chFileName = "/testAudio_Effect_Delay1ch.wav";
    const std::string g_AudioTestDelayed2chFileName[] = {"/testAudio_Effect_Delay2chFinalMix.wav", "/testAudio_Effect_Delay2chSubMix.wav"};
    const std::string g_AudioTestDelayed4chFileName[] = {"/testAudio_Effect_Delay4chFinalMix.wav", "/testAudio_Effect_Delay4chSubMix.wav"};
    const std::string g_AudioTestDelayed6chFileName[] = {"/testAudio_Effect_Delay6chFinalMix.wav", "/testAudio_Effect_Delay6chSubMix.wav"};
    const float g_AudioTestQfPrecision = 1.0f / std::powf(2, static_cast<float>(QF_FRACTIONAL_BIT_COUNT));

    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WorkBuffer[1024 * 1024 * 3];
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WorkBuffer2[1024 * 1024 * 2];
}
// ========================================================================================
// Test cases
// ========================================================================================

template<typename T>
void TestAddRemoveDelay_SuccessImpl(nn::audio::AudioRendererConfig* pConfig, T* pMix, void* buffer, size_t bufferSize, int channelCount, nn::TimeSpan delayTime)
{
    nn::audio::DelayType delay;
    ASSERT_TRUE(nn::audio::AddDelay(pConfig, &delay, buffer, bufferSize, pMix, delayTime, channelCount).IsSuccess());
    nn::audio::SetDelayEnabled(&delay, false);
    ASSERT_TRUE(RemoveDelay(pConfig, &delay, pMix) == buffer);
}

TEST(TestAddRemoveDelay, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::mem::StandardAllocator effectAllocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));

    nnt::audio::EffectScopedRenderer renderer(allocator);

    const auto param = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    const nn::TimeSpan delayTime = nn::TimeSpan::FromSeconds(1);
    const int channelCount = 2;
    const size_t extBufSize = nn::audio::GetRequiredBufferSizeForDelay(delayTime, param.sampleRate, channelCount);
    void* delayBuffer = effectAllocator.Allocate(extBufSize);
    NN_ASSERT_NOT_NULL(delayBuffer);

    renderer.RendererSetup(&param);
    TestAddRemoveDelay_SuccessImpl(renderer.GetConfig(), renderer.GetSubMix(), delayBuffer, extBufSize, channelCount, delayTime);
    TestAddRemoveDelay_SuccessImpl(renderer.GetConfig(), renderer.GetFinalMix(), delayBuffer, extBufSize, channelCount, delayTime);

    renderer.RendererShutdown();
    effectAllocator.Free(delayBuffer);
    effectAllocator.Finalize();
}

TEST(TestAddRemoveDelay, SuccessStopRenderer)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::mem::StandardAllocator effectAllocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));

    nnt::audio::EffectScopedRenderer renderer(allocator);

    const auto param = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    const nn::TimeSpan delayTime = nn::TimeSpan::FromSeconds(1);
    const int channelCount = 2;
    const size_t extBufSize = nn::audio::GetRequiredBufferSizeForDelay(delayTime, param.sampleRate, channelCount);
    void* delayBuffer = effectAllocator.Allocate(extBufSize);
    NN_ASSERT_NOT_NULL(delayBuffer);

    renderer.RendererSetup(&param);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    nn::audio::DelayType delay;
    NNT_ASSERT_RESULT_SUCCESS(nn::audio::AddDelay(renderer.GetConfig(), &delay, delayBuffer, extBufSize, renderer.GetSubMix(), delayTime, channelCount));

    renderer.Start();
    renderer.Wait(); // wait an audio frame
    renderer.Stop(); // stop renderer
    nn::audio::SetDelayEnabled(&delay, false);
    EXPECT_DEATH_IF_SUPPORTED(RemoveDelay(renderer.GetConfig(), &delay, renderer.GetSubMix()), ""); // Update を呼んでいないので、まだ消せない
    renderer.Update(); // Stop 後の Update ですべての Effect は removable になる想定
    ASSERT_TRUE(RemoveDelay(renderer.GetConfig(), &delay, renderer.GetSubMix()) == delayBuffer); // 削除成功

    renderer.RendererShutdown();
    effectAllocator.Free(delayBuffer);
}

TEST(TestAddRemoveDelay, SuccessMultiAddRemove)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::mem::StandardAllocator effectAllocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));

    nnt::audio::EffectScopedRenderer renderer(allocator);

    const auto param = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    const nn::TimeSpan delayTime = nn::TimeSpan::FromSeconds(1);
    const int channelCount = 2;
    const size_t delaySize = nn::audio::GetRequiredBufferSizeForDelay(delayTime, param.sampleRate, channelCount);

    renderer.RendererSetup(&param);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    nn::audio::DelayType delay1, delay2;
    void* delayBuffer1 = effectAllocator.Allocate(delaySize);
    NN_ASSERT_NOT_NULL(delayBuffer1);
    void* delayBuffer2 = effectAllocator.Allocate(delaySize);
    NN_ASSERT_NOT_NULL(delayBuffer2);

    // 事前に二つ登録
    ASSERT_TRUE(nn::audio::AddDelay(renderer.GetConfig(), &delay1, delayBuffer1, delaySize, renderer.GetSubMix(), delayTime, channelCount).IsSuccess());
    ASSERT_TRUE(nn::audio::AddDelay(renderer.GetConfig(), &delay2, delayBuffer2, delaySize, renderer.GetSubMix(), delayTime, channelCount).IsSuccess());

    nn::audio::SetDelayEnabled(&delay1, false);
    ASSERT_FALSE(nn::audio::IsDelayEnabled(&delay1));
    nn::audio::SetDelayEnabled(&delay1, true);
    ASSERT_TRUE(nn::audio::IsDelayEnabled(&delay1));

    // 最初に登録した方を削除。
    nn::audio::SetDelayEnabled(&delay1, false);
    nn::audio::RemoveDelay(renderer.GetConfig(),&delay1, renderer.GetSubMix());
    // 一度消したものの、再削除
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::RemoveDelay(renderer.GetConfig(),&delay1, renderer.GetSubMix()), "");

    renderer.Start();
    renderer.Update();
    renderer.Wait();
    for (auto i = 0; i < 1; i ++) // 10 は適当な数。深い意味はない。
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::audio::AddDelay(renderer.GetConfig(), &delay1, delayBuffer1, delaySize, renderer.GetSubMix(), delayTime, channelCount));
        // 一度も Update する前であれば Start 中でも消せる
        nn::audio::SetDelayEnabled(&delay1, false);
        ASSERT_TRUE(nn::audio::RemoveDelay(renderer.GetConfig(), &delay1, renderer.GetSubMix()) == delayBuffer1);
        NNT_ASSERT_RESULT_SUCCESS(nn::audio::AddDelay(renderer.GetConfig(), &delay1, delayBuffer1, delaySize, renderer.GetSubMix(), delayTime, channelCount));

        renderer.Wait();
        renderer.Update();
        // Update したので、disabled するまで消せない。
        EXPECT_DEATH_IF_SUPPORTED(nn::audio::RemoveDelay(renderer.GetConfig(),&delay1, renderer.GetSubMix()), "");
        nn::audio::SetDelayEnabled(&delay1, false); // renderer が start しているので、false にしないと消せない。

        // 無効化直後の Remove。まだ消せる状態ではない。
        EXPECT_DEATH_IF_SUPPORTED(nn::audio::RemoveDelay(renderer.GetConfig(),&delay1, renderer.GetSubMix()), "");

        while(IsDelayRemovable(&delay1) == false)
        {
            renderer.Wait();
            renderer.Update();
        }
        ASSERT_TRUE(nn::audio::RemoveDelay(renderer.GetConfig(), &delay1, renderer.GetSubMix()) == delayBuffer1);
    }
    renderer.RendererShutdown();

    effectAllocator.Free(delayBuffer1);
    effectAllocator.Free(delayBuffer2);
}

TEST(TestAddRemoveDelay, SuccessAddRemoveManyTimes)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::mem::StandardAllocator effectAllocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));

    nnt::audio::EffectScopedRenderer renderer(allocator);

    const auto param = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    const int channelCountMax = 4;
    const nn::TimeSpan delayTime = nn::TimeSpan::FromSeconds(1);
    const size_t delaySize = nn::audio::GetRequiredBufferSizeForDelay(delayTime, param.sampleRate, channelCountMax);

    renderer.RendererSetup(&param);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    nn::audio::DelayType delay;
    void* delayBuffer = effectAllocator.Allocate(delaySize);
    NN_ASSERT_NOT_NULL(delayBuffer);

    // Renderer を開始
    renderer.Start();
    renderer.Update();
    renderer.Wait();

    for (auto i = 0; i < 50; ++i) // 50 は適当な数。深い意味はない。
    {
        const int channelCountTable[] = {1, 2, 4};
        const int channelCount = channelCountTable[i % (sizeof(channelCountTable) / sizeof(channelCountTable[0]))];

        // delay 用バッファに適当な値を書き込んで、ダーティな状態にしておく
        memset(delayBuffer, i, delaySize);

        // delay を renderer に追加し Update を呼び出すことで、delayBuffer が指すメモリを DSP が触っている状態にする
        NNT_ASSERT_RESULT_SUCCESS(nn::audio::AddDelay(renderer.GetConfig(), &delay, delayBuffer, delaySize, renderer.GetSubMix(), delayTime, channelCount));
        renderer.Update();
        renderer.Wait();

        // あえて workBuffer を破壊しても停止することなく動作を続ける
        memset(delayBuffer, i, delaySize);
        nn::os::FlushDataCache(delayBuffer, delaySize);

        // 外すために無効化
        nn::audio::SetDelayEnabled(&delay, false);

        // 外れるまで待機
        while(IsDelayRemovable(&delay) == false)
        {
            renderer.Update();
            renderer.Wait();
        }

        // delay を外す
        ASSERT_TRUE(nn::audio::RemoveDelay(renderer.GetConfig(), &delay, renderer.GetSubMix()) == delayBuffer);
    }

    renderer.RendererShutdown();
    effectAllocator.Free(delayBuffer);
}

namespace {
void AllocateTestBuffer(void** ppAllocatedAddress, void** ppAlignedAddress, size_t allocSize) NN_NOEXCEPT
{
    auto buf = malloc(allocSize);
    NN_ASSERT_NOT_NULL(buf);
    auto alignedBuf = nn::util::align_up(reinterpret_cast<uintptr_t>(buf), nn::audio::MemoryPoolType::AddressAlignment);
    *ppAllocatedAddress = buf;
    *ppAlignedAddress = reinterpret_cast<void*>(alignedBuf);
}
}

TEST(TestAddRemoveDelay, MultiRenderer)
{
    const size_t workBufferSize = 1024 * 1024 * 2;
    void* workBuffer1 = nullptr;
    void* workBuffer2 = nullptr;
    void* effectBuffer1 = nullptr;
    void* effectBuffer2 = nullptr;
    void* alignedWorkBuffer1 = nullptr;
    void* alignedWorkBuffer2 = nullptr;
    void* alignedEffectBuffer1 = nullptr;
    void* alignedEffectBuffer2 = nullptr;

    AllocateTestBuffer(&workBuffer1, &alignedWorkBuffer1, workBufferSize);
    AllocateTestBuffer(&workBuffer2, &alignedWorkBuffer2, workBufferSize);
    AllocateTestBuffer(&effectBuffer1, &alignedEffectBuffer1, workBufferSize);
    AllocateTestBuffer(&effectBuffer2, &alignedEffectBuffer2, workBufferSize);
    nn::mem::StandardAllocator allocator1(reinterpret_cast<void*>(alignedWorkBuffer1), workBufferSize);
    nn::mem::StandardAllocator allocator2(reinterpret_cast<void*>(alignedWorkBuffer2), workBufferSize);
    nn::mem::StandardAllocator effectAllocator1(reinterpret_cast<void*>(alignedEffectBuffer1), workBufferSize);
    nn::mem::StandardAllocator effectAllocator2(reinterpret_cast<void*>(alignedEffectBuffer2), workBufferSize);

    nnt::audio::EffectScopedRenderer renderer1(allocator1);
    nnt::audio::EffectScopedRenderer renderer2(allocator2);

    const auto param = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    const nn::TimeSpan delayTime = nn::TimeSpan::FromSeconds(1);
    const int channelCount = 2;
    const size_t delaySize = nn::audio::GetRequiredBufferSizeForDelay(delayTime, param.sampleRate, channelCount);

    renderer1.RendererSetup(&param);
    renderer2.RendererSetup(&param);

    nn::audio::MemoryPoolType memoryPool1;
    nn::audio::MemoryPoolType memoryPool2;
    nn::audio::AcquireMemoryPool(renderer1.GetConfig(), &memoryPool1, reinterpret_cast<void*>(alignedEffectBuffer1), workBufferSize);
    nn::audio::RequestAttachMemoryPool(&memoryPool1);
    nn::audio::AcquireMemoryPool(renderer2.GetConfig(), &memoryPool2, reinterpret_cast<void*>(alignedEffectBuffer2), workBufferSize);
    nn::audio::RequestAttachMemoryPool(&memoryPool2);

    nn::audio::DelayType delay1, delay2;
    void* delayBuffer1 = effectAllocator1.Allocate(delaySize);
    void* delayBuffer2 = effectAllocator2.Allocate(delaySize);
    NN_ASSERT_NOT_NULL(delayBuffer1);
    NN_ASSERT_NOT_NULL(delayBuffer2);

    ASSERT_TRUE(nn::audio::AddDelay(renderer1.GetConfig(), &delay1, delayBuffer1, delaySize, renderer1.GetSubMix(), delayTime, channelCount).IsSuccess());
    ASSERT_TRUE(nn::audio::AddDelay(renderer2.GetConfig(), &delay2, delayBuffer2, delaySize, renderer2.GetSubMix(), delayTime, channelCount).IsSuccess());

    renderer1.Start();
    renderer2.Start();
    renderer1.Update();
    renderer2.Update();
    renderer1.Wait();
    renderer2.Wait();

    EXPECT_EQ(IsDelayRemovable(&delay1), false);
    EXPECT_EQ(IsDelayRemovable(&delay2), false);

    // renderer1 のみ Stop させ renderer2 への影響のないことを確認する
    renderer1.Stop();
    nn::audio::SetDelayEnabled(&delay1, false);

    const int timeoutCount = 100;
    auto count = 0;
    for (count = 0; count < 100; ++count)
    {
        if (IsDelayRemovable(&delay1) == true)
        {
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        renderer1.Update();
    }
    ASSERT_LT(count, timeoutCount);

    EXPECT_EQ(IsDelayRemovable(&delay1), true);
    EXPECT_EQ(IsDelayRemovable(&delay2), false);
    ASSERT_TRUE(nn::audio::RemoveDelay(renderer1.GetConfig(), &delay1, renderer1.GetSubMix()) == delayBuffer1);

    renderer1.RendererShutdown();

    renderer2.Wait();
    renderer2.Update();

    EXPECT_EQ(IsDelayRemovable(&delay2), false);
    nn::audio::SetDelayEnabled(&delay2, false);
    for (count = 0; count < 100; ++count)
    {
        if (IsDelayRemovable(&delay2) == true)
        {
            break;
        }

        renderer2.Wait();
        renderer2.Update();
    }
    ASSERT_LT(count, timeoutCount);

    EXPECT_EQ(IsDelayRemovable(&delay2), true);
    ASSERT_TRUE(nn::audio::RemoveDelay(renderer2.GetConfig(), &delay2, renderer2.GetSubMix()) == delayBuffer2);

    renderer2.RendererShutdown();

    effectAllocator1.Free(delayBuffer1);
    effectAllocator2.Free(delayBuffer2);

    allocator1.Finalize();
    allocator2.Finalize();
    effectAllocator1.Finalize();
    effectAllocator2.Finalize();

    free(workBuffer1);
    free(workBuffer2);
    free(effectBuffer1);
    free(effectBuffer2);
} // NOLINT(readability/fn_size)


#if !defined(NN_SDK_BUILD_RELEASE)
template<typename T>
void TestAddRemoveDelay_PreconditionImpl(nn::audio::AudioRendererConfig* pConfig, T* pMix, void* buffer, size_t bufferSize, int channelCount, nn::TimeSpan delayTime)
{
    nn::audio::DelayType delay;

    nn::TimeSpan invalidDelayTime = nn::TimeSpan::FromDays(1);
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddDelay(nullptr, &delay, buffer, bufferSize, pMix, delayTime, channelCount), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddDelay(pConfig, nullptr, buffer, bufferSize, pMix, delayTime, channelCount), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddDelay(pConfig, &delay, nullptr, bufferSize, pMix, delayTime, channelCount), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddDelay(pConfig, &delay, buffer, bufferSize, static_cast<T*>(nullptr), delayTime, channelCount), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddDelay(pConfig, &delay, buffer, bufferSize, pMix, invalidDelayTime, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddDelay(pConfig, &delay, buffer, bufferSize, pMix, delayTime, -1), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddDelay(pConfig, &delay, buffer, bufferSize, pMix, delayTime, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddDelay(pConfig, &delay, buffer, bufferSize, pMix, delayTime, 3), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddDelay(pConfig, &delay, buffer, bufferSize, pMix, delayTime, 5), "");

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::RemoveDelay(nullptr, &delay, pMix), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::RemoveDelay(pConfig, nullptr, pMix), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::RemoveDelay(pConfig, &delay, static_cast<T*>(nullptr)), "");
}

TEST(TestAddRemoveDelay, Precondition)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::mem::StandardAllocator effectAllocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));

    nnt::audio::EffectScopedRenderer renderer(allocator);

    const auto param = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    const nn::TimeSpan delayTime = nn::TimeSpan::FromSeconds(1);
    const int channelCount = 2;
    const size_t delaySize = nn::audio::GetRequiredBufferSizeForDelay(delayTime, param.sampleRate, channelCount);

    renderer.RendererSetup(&param);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    void* delayBuffer = effectAllocator.Allocate(delaySize);
    NN_ASSERT_NOT_NULL(delayBuffer);

    TestAddRemoveDelay_PreconditionImpl(renderer.GetConfig(), renderer.GetSubMix(), delayBuffer, delaySize, channelCount, delayTime);
    TestAddRemoveDelay_PreconditionImpl(renderer.GetConfig(), renderer.GetFinalMix(), delayBuffer, delaySize, channelCount, delayTime);

    renderer.RendererShutdown();

    effectAllocator.Free(delayBuffer);

}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(TestGetSetInputOutputDelay, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::mem::StandardAllocator effectAllocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));

    nnt::audio::EffectScopedRenderer renderer(allocator);

    const auto param = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    const nn::TimeSpan delayTime = nn::TimeSpan::FromSeconds(1);
    const int channelCount = 4;
    const size_t delaySize = nn::audio::GetRequiredBufferSizeForDelay(delayTime, param.sampleRate, channelCount);

    renderer.RendererSetup(&param);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    nn::audio::FinalMixType* finalMix = renderer.GetFinalMix();
    nn::audio::DelayType delay;
    void* delayBuffer = effectAllocator.Allocate(delaySize);
    NN_ASSERT_NOT_NULL(delayBuffer);

    ASSERT_TRUE(nn::audio::AddDelay(renderer.GetConfig(), &delay, delayBuffer, delaySize, finalMix, delayTime, channelCount).IsSuccess());

    const int8_t inputCh[channelCount] = { 1, 2, 3, 4 };
    const int8_t outputCh[channelCount] = { 5, 6, 7, 8 };
    nn::audio::SetDelayInputOutput(&delay, inputCh, outputCh, channelCount);

    const int arrayLen = channelCount;
    int outCount = 0;
    int8_t outInputCh[arrayLen] = { 0 };
    int8_t outOutputCh[arrayLen] = { 0 };
    nn::audio::GetDelayInputOutput(&delay, outInputCh, outOutputCh, &outCount, arrayLen);

    EXPECT_EQ(outCount, channelCount);
    for (auto i = 0; i < outCount; ++i)
    {
        EXPECT_EQ(outInputCh[i], inputCh[i]);
        EXPECT_EQ(outOutputCh[i], outputCh[i]);
    }

    renderer.RendererShutdown();
    effectAllocator.Free(delayBuffer);
}


#if !defined(NN_SDK_BUILD_RELEASE)
TEST(TestGetSetInputOutputDelay, Precondition)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::mem::StandardAllocator effectAllocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));

    nnt::audio::EffectScopedRenderer renderer(allocator);

    const auto param = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    const nn::TimeSpan delayTime = nn::TimeSpan::FromSeconds(1);
    const int channelCount = 2;
    const int8_t inputCh[channelCount] = { 1, 2 };
    const int8_t outputCh[channelCount] = { 3, 4 };
    const size_t delaySize = nn::audio::GetRequiredBufferSizeForDelay(delayTime, param.sampleRate, channelCount);

    renderer.RendererSetup(&param);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    nn::audio::FinalMixType* finalMix = renderer.GetFinalMix();
    nn::audio::DelayType delay;
    void* delayBuffer = effectAllocator.Allocate(delaySize);
    NN_ASSERT_NOT_NULL(delayBuffer);
    ASSERT_TRUE(nn::audio::AddDelay(renderer.GetConfig(), &delay, delayBuffer, delaySize, finalMix, delayTime, channelCount).IsSuccess());

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetDelayInputOutput(nullptr, inputCh, outputCh, channelCount), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetDelayInputOutput(&delay, nullptr, outputCh, channelCount), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetDelayInputOutput(&delay, inputCh, nullptr, channelCount), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetDelayInputOutput(&delay, inputCh, outputCh, -1), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetDelayInputOutput(&delay, inputCh, outputCh, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetDelayInputOutput(&delay, inputCh, outputCh, 3), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetDelayInputOutput(&delay, inputCh, outputCh, 4), ""); // check if "<= GetDelayChannelCountMax()"
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetDelayInputOutput(&delay, inputCh, outputCh, 5), "");

    nn::audio::SetDelayInputOutput(&delay, inputCh, outputCh, channelCount);

    const int arrayLen = 4;
    int outCount = 0;
    int8_t outInputCh[arrayLen] = { 0 };
    int8_t outOutputCh[arrayLen] = { 0 };

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetDelayInputOutput(nullptr, outInputCh, outOutputCh, &outCount, arrayLen), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetDelayInputOutput(&delay, nullptr, outOutputCh, &outCount, arrayLen), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetDelayInputOutput(&delay, outInputCh, nullptr, &outCount, arrayLen), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetDelayInputOutput(&delay, outInputCh, outOutputCh, nullptr, arrayLen), "");
    // TODO:: add some precondition checks.

    renderer.RendererShutdown();
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(TestGetSetParametersDelay, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::mem::StandardAllocator effectAllocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));

    nnt::audio::EffectScopedRenderer renderer(allocator);

    const auto param = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    const nn::TimeSpan delayTime = nn::TimeSpan::FromSeconds(1);
    const int channelCount = 4;
    const size_t delaySize = nn::audio::GetRequiredBufferSizeForDelay(delayTime, param.sampleRate, channelCount);

    renderer.RendererSetup(&param);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    nn::audio::FinalMixType* finalMix = renderer.GetFinalMix();
    nn::audio::DelayType delay;
    void* delayBuffer = effectAllocator.Allocate(delaySize);
    NN_ASSERT_NOT_NULL(delayBuffer);
    ASSERT_TRUE(nn::audio::AddDelay(renderer.GetConfig(), &delay, delayBuffer, delaySize, finalMix, delayTime, channelCount).IsSuccess());

    // delayTimeMax
    EXPECT_EQ(delayTime, nn::audio::GetDelayTimeMax(&delay));

    // delayTime
    nn::TimeSpan t = nn::TimeSpan::FromMilliSeconds(10);
    nn::audio::SetDelayTime(&delay, t);
    EXPECT_NEAR(static_cast<float>(t.GetMilliSeconds()), static_cast<float>(nn::audio::GetDelayTime(&delay).GetMilliSeconds()), g_AudioTestQfPrecision);

    // inGain
    float inGain = 0.1f;
    nn::audio::SetDelayInGain(&delay, inGain);
    EXPECT_NEAR(inGain, nn::audio::GetDelayInGain(&delay), g_AudioTestQfPrecision);

    // feedbackGain
    float feedbackGain = 0.2f;
    nn::audio::SetDelayFeedbackGain(&delay, feedbackGain);
    EXPECT_NEAR(feedbackGain, nn::audio::GetDelayFeedbackGain(&delay), g_AudioTestQfPrecision);

    // dryGain
    float dryGain = 0.3f;
    nn::audio::SetDelayDryGain(&delay, dryGain);
    EXPECT_NEAR(dryGain, nn::audio::GetDelayDryGain(&delay), g_AudioTestQfPrecision);

    // channelSpread
    float spread = 0.4f;
    nn::audio::SetDelayChannelSpread(&delay, spread);
    EXPECT_NEAR(spread, nn::audio::GetDelayChannelSpread(&delay), g_AudioTestQfPrecision);

    // lowPassAmount
    float lpa = 0.5f;
    nn::audio::SetDelayLowPassAmount(&delay, lpa);
    EXPECT_NEAR(lpa, nn::audio::GetDelayLowPassAmount(&delay), g_AudioTestQfPrecision);

    renderer.RendererShutdown();
}

TEST(TestSetDelayParameters, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::mem::StandardAllocator effectAllocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));

    nnt::audio::EffectScopedRenderer renderer(allocator);

    const auto param = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    const nn::TimeSpan delayTime = nn::TimeSpan::FromSeconds(1);
    const int channelCount = 4;
    const size_t delaySize = nn::audio::GetRequiredBufferSizeForDelay(delayTime, param.sampleRate, channelCount);
    renderer.RendererSetup(&param);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    nn::audio::FinalMixType* finalMix = renderer.GetFinalMix();
    nn::audio::DelayType delay;
    void* delayBuffer = effectAllocator.Allocate(delaySize);
    NN_ASSERT_NOT_NULL(delayBuffer);
    ASSERT_TRUE(nn::audio::AddDelay(renderer.GetConfig(), &delay, delayBuffer, delaySize, finalMix, delayTime, channelCount).IsSuccess());

    const float validValuesFloat[] = {0.0f, 1.0f};
    const nn::TimeSpan validValuesTime[] = {nn::TimeSpan::FromMilliSeconds(0), nn::audio::GetDelayTimeMax(&delay)};
    // delayTimeMax
    EXPECT_EQ(delayTime, nn::audio::GetDelayTimeMax(&delay));

    // delayTime
    nn::TimeSpan t = nn::TimeSpan::FromMilliSeconds(10);
    float inGain = 0.1f;
    float feedbackGain = 0.2f;
    float dryGain = 0.3f;
    float lpa = 0.5f;
    float spread = 0.4f;

    nn::audio::DelayParameterSet delayParam;
    delayParam.delayTime = t;
    delayParam.inGain = inGain;
    delayParam.feedbackGain = feedbackGain;
    delayParam.dryGain = dryGain;
    delayParam.lowPassAmount = lpa;
    delayParam.channelSpread = spread;

    nn::audio::SetDelayParameters(&delay, &delayParam);

    nn::audio::DelayParameterSet returnedParam;
    returnedParam.delayTime = nn::TimeSpan::FromMilliSeconds(10);;
    returnedParam.inGain = 0.0f;
    returnedParam.feedbackGain = 0.0f;
    returnedParam.dryGain = 0.0f;
    returnedParam.lowPassAmount = 0.0f;
    returnedParam.channelSpread = 0.0f;

    returnedParam = nn::audio::GetDelayParameters(&delay);
    EXPECT_EQ(returnedParam.delayTime, t);
    EXPECT_NEAR(inGain, returnedParam.inGain, g_AudioTestQfPrecision);
    EXPECT_NEAR(feedbackGain, returnedParam.feedbackGain, g_AudioTestQfPrecision);
    EXPECT_NEAR(dryGain, returnedParam.dryGain, g_AudioTestQfPrecision);
    EXPECT_NEAR(lpa, returnedParam.lowPassAmount, g_AudioTestQfPrecision);
    EXPECT_NEAR(spread, returnedParam.channelSpread, g_AudioTestQfPrecision);
    for( const auto value : validValuesFloat)
    {
        delayParam.inGain = value;
        delayParam.feedbackGain = value;
        delayParam.dryGain = value;
        delayParam.lowPassAmount = value;
        delayParam.channelSpread = value;

        nn::audio::SetDelayParameters(&delay, &delayParam);
        returnedParam = nn::audio::GetDelayParameters(&delay);

        EXPECT_EQ(returnedParam.delayTime, delayParam.delayTime);
        EXPECT_NEAR(value, returnedParam.inGain, g_AudioTestQfPrecision);
        EXPECT_NEAR(value, returnedParam.feedbackGain, g_AudioTestQfPrecision);
        EXPECT_NEAR(value, returnedParam.dryGain, g_AudioTestQfPrecision);
        EXPECT_NEAR(value, returnedParam.lowPassAmount, g_AudioTestQfPrecision);
        EXPECT_NEAR(value, returnedParam.channelSpread, g_AudioTestQfPrecision);
    }
    for( auto i = 0; i < 2; ++i)
    {
        delayParam.delayTime = validValuesTime[i];

        nn::audio::SetDelayParameters(&delay, &delayParam);
        returnedParam = nn::audio::GetDelayParameters(&delay);

        EXPECT_EQ(returnedParam.delayTime, validValuesTime[i]);
        EXPECT_NEAR(delayParam.inGain, returnedParam.inGain, g_AudioTestQfPrecision);
        EXPECT_NEAR(delayParam.feedbackGain, returnedParam.feedbackGain, g_AudioTestQfPrecision);
        EXPECT_NEAR(delayParam.dryGain, returnedParam.dryGain, g_AudioTestQfPrecision);
        EXPECT_NEAR(delayParam.lowPassAmount, returnedParam.lowPassAmount, g_AudioTestQfPrecision);
        EXPECT_NEAR(delayParam.channelSpread, returnedParam.channelSpread, g_AudioTestQfPrecision);
    }

    renderer.RendererShutdown();
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(TestGetSetParametersDelay, Precondition)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::mem::StandardAllocator effectAllocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));

    nnt::audio::EffectScopedRenderer renderer(allocator);

    const auto param = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    const nn::TimeSpan delayTime = nn::TimeSpan::FromSeconds(1);
    const int channelCount = 4;
    const size_t delaySize = nn::audio::GetRequiredBufferSizeForDelay(delayTime, param.sampleRate, channelCount);

    renderer.RendererSetup(&param);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    nn::audio::FinalMixType* finalMix = renderer.GetFinalMix();
    nn::audio::DelayType delay;
    void* delayBuffer = effectAllocator.Allocate(delaySize);
    NN_ASSERT_NOT_NULL(delayBuffer);
    ASSERT_TRUE(nn::audio::AddDelay(renderer.GetConfig(), &delay, delayBuffer, delaySize, finalMix, delayTime, channelCount).IsSuccess());

    // delayTimeMax
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetDelayTimeMax(nullptr), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetDelayTime(nullptr, nn::TimeSpan::FromSeconds(1)), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetDelayTime(nullptr), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetDelayInGain(nullptr, 0.0f), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetDelayInGain(nullptr), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetDelayFeedbackGain(nullptr, 0.0f), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetDelayFeedbackGain(nullptr), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetDelayDryGain(nullptr, 0.0f), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetDelayDryGain(nullptr), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetDelayChannelSpread(nullptr, 0.0f), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetDelayChannelSpread(nullptr), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetDelayLowPassAmount(nullptr, 0.0f), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetDelayLowPassAmount(nullptr), "");

    renderer.RendererShutdown();
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(TestDelay6chWaveComp, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    for (auto i = 0; i < 2; ++i)
    {

#if defined(NN_BUILD_CONFIG_OS_WIN)
        nn::os::Event event(nn::os::EventClearMode_AutoClear);
        nn::audio::SetFastRenderingMode(true, &event);
        EXPECT_TRUE(nn::audio::GetFastRenderingMode());
#endif
        std::string TargetFile(nnt::audio::util::GetResultDataDirectory() + g_AudioTestDelayed6chFileName[i]);
        std::string GoldenFile(g_RefernerceDataFolder + g_AudioTestDelayed6chFileName[i]);
        std::string SourceFile(g_SourceDataFolder + g_AudioTestSourceFileName);
        int testChannelCount = 6;

        nn::audio::AudioRendererParameter parameter = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
        nnt::audio::WaveComparisonTestBase testBase(parameter);
        testBase.Initialize();
        nn::audio::MemoryPoolType effectPool;
        nn::audio::AcquireMemoryPool(testBase.GetConfig(), &effectPool, g_WorkBuffer, sizeof(g_WorkBuffer));
        nn::audio::RequestAttachMemoryPool(&effectPool);

        int8_t bus[] = { 0, 1, 2, 3, 4, 5 }; // エフェクトの最大チャンネル数

                                             // TargetEffect Setup
                                             //////////////////////////////////////////////////////////////////////////
        size_t delaySize = nn::audio::GetRequiredBufferSizeForDelay(nn::TimeSpan::FromSeconds(2), parameter.sampleRate, testChannelCount);
        void* delayBuffer = allocator.Allocate(delaySize, nn::audio::BufferAlignSize);
        NN_ASSERT_NOT_NULL(delayBuffer);
        {
            nn::TimeSpan delayTimeMax = nn::TimeSpan::FromMilliSeconds(200);
            nn::audio::DelayType delay;
            if (i == 0)
            {
                nn::Result result = nn::audio::AddDelay(testBase.GetConfig(), &delay, delayBuffer, delaySize, testBase.GetFinalMix(), delayTimeMax, testChannelCount);
                ASSERT_TRUE(result.IsSuccess());
            }
            else
            {
                nn::Result result = nn::audio::AddDelay(testBase.GetConfig(), &delay, delayBuffer, delaySize, testBase.GetSubMix(), delayTimeMax, testChannelCount);
                ASSERT_TRUE(result.IsSuccess());
            }

            nn::audio::SetDelayInputOutput(&delay, bus, bus, testChannelCount);
            nn::audio::SetDelayEnabled(&delay, true);
            nn::audio::SetDelayTime(&delay, nn::TimeSpan::FromMilliSeconds(200));
            nn::audio::SetDelayFeedbackGain(&delay, 0.7f);
            nn::audio::SetDelayChannelSpread(&delay, 0.256f);
            nn::audio::SetDelayLowPassAmount(&delay, 0.9f);
            nn::audio::SetDelayDryGain(&delay, 0.5f);
            nn::audio::SetDelayInGain(&delay, 0.4f);
            nn::audio::SetDelayEnabled(&delay, true);
        }

        testBase.PrepareRecording(TargetFile);
        testBase.PrepareSourceVoice(SourceFile);

        testBase.Start();

        while (testBase.Record())
        {
#if defined(NN_BUILD_CONFIG_OS_WIN)
            nn::audio::TriggerRendering();
            event.Wait();
#endif
            testBase.Wait();
            testBase.Update();
        }

        testBase.Finalize();

#if defined(NN_BUILD_CONFIG_OS_WIN)
        // AudioRenderer がクローズされると FastRenderingMode は解除
        EXPECT_FALSE(nn::audio::GetFastRenderingMode());
#endif
        allocator.Free(delayBuffer);

        // 波形比較
        nnt::audio::util::SimpleVerify(TargetFile, GoldenFile);
    }
    allocator.Finalize();
}

TEST(TestDelay4chWaveComp, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    for (auto i = 0; i < 2; ++i)
    {

#if defined(NN_BUILD_CONFIG_OS_WIN)
        nn::os::Event event(nn::os::EventClearMode_AutoClear);
        nn::audio::SetFastRenderingMode(true, &event);
        EXPECT_TRUE(nn::audio::GetFastRenderingMode());
#endif
        std::string TargetFile(nnt::audio::util::GetResultDataDirectory() + g_AudioTestDelayed4chFileName[i]);
        std::string GoldenFile(g_RefernerceDataFolder + g_AudioTestDelayed4chFileName[i]);
        std::string SourceFile(g_SourceDataFolder + g_AudioTestSourceFileName);
        int testChannelCount = 4;

        nn::audio::AudioRendererParameter parameter = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
        nnt::audio::WaveComparisonTestBase testBase(parameter);
        testBase.Initialize();
        nn::audio::MemoryPoolType effectPool;
        nn::audio::AcquireMemoryPool(testBase.GetConfig(), &effectPool, g_WorkBuffer, sizeof(g_WorkBuffer));
        nn::audio::RequestAttachMemoryPool(&effectPool);

        int8_t bus[] = {0, 1, 2, 3, 4, 5}; // エフェクトの最大チャンネル数

        // TargetEffect Setup
        //////////////////////////////////////////////////////////////////////////
        size_t delaySize = nn::audio::GetRequiredBufferSizeForDelay(nn::TimeSpan::FromSeconds(2), parameter.sampleRate, testChannelCount);
        void* delayBuffer = allocator.Allocate(delaySize, nn::audio::BufferAlignSize);
        NN_ASSERT_NOT_NULL(delayBuffer);
        {
            nn::TimeSpan delayTimeMax = nn::TimeSpan::FromMilliSeconds(200);
            nn::audio::DelayType delay;
            if (i == 0)
            {
                nn::Result result = nn::audio::AddDelay(testBase.GetConfig(), &delay, delayBuffer, delaySize, testBase.GetFinalMix(), delayTimeMax, testChannelCount);
                ASSERT_TRUE(result.IsSuccess());
            }
            else
            {
                nn::Result result = nn::audio::AddDelay(testBase.GetConfig(), &delay, delayBuffer, delaySize, testBase.GetSubMix(), delayTimeMax, testChannelCount);
                ASSERT_TRUE(result.IsSuccess());
            }

            nn::audio::SetDelayInputOutput(&delay, bus, bus, testChannelCount);
            nn::audio::SetDelayEnabled(&delay, true);
            nn::audio::SetDelayTime(&delay, nn::TimeSpan::FromMilliSeconds(200));
            nn::audio::SetDelayFeedbackGain(&delay, 0.7f);
            nn::audio::SetDelayChannelSpread(&delay, 0.256f);
            nn::audio::SetDelayLowPassAmount(&delay, 0.9f);
            nn::audio::SetDelayDryGain(&delay, 0.5f);
            nn::audio::SetDelayInGain(&delay, 0.4f);
            nn::audio::SetDelayEnabled(&delay, true);
        }

        testBase.PrepareRecording(TargetFile);
        testBase.PrepareSourceVoice(SourceFile);

        testBase.Start();

        while (testBase.Record())
        {
#if defined(NN_BUILD_CONFIG_OS_WIN)
            nn::audio::TriggerRendering();
            event.Wait();
#endif
            testBase.Wait();
            testBase.Update();
        }

        testBase.Finalize();

#if defined(NN_BUILD_CONFIG_OS_WIN)
        // AudioRenderer がクローズされると FastRenderingMode は解除
        EXPECT_FALSE(nn::audio::GetFastRenderingMode());
#endif
        allocator.Free(delayBuffer);

        // 波形比較
        nnt::audio::util::SimpleVerify(TargetFile, GoldenFile);
    }
    allocator.Finalize();
}

TEST(TestDelay2chWaveComp, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    for (auto i = 0; i < 2; ++i)
    {
        std::string TargetFile(nnt::audio::util::GetResultDataDirectory() + g_AudioTestDelayed2chFileName[i]);
        std::string GoldenFile(g_RefernerceDataFolder + g_AudioTestDelayed2chFileName[i]);
        std::string SourceFile(g_SourceDataFolder + g_AudioTestSourceFileName);
        int testChannelCount = 2;

        nn::audio::AudioRendererParameter parameter = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
        nnt::audio::WaveComparisonTestBase testBase(parameter);
        testBase.Initialize();
        nn::audio::MemoryPoolType effectPool;
        nn::audio::AcquireMemoryPool(testBase.GetConfig(), &effectPool, g_WorkBuffer, sizeof(g_WorkBuffer));
        nn::audio::RequestAttachMemoryPool(&effectPool);

        int8_t bus[] = {0, 1, 2, 3, 4, 5}; // エフェクトの最大チャンネル数

        // TargetEffect Setup
        //////////////////////////////////////////////////////////////////////////
        size_t delaySize = nn::audio::GetRequiredBufferSizeForDelay(nn::TimeSpan::FromSeconds(2), parameter.sampleRate, testChannelCount);
        void* delayBuffer = allocator.Allocate(delaySize, nn::audio::BufferAlignSize);
        NN_ASSERT_NOT_NULL(delayBuffer);
        {
            nn::TimeSpan delayTimeMax = nn::TimeSpan::FromMilliSeconds(200);
            nn::audio::DelayType delay;
            if (i == 0)
            {
                nn::Result result = nn::audio::AddDelay(testBase.GetConfig(), &delay, delayBuffer, delaySize, testBase.GetFinalMix(), delayTimeMax, testChannelCount);
                ASSERT_TRUE(result.IsSuccess());
            }
            else
            {
                nn::Result result = nn::audio::AddDelay(testBase.GetConfig(), &delay, delayBuffer, delaySize, testBase.GetSubMix(), delayTimeMax, testChannelCount);
                ASSERT_TRUE(result.IsSuccess());
            }
            nn::audio::SetDelayInputOutput(&delay, bus, bus, testChannelCount);
            nn::audio::SetDelayEnabled(&delay, true);
            nn::audio::SetDelayTime(&delay, delayTimeMax);
            nn::audio::SetDelayFeedbackGain(&delay, 0.7f);
            nn::audio::SetDelayChannelSpread(&delay, 0.256f);
            nn::audio::SetDelayLowPassAmount(&delay, 0.9f);
            nn::audio::SetDelayDryGain(&delay, 0.5f);
            nn::audio::SetDelayInGain(&delay, 0.4f);
            nn::audio::SetDelayEnabled(&delay, true);
        }

        testBase.PrepareRecording(TargetFile);
        testBase.PrepareSourceVoice(SourceFile);

        testBase.Start();

        while (testBase.Record())
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16)); // yet-another Wait for Vsync
            testBase.Update();
        }

        testBase.Finalize();
        allocator.Free(delayBuffer);

        // 波形比較
        nnt::audio::util::SimpleVerify(TargetFile, GoldenFile);
    }
}

TEST(TestDelay1chWaveComp, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    const std::string SourceFile = g_SourceDataFolder + g_AudioTestSourceFileName;
    const std::string TargetFile = nnt::audio::util::GetResultDataDirectory() + g_AudioTestDelayed1chFileName;
    const std::string GoldenFile = g_RefernerceDataFolder + g_AudioTestDelayed1chFileName;

    nn::audio::AudioRendererParameter parameter = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    nnt::audio::WaveComparisonTestBase testBase(parameter);
    testBase.Initialize();
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(testBase.GetConfig(), &effectPool, g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    // TargetEffect Setup
    //////////////////////////////////////////////////////////////////////////
    size_t delaySize = nn::audio::GetRequiredBufferSizeForDelay(nn::TimeSpan::FromSeconds(2), parameter.sampleRate, 1);
    void* delayBuffer1 = allocator.Allocate(delaySize, nn::audio::BufferAlignSize);
    void* delayBuffer2 = allocator.Allocate(delaySize, nn::audio::BufferAlignSize);
    NN_ASSERT_NOT_NULL(delayBuffer1);
    NN_ASSERT_NOT_NULL(delayBuffer2);
    int testChannelCount = 1;
    {
        nn::TimeSpan delayTimeMax = nn::TimeSpan::FromSeconds(2);
        nn::audio::DelayType delay1;

        nn::Result result = nn::audio::AddDelay(testBase.GetConfig(), &delay1, delayBuffer1, delaySize, testBase.GetFinalMix(), delayTimeMax, testChannelCount);
        ASSERT_TRUE(result.IsSuccess());
        int8_t bus1[] = {0};
        nn::audio::SetDelayInputOutput(&delay1, bus1, bus1, sizeof(bus1));
        nn::audio::SetDelayEnabled(&delay1, true);
        nn::audio::SetDelayTime(&delay1, nn::TimeSpan::FromMilliSeconds(200));
        nn::audio::SetDelayFeedbackGain(&delay1, 0.7f);
        nn::audio::SetDelayChannelSpread(&delay1, 0.256f);
        nn::audio::SetDelayLowPassAmount(&delay1, 0.9f);
        nn::audio::SetDelayDryGain(&delay1, 0.5f);
        nn::audio::SetDelayInGain(&delay1, 0.4f);

        nn::audio::DelayType delay2;
        result = nn::audio::AddDelay(testBase.GetConfig(), &delay2, delayBuffer2, delaySize, testBase.GetSubMix(), delayTimeMax, testChannelCount);
        ASSERT_TRUE(result.IsSuccess());
        int8_t bus2[] = {1};
        nn::audio::SetDelayInputOutput(&delay2, bus2, bus2, sizeof(bus2));
        nn::audio::SetDelayEnabled(&delay2, true);
        nn::audio::SetDelayTime(&delay2, nn::TimeSpan::FromMilliSeconds(200));
        nn::audio::SetDelayFeedbackGain(&delay2, 0.7f);
        nn::audio::SetDelayChannelSpread(&delay2, 0.256f);
        nn::audio::SetDelayLowPassAmount(&delay2, 0.9f);
        nn::audio::SetDelayDryGain(&delay2, 0.5f);
        nn::audio::SetDelayInGain(&delay2, 0.4f);
    }

    testBase.PrepareRecording(TargetFile);
    testBase.PrepareSourceVoice(SourceFile);

    testBase.Start();

    while (testBase.Record())
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16)); // yet-another Wait for Vsync
        testBase.Update();
    }

    testBase.Finalize();

    // 波形比較
    nnt::audio::util::SimpleVerify(TargetFile, GoldenFile);
}

TEST(TestDelay, ResourceLimit)
{
    const int EffectCount = 4;
    const int EffectChannelCount = 6;

    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.effectCount = EffectCount;
    parameter.mixBufferCount = EffectChannelCount;

    nnt::audio::util::ScopedAudioRenderer sar(parameter);
    nn::audio::FinalMixType finalMix;
    nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, EffectChannelCount);

    nn::audio::DelayType delay[EffectCount + 1];

    const auto delayTime = nn::TimeSpan::FromSeconds(1);
    auto size = nn::audio::GetRequiredBufferSizeForDelay(delayTime, parameter.sampleRate, EffectChannelCount);
    auto buffer = allocator.Allocate(size, nn::audio::BufferAlignSize);
    EXPECT_TRUE(buffer != nullptr);

    for (auto i = 0; i < EffectCount; ++i)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AddDelay(&sar.GetConfig(), &delay[i], buffer, size, &finalMix, delayTime, EffectChannelCount));
    }
    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultOutOfResource, nn::audio::AddDelay(&sar.GetConfig(), &delay[EffectCount], buffer, size, &finalMix, delayTime, EffectChannelCount));

    for (auto i = 0; i < EffectCount; ++i)
    {
        nn::audio::SetDelayEnabled(&delay[i], false);
        EXPECT_TRUE(buffer == nn::audio::RemoveDelay(&sar.GetConfig(), &delay[i], &finalMix));
    }

    allocator.Free(buffer);
}

TEST(TestDelay, UpdateAudioRendererOnEachStep)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::mem::StandardAllocator effectAllocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));

    nnt::audio::EffectScopedRenderer renderer(allocator);

    const auto param = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    const int channelCountMax = 4;
    const nn::TimeSpan delayTime = nn::TimeSpan::FromSeconds(1);
    const size_t delaySize = nn::audio::GetRequiredBufferSizeForDelay(delayTime, param.sampleRate, channelCountMax);

    renderer.RendererSetup(&param);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    nn::audio::DelayType delay;
    void* delayBuffer = effectAllocator.Allocate(delaySize);
    NN_ASSERT_NOT_NULL(delayBuffer);

    // Renderer を開始
    renderer.Start();
    renderer.Update();
    renderer.Wait();

    for (auto i = 0; i < 10; ++i) // 10 は適当な数。深い意味はない。
    {
        const int channelCountTable[] = {1, 2, 4};
        const int channelCount = channelCountTable[i % (sizeof(channelCountTable) / sizeof(channelCountTable[0]))];
        EXPECT_LE(channelCount, channelCountMax);

        NNT_ASSERT_RESULT_SUCCESS(nn::audio::AddDelay(renderer.GetConfig(), &delay, delayBuffer, delaySize, renderer.GetSubMix(), delayTime, channelCount));
        renderer.Update();
        renderer.Wait();

        nn::audio::SetDelayTime(&delay, nn::TimeSpan::FromMilliSeconds(10));
        renderer.Update();
        renderer.Wait();

        nn::audio::SetDelayInGain(&delay, 0.1f);
        renderer.Update();
        renderer.Wait();

        nn::audio::SetDelayFeedbackGain(&delay, 0.2f);
        renderer.Update();
        renderer.Wait();

        nn::audio::SetDelayDryGain(&delay, 0.3f);
        renderer.Update();
        renderer.Wait();

        nn::audio::SetDelayChannelSpread(&delay, 0.4f);
        nn::audio::SetDelayLowPassAmount(&delay, 0.5f);
        renderer.Update();
        renderer.Wait();

        // 外すために無効化
        nn::audio::SetDelayEnabled(&delay, false);

        // 外れるまで待機
        while(IsDelayRemovable(&delay) == false)
        {
            renderer.Update();
            renderer.Wait();
        }

        // delay を外す
        ASSERT_TRUE(nn::audio::RemoveDelay(renderer.GetConfig(), &delay, renderer.GetSubMix()) == delayBuffer);

        renderer.Update();
        renderer.Wait();
    }

    renderer.RendererShutdown();
    effectAllocator.Free(delayBuffer);
} // NOLINT(readability/fn_size)
