﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cmath>
#include <queue>

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/fs.h>
#include <nn/nn_TimeSpan.h>
#include <nn/util/util_BytePtr.h>

#include <nn/audio.h>
//#include <nns/audio/audio_WavFormat.h>

#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>

#include <nn/settings/settings_DebugPad.h>

#include <nnt/audioUtil/testAudio_Util.h>
#include <nnt/audioUtil/testAudio_Constants.h>
#include <nn/audio/audio_Profiler.h>

namespace {

// choose rendering engine sample rate
    const int RenderRate = 48000;
//const int RenderRate = 32000;
    const int RenderCount = (RenderRate / 200);

// choose number of files to play
    const int SeCount = 64;

// - and add / remove them to / from the files lists
    const char* g_SeFileNames[] =
    {
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
        "asset:/Source/shutter.adpcm",
    };

// nn::audio::VoiceTypeの数
    const int VoiceNum = 128;


//--------------------------------------------------------------------------------
// メモリプール管理クラス
//--------------------------------------------------------------------------------
    class MemoryPoolManager
    {
    private:
        //--------------------------------------------------------------------------------
        //! struct
        //--------------------------------------------------------------------------------
        // メモリプールパラメータ
        struct MemoryPoolParam
        {
            //! コンストラクタ
            MemoryPoolParam()
                    : mRefCount(0)
                    , mpBuffer(nullptr)
                {
                    mMemoryPoolType._pMemoryPoolInfo = nullptr;
                };
            unsigned int                mRefCount;            //!< 参照カウンタ
            nn::audio::MemoryPoolType    mMemoryPoolType;    //!< nn::audio::MemoryPoolType
            void*                        mpBuffer;            //!< メモリプールにアタッチするバッファ
        };
        //--------------------------------------------------------------------------------

    public:
        //--------------------------------------------------------------------------------
        //! static const
        //--------------------------------------------------------------------------------
        static const int INVALID_MEMORY_POOL_INDEX = -1;    //!< メモリプールインデックス初期値
        static const int MEMORY_POOL_PARAM_NUM = 256;        //!< メモリプールパラメータの数
        //--------------------------------------------------------------------------------

        //--------------------------------------------------------------------------------
        //! コンストラクタ
        //--------------------------------------------------------------------------------
        MemoryPoolManager(){}
        //--------------------------------------------------------------------------------
        //! デストラクタ
        //--------------------------------------------------------------------------------
        virtual ~MemoryPoolManager(){}
        //--------------------------------------------------------------------------------
        //! アップデート
        //--------------------------------------------------------------------------------
        void updateMemoryPool(nn::audio::AudioRendererConfig& config)
            {
                for(int i = 0; i < MEMORY_POOL_PARAM_NUM; i++)
                {
                    MemoryPoolParam* pParam = &mMemoryPoolParam[i];
                    if(pParam->mMemoryPoolType._pMemoryPoolInfo != nullptr && nn::audio::GetMemoryPoolState(&pParam->mMemoryPoolType) == nn::audio::MemoryPoolType::State_Detached)
                    {
                        nn::audio::ReleaseMemoryPool(&config, &pParam->mMemoryPoolType);
                        pParam->mpBuffer = nullptr;
                    }
                }
            }
        //--------------------------------------------------------------------------------
        //! メモリプールの確保
        //--------------------------------------------------------------------------------
        int allocateMemoryPool(void* pBuffer, const size_t size, nn::audio::AudioRendererConfig& config)
            {
                for(int i = 0; i < MEMORY_POOL_PARAM_NUM; i++)
                {
                    MemoryPoolParam* pParam = &mMemoryPoolParam[i];
                    if(pParam->mpBuffer == pBuffer)
                    {
                        if(pParam->mMemoryPoolType._pMemoryPoolInfo != nullptr && nn::audio::GetMemoryPoolState(&pParam->mMemoryPoolType) == nn::audio::MemoryPoolType::State_RequestDetach)
                        {
                            // デタッチリクエスト中のときは確保状態にあるのでアタッチのみ行う
                            nn::audio::RequestAttachMemoryPool(&pParam->mMemoryPoolType);
                        }
                        pParam->mRefCount++;
                        return i;
                    }
                }
                for(int i = 0; i < MEMORY_POOL_PARAM_NUM; i++)
                {
                    MemoryPoolParam* pParam = &mMemoryPoolParam[i];
                    if(pParam->mMemoryPoolType._pMemoryPoolInfo == nullptr)
                    {
                        nn::audio::AcquireMemoryPool(&config, &pParam->mMemoryPoolType, pBuffer, size);
                        nn::audio::RequestAttachMemoryPool(&pParam->mMemoryPoolType);
                        pParam->mpBuffer = pBuffer;
                        pParam->mRefCount = 1;
                        return i;
                    }
                }
                return INVALID_MEMORY_POOL_INDEX;
            }
        //--------------------------------------------------------------------------------
        //! メモリプールの解放
        //--------------------------------------------------------------------------------
        void releaseMemoryPool(const int index)
            {
                if(index >= MEMORY_POOL_PARAM_NUM)
                {
                    return;
                }
                MemoryPoolParam* pParam = &mMemoryPoolParam[index];
                if(pParam->mRefCount == 0)
                {
                    return;
                }
                pParam->mRefCount--;
                if(pParam->mRefCount == 0)
                {
                    nn::audio::RequestDetachMemoryPool(&pParam->mMemoryPoolType);
                }
            }
        //--------------------------------------------------------------------------------
        //! メモリプールのアラインサイズを取得
        //--------------------------------------------------------------------------------
        size_t getAlignSize(const size_t size) const
            {
                if(size % nn::os::MemoryPageSize == 0)
                {
                    return size;
                }
                return (size + nn::os::MemoryPageSize) & ~(nn::os::MemoryPageSize - 1);
            }
    private:
        MemoryPoolParam        mMemoryPoolParam[MEMORY_POOL_PARAM_NUM];    //!< メモリプールパラメータ
    };
//--------------------------------------------------------------------------------
// ボイス管理構造体
//--------------------------------------------------------------------------------
    struct Voice
    {
        Voice()
                : memoryPoolIndex(MemoryPoolManager::INVALID_MEMORY_POOL_INDEX)
            {
                memset(&voiceType, 0, sizeof(voiceType));
                memset(&waveBuffer, 0, sizeof(waveBuffer));
            }
        nn::audio::VoiceType voiceType;
        nn::audio::WaveBuffer waveBuffer;
        int memoryPoolIndex;
    };
//--------------------------------------------------------------------------------
};

nn::mem::StandardAllocator g_Allocator;
NN_ALIGNAS(4096) char g_WorkBuffer[8 * 1024 * 1024];

void* Allocate(size_t size)
{
    return g_Allocator.Allocate(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    g_Allocator.Free(p);
}

void InitializeFileSystem()
{
    nn::fs::SetAllocator(Allocate, Deallocate);

    std::string mountPath = "";
    nnt::audio::util::GetMountPath(mountPath);

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost(g_MountName.c_str(), mountPath.c_str()));
}

void FinalizeFileSystem()
{
    nn::fs::Unmount("asset");
}

void InitializeHidDevices()
{
    nn::hid::InitializeDebugPad();

    //キーボードのキーを DebugPad のボタンに割り当てます。
    nn::settings::DebugPadKeyboardMap map;
    nn::settings::GetDebugPadKeyboardMap(&map);
    map.buttonA      = nn::hid::KeyboardKey::A::Index;
    map.buttonB      = nn::hid::KeyboardKey::B::Index;
    map.buttonX      = nn::hid::KeyboardKey::X::Index;
    map.buttonY      = nn::hid::KeyboardKey::Y::Index;
    map.buttonL      = nn::hid::KeyboardKey::L::Index;
    map.buttonR      = nn::hid::KeyboardKey::R::Index;
    map.buttonLeft  = nn::hid::KeyboardKey::LeftArrow::Index;
    map.buttonRight = nn::hid::KeyboardKey::RightArrow::Index;
    map.buttonUp     = nn::hid::KeyboardKey::UpArrow::Index;
    map.buttonDown  = nn::hid::KeyboardKey::DownArrow::Index;
    map.buttonStart = nn::hid::KeyboardKey::Space::Index;
    nn::settings::SetDebugPadKeyboardMap(map);
}


extern "C" void nnMain()
{
    NN_LOG("Cpu profiler demo test start\n");
    nn::audio::StartCpuProfiler();

    MemoryPoolManager memoryPoolManager;
    g_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));
    InitializeFileSystem();
    InitializeHidDevices();

    nn::hid::DebugPadState debugPadState[nn::hid::DebugPadStateCountMax];

    nn::audio::AudioRendererConfig config;
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = RenderRate;
    parameter.sampleCount = RenderCount;
    parameter.mixBufferCount = 2; // FinalMix(2)
    parameter.voiceCount = VoiceNum;
    parameter.subMixCount = 0;
    parameter.sinkCount = 1;
    parameter.effectCount = 0;
    parameter.performanceFrameCount = 0;
    NN_ABORT_UNLESS(nn::audio::IsValidAudioRendererParameter(parameter), "Invalid AudioRendererParameter specified.");

    int channelCount = 2;
    int8_t mainBus[2];
    mainBus[nn::audio::ChannelMapping_FrontLeft] = 0;
    mainBus[nn::audio::ChannelMapping_FrontRight] = 1;

    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer = g_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(workBuffer);

    nn::os::SystemEvent systemEvent;
    nn::audio::AudioRendererHandle handle;
    NN_ABORT_UNLESS(nn::audio::OpenAudioRenderer(&handle, &systemEvent, parameter, workBuffer, workBufferSize).IsSuccess(), "Failed to open AudioRenderer");

    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
    void* configBuffer = g_Allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(configBuffer);
    nn::audio::InitializeAudioRendererConfig(&config, parameter, configBuffer, configBufferSize);

    nn::audio::FinalMixType finalMix;
    nn::audio::AcquireFinalMix(&config, &finalMix, channelCount);

    nn::audio::DeviceSinkType deviceSink;
    nn::Result result = nn::audio::AddDeviceSink(&config, &deviceSink, &finalMix, mainBus, channelCount, "MainAudioOut");
    NN_ABORT_UNLESS(result.IsSuccess());

    result = nn::audio::RequestUpdateAudioRenderer(handle, &config);
    NN_ABORT_UNLESS(result.IsSuccess());
    result = nn::audio::StartAudioRenderer(handle);
    NN_ABORT_UNLESS(result.IsSuccess());


    void* dataSe[SeCount];
    size_t dataSeSize[SeCount];
    Voice voice[VoiceNum];

    // ADPCMデータ読み込み
    // メモリプールで一括管理を行う都合上、ヘッダと波形部分をまとめて確保する
    // 波形データは先頭から128バイトオフセットしたところに読み込む
    for (int i = 0; i < SeCount; ++i)
    {
        nn::fs::FileHandle handle;
        int64_t fileSize;
        nn::fs::OpenFile(&handle, g_SeFileNames[i], nn::fs::OpenMode_Read);
        nn::fs::GetFileSize(&fileSize, handle);
        size_t fileSizeAlign = memoryPoolManager.getAlignSize(fileSize);
        dataSe[i] = g_Allocator.Allocate(fileSizeAlign + 128, nn::audio::BufferAlignSize);
        memset(dataSe[i], 0, fileSizeAlign + 128);
        dataSeSize[i] = fileSize - sizeof(nn::audio::AdpcmHeaderSize);

        uint8_t adpcmheader[nn::audio::AdpcmHeaderSize];
        nn::fs::ReadFile(handle, 0, adpcmheader, sizeof(adpcmheader));
        nn::fs::ReadFile(handle, 0, (char*)dataSe[i] + 128, fileSize - sizeof(adpcmheader));
        nn::fs::CloseFile(handle);
        nn::audio::ParseAdpcmHeader((nn::audio::AdpcmHeaderInfo*)dataSe[i], adpcmheader, sizeof(adpcmheader));
    }

    int64_t prevPadSamplingNumber = 0;
    int waitCount = 0;    // 適度にウェイトを入れるためのカウンタ
    for (;;)
    {
        nn::hid::GetDebugPadStates(debugPadState, nn::hid::DebugPadStateCountMax);
        int64_t padSamplingCount = debugPadState[0].samplingNumber - prevPadSamplingNumber;
        if(padSamplingCount >= nn::hid::DebugPadStateCountMax)
        {
            padSamplingCount = nn::hid::DebugPadStateCountMax - 1;
        }
        prevPadSamplingNumber = debugPadState[0].samplingNumber;
        nn::hid::DebugPadButtonSet debugPadButtonDown(debugPadState[0].buttons & ~debugPadState[padSamplingCount].buttons);

        if(debugPadButtonDown.Test< ::nn::hid::DebugPadButton::Start >())
        {
            break;
        }

        systemEvent.Wait();

        // ボイス更新
        for(int v = 0; v < VoiceNum; v++)
        {
            if(voice[v].voiceType._pVoiceInfo != nullptr && nn::audio::GetReleasedWaveBuffer(&voice[v].voiceType) != nullptr)
            {
                nn::audio::ReleaseVoiceSlot(&config, &voice[v].voiceType);
                memoryPoolManager.releaseMemoryPool(voice[v].memoryPoolIndex);
                voice[v].memoryPoolIndex = MemoryPoolManager::INVALID_MEMORY_POOL_INDEX;
            }
        }

        if(waitCount == 0)
        {
            // waitCountが0になったらボイスを再生
            for(int i = 0; i < 5; i++)
            {
                for(int v = 0; v < VoiceNum; v++)
                {
                    if(voice[v].voiceType._pVoiceInfo == nullptr)
                    {
                        int dataIndex = rand() % SeCount;    // どれかをランダムで再生
                        nn::audio::AdpcmHeaderInfo* headerInfo = (nn::audio::AdpcmHeaderInfo*)dataSe[dataIndex];
                        voice[v].waveBuffer.buffer = (char*)dataSe[dataIndex] + 128;    // ヘッダ部分をオフセットしたところを指定
                        voice[v].waveBuffer.size = dataSeSize[dataIndex];
                        voice[v].waveBuffer.startSampleOffset = 0;
                        voice[v].waveBuffer.endSampleOffset = headerInfo->sampleCount;
                        voice[v].waveBuffer.loop = false;
                        voice[v].waveBuffer.isEndOfStream = false;
                        voice[v].waveBuffer.pContext = &headerInfo->loopContext;
                        voice[v].waveBuffer.contextSize = sizeof(nn::audio::AdpcmContext);

                        NN_LOG("%s buf=%x size=%x smpl=%d\n", g_SeFileNames[dataIndex], voice[v].waveBuffer.buffer, voice[v].waveBuffer.size, voice[v].waveBuffer.endSampleOffset);

                        voice[v].memoryPoolIndex = memoryPoolManager.allocateMemoryPool(dataSe[dataIndex], memoryPoolManager.getAlignSize(dataSeSize[dataIndex]), config);
                        nn::audio::AcquireVoiceSlot(&config, &voice[v].voiceType, 32000, 1, nn::audio::SampleFormat_Adpcm, nn::audio::VoiceType::PriorityHighest, &headerInfo->parameter, sizeof(nn::audio::AdpcmParameter));
                        nn::audio::SetVoiceDestination(&config, &voice[v].voiceType, &finalMix);
                        nn::audio::SetVoicePlayState(&voice[v].voiceType, nn::audio::VoiceType::PlayState_Play);
                        nn::audio::SetVoiceMixVolume(&voice[v].voiceType, &finalMix, 0.707f / 2, 0, mainBus[nn::audio::ChannelMapping_FrontLeft]);
                        nn::audio::SetVoiceMixVolume(&voice[v].voiceType, &finalMix, 0.707f / 2, 0, mainBus[nn::audio::ChannelMapping_FrontRight]);
                        nn::audio::AppendWaveBuffer(&voice[v].voiceType, &voice[v].waveBuffer);
                        break;
                    }
                }
            }
            waitCount = rand() % 20 + 20; // 適度に待ち時間を入れる
        }
        waitCount--;
        nn::audio::RequestUpdateAudioRenderer(handle, &config);
        memoryPoolManager.updateMemoryPool(config);
    }

    // レンダリングを終了します。
    nn::audio::StopAudioRenderer(handle);
    nn::audio::CloseAudioRenderer(handle);

    for (int i = 0; i < SeCount; ++i)
    {
        if (dataSe[i])
        {
            g_Allocator.Free(dataSe[i]);
            dataSe[i] = nullptr;
        }
    }
    if (configBuffer)
    {
        g_Allocator.Free(configBuffer);
        configBuffer = nullptr;
    }
    if (workBuffer)
    {
        g_Allocator.Free(workBuffer);
        workBuffer = nullptr;
    }

    FinalizeFileSystem();
    nn::audio::StopCpuProfiler();
}  // NOLINT(readability/fn_size)
