﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/audio.h>
#include <nn/mem.h>
#include <nn/os.h>
#include <nn/nn_Log.h>

namespace {

const nn::audio::AudioRendererParameter GetDefaultAudioRendererParameter()
{
    nn::audio::AudioRendererParameter param;
    nn::audio::InitializeAudioRendererParameter(&param);
    param.sampleRate = 48000;
    param.sampleCount = 240;
    param.mixBufferCount = 16;
    param.subMixCount = 0;
    param.voiceCount = 24;
    param.sinkCount = 1;
    param.effectCount = 0;
    param.performanceFrameCount = 0;
    return param;
}

NN_ALIGNAS(4096) char g_RendererBuffer[1024 * 1024];
NN_AUDIO_ALIGNAS_AUDIO_OUT_BUFFER_ALIGN char g_OutBuffer[nn::audio::AudioOutBuffer::SizeGranularity];

}

TEST(AudioRendererAudioOut, OpenRendererOpenOutCloseOutCloseRenderer)
{
    nn::mem::StandardAllocator allocator(g_RendererBuffer, sizeof(g_RendererBuffer));

    nn::audio::AudioRendererHandle handle;
    void* workBuffer;
    nn::audio::AudioRendererParameter rendererParameter = GetDefaultAudioRendererParameter();
    nn::audio::AudioOut out;
    nn::audio::AudioOutParameter outParameter;
    nn::audio::InitializeAudioOutParameter(&outParameter);

    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(rendererParameter);
    workBuffer = allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&handle, rendererParameter, workBuffer, workBufferSize));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(&out, outParameter));

    nn::audio::CloseAudioOut(&out);
    nn::audio::CloseAudioRenderer(handle);

    allocator.Free(workBuffer);
}

TEST(AudioRendererAudioOut, OpenRendererOpenOutCloseRendererCloseOut)
{
    nn::mem::StandardAllocator allocator(g_RendererBuffer, sizeof(g_RendererBuffer));

    nn::audio::AudioRendererHandle handle;
    void* workBuffer;
    nn::audio::AudioRendererParameter rendererParameter = GetDefaultAudioRendererParameter();
    nn::audio::AudioOut out;
    nn::audio::AudioOutParameter outParameter;
    nn::audio::InitializeAudioOutParameter(&outParameter);

    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(rendererParameter);
    workBuffer = allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&handle, rendererParameter, workBuffer, workBufferSize));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(&out, outParameter));

    nn::audio::CloseAudioRenderer(handle);
    nn::audio::CloseAudioOut(&out);

    allocator.Free(workBuffer);
}

TEST(AudioRendererAudioOut, OpenOutOpenRendererCloseRendererCloseOut)
{
    nn::mem::StandardAllocator allocator(g_RendererBuffer, sizeof(g_RendererBuffer));

    nn::audio::AudioRendererHandle handle;
    void* workBuffer;
    nn::audio::AudioRendererParameter rendererParameter = GetDefaultAudioRendererParameter();
    nn::audio::AudioOut out;
    nn::audio::AudioOutParameter outParameter;
    nn::audio::InitializeAudioOutParameter(&outParameter);

    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(rendererParameter);
    workBuffer = allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(&out, outParameter));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&handle, rendererParameter, workBuffer, workBufferSize));

    nn::audio::CloseAudioRenderer(handle);
    nn::audio::CloseAudioOut(&out);

    allocator.Free(workBuffer);
}

TEST(AudioRendererAudioOut, OpenOutOpenRendererCloseOutCloseRenderer)
{
    nn::mem::StandardAllocator allocator(g_RendererBuffer, sizeof(g_RendererBuffer));

    nn::audio::AudioRendererHandle handle;
    void* workBuffer;
    nn::audio::AudioRendererParameter rendererParameter = GetDefaultAudioRendererParameter();
    nn::audio::AudioOut out;
    nn::audio::AudioOutParameter outParameter;
    nn::audio::InitializeAudioOutParameter(&outParameter);

    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(rendererParameter);
    workBuffer = allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(&out, outParameter));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&handle, rendererParameter, workBuffer, workBufferSize));

    nn::audio::CloseAudioOut(&out);
    nn::audio::CloseAudioRenderer(handle);
    allocator.Free(workBuffer);
}

TEST(AudioRendererAudioOut, AppendSimultaneousBuffers)
{
    const int rendererCount = 1; //nn::audio::AudioRendererCountMax;

    nn::mem::StandardAllocator allocator(g_RendererBuffer, sizeof(g_RendererBuffer));

    nn::audio::AudioOutParameter audioOutParameter;
    nn::audio::InitializeAudioOutParameter(&audioOutParameter);
    nn::audio::AudioOut audioOut;

    nn::audio::AudioRendererHandle handles[rendererCount];
    nn::audio::AudioRendererConfig configs[rendererCount];
    nn::audio::VoiceType voices[rendererCount];
    nn::audio::FinalMixType finalMixes[rendererCount];
    nn::audio::WaveBuffer waveBuffers[rendererCount];
    void* workBuffers[rendererCount];
    nn::os::SystemEvent systemEventRenderer;
    nn::os::SystemEvent systemEventOut;
    void* configBuffers[rendererCount];

    auto audioRendererParameter = GetDefaultAudioRendererParameter();

    for (int i = 0; i < rendererCount; ++i)
    {
        size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(audioRendererParameter);
        workBuffers[i] = allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&handles[i], &systemEventRenderer, audioRendererParameter, workBuffers[i], workBufferSize));

        size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(audioRendererParameter);
        configBuffers[i] = allocator.Allocate(configBufferSize, nn::audio::BufferAlignSize);
        nn::audio::InitializeAudioRendererConfig(&configs[i], audioRendererParameter, configBuffers[i], configBufferSize);
    }

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(&audioOut, &systemEventOut, audioOutParameter));
    EXPECT_NE(systemEventOut.GetReadableHandle(), nn::os::InvalidNativeHandle);
    EXPECT_EQ(nn::audio::GetReleasedAudioOutBuffer(&audioOut), nullptr);

    for (int i = 0; i < rendererCount; ++i)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioRenderer(handles[i]));
    }

    nn::audio::MemoryPoolType memoryPools[rendererCount];
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t pcmBuffer[rendererCount][nn::audio::MemoryPoolType::SizeGranularity / sizeof(int16_t)];

    for (int i = 0; i < rendererCount; ++i)
    {
        EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&configs[i], &voices[i], 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
        EXPECT_TRUE(nn::audio::AcquireFinalMix(&configs[i], &finalMixes[i], 1));
        nn::audio::SetVoiceDestination(&configs[i], &voices[i], &finalMixes[i]);
        nn::audio::SetVoicePlayState(&voices[i], nn::audio::VoiceType::PlayState_Play);

        EXPECT_TRUE(nn::audio::AcquireMemoryPool(&configs[i], &memoryPools[i], pcmBuffer[i], sizeof(pcmBuffer[i])));
        EXPECT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPools[i]));

        waveBuffers[i].buffer = pcmBuffer[i];
        waveBuffers[i].size = audioRendererParameter.sampleCount * sizeof(pcmBuffer[i][0]);
        waveBuffers[i].startSampleOffset = 0;
        waveBuffers[i].endSampleOffset = static_cast<int32_t>(waveBuffers[i].size / sizeof(pcmBuffer[i][0]));
        waveBuffers[i].loop = 0;
        waveBuffers[i].isEndOfStream = true;

        nn::audio::AppendWaveBuffer(&voices[i], &waveBuffers[i]);
        EXPECT_TRUE(nn::audio::GetReleasedWaveBuffer(&voices[i]) == nullptr);
        systemEventRenderer.Wait();
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(handles[i], &configs[i]));
    }

    nn::audio::AudioOutBuffer audioOutBuffer;
    nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_OutBuffer, sizeof(g_OutBuffer), sizeof(g_OutBuffer));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioOut(&audioOut));
    nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer);
    EXPECT_EQ(nn::audio::GetReleasedAudioOutBuffer(&audioOut), nullptr);
    nn::audio::AudioOutBuffer* pReleased = nullptr;

    systemEventOut.Wait();

    for (int i = 0; i < rendererCount; ++i)
    {
        const nn::audio::WaveBuffer* releasedWaveBuffer = nullptr;
        while ((releasedWaveBuffer = nn::audio::GetReleasedWaveBuffer(&voices[i])) == nullptr)
        {
            systemEventRenderer.Wait();
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(handles[i], &configs[i]));
        }
        EXPECT_TRUE(releasedWaveBuffer == &waveBuffers[i]);
        EXPECT_TRUE(nn::audio::GetReleasedWaveBuffer(&voices[i]) == nullptr);

        EXPECT_TRUE(nn::audio::GetVoicePlayedSampleCount(&voices[i]) == 0);
    }

    pReleased = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
    EXPECT_EQ(pReleased, &audioOutBuffer);
    pReleased = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
    EXPECT_EQ(nn::audio::GetReleasedAudioOutBuffer(&audioOut), nullptr);

    for (int i = 0; i < rendererCount; ++i)
    {
        nn::audio::StopAudioRenderer(handles[i]);
    }

    for (int i = 0; i < rendererCount; ++i)
    {
        nn::audio::CloseAudioRenderer(handles[i]);
        allocator.Free(workBuffers[i]);
    }

    if (nn::audio::GetAudioOutState(&audioOut) == nn::audio::AudioOutState_Started)
    {
        nn::audio::StopAudioOut(&audioOut);
    }
    nn::audio::CloseAudioOut(&audioOut);

}

TEST(AudioRendererAudioOut, AppendSimultaneousBuffersMultiThread)
{
    static NN_OS_ALIGNAS_THREAD_STACK uint8_t audioRendererThreadStack[32768], audioOutThreadStack[32768];
    nn::os::ThreadType audioRendererThread, audioOutThread;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&audioRendererThread,
        [](void* args)
        {
            NN_UNUSED(args);
            const int rendererCount = 1; //nn::audio::AudioRendererCountMax;

            nn::mem::StandardAllocator allocator(g_RendererBuffer, sizeof(g_RendererBuffer));

            nn::audio::AudioRendererHandle handles[rendererCount];
            nn::audio::AudioRendererConfig configs[rendererCount];
            nn::audio::VoiceType voices[rendererCount];
            nn::audio::FinalMixType finalMixes[rendererCount];
            nn::audio::WaveBuffer waveBuffers[rendererCount];
            void* workBuffers[rendererCount];
            nn::os::SystemEvent systemEventRenderer;
            void* configBuffers[rendererCount];

            auto audioRendererParameter = GetDefaultAudioRendererParameter();

            for (int i = 0; i < rendererCount; ++i)
            {
                size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(audioRendererParameter);
                workBuffers[i] = allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
                NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&handles[i], &systemEventRenderer, audioRendererParameter, workBuffers[i], workBufferSize));

                size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(audioRendererParameter);
                configBuffers[i] = allocator.Allocate(configBufferSize, nn::audio::BufferAlignSize);
                nn::audio::InitializeAudioRendererConfig(&configs[i], audioRendererParameter, configBuffers[i], configBufferSize);
            }

            for (int i = 0; i < rendererCount; ++i)
            {
                NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioRenderer(handles[i]));
            }

            nn::audio::MemoryPoolType memoryPools[rendererCount];
            NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t pcmBuffer[rendererCount][nn::audio::MemoryPoolType::SizeGranularity];

            for (int i = 0; i < rendererCount; ++i)
            {
                EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&configs[i], &voices[i], 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
                EXPECT_TRUE(nn::audio::AcquireFinalMix(&configs[i], &finalMixes[i], 1));
                nn::audio::SetVoiceDestination(&configs[i], &voices[i], &finalMixes[i]);
                nn::audio::SetVoicePlayState(&voices[i], nn::audio::VoiceType::PlayState_Play);

                EXPECT_TRUE(nn::audio::AcquireMemoryPool(&configs[i], &memoryPools[i], pcmBuffer[i], sizeof(pcmBuffer[i])));
                EXPECT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPools[i]));

                waveBuffers[i].buffer = pcmBuffer[i];
                waveBuffers[i].size = audioRendererParameter.sampleCount * sizeof(pcmBuffer[i][0]);
                waveBuffers[i].startSampleOffset = 0;
                waveBuffers[i].endSampleOffset = static_cast<int32_t>(waveBuffers[i].size / sizeof(pcmBuffer[i][0]));
                waveBuffers[i].loop = 0;
                waveBuffers[i].isEndOfStream = true;

                nn::audio::AppendWaveBuffer(&voices[i], &waveBuffers[i]);
                EXPECT_TRUE(nn::audio::GetReleasedWaveBuffer(&voices[i]) == nullptr);
                NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(handles[i], &configs[i]));
                systemEventRenderer.Wait();
            }

            // Wait for playback done (2048 samples, 43 ms)
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

            for (int i = 0; i < rendererCount; ++i)
            {
                NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(handles[i], &configs[i]));
                systemEventRenderer.Wait();

                const auto p = nn::audio::GetReleasedWaveBuffer(&voices[i]);
                EXPECT_TRUE(p == &waveBuffers[i]);
                EXPECT_TRUE(nn::audio::GetReleasedWaveBuffer(&voices[i]) == nullptr);

                EXPECT_TRUE(nn::audio::GetVoicePlayedSampleCount(&voices[i]) == 0);
            }

            for (int i = 0; i < rendererCount; ++i)
            {
                nn::audio::StopAudioRenderer(handles[i]);
            }

            for (int i = 0; i < rendererCount; ++i)
            {
                nn::audio::CloseAudioRenderer(handles[i]);
                allocator.Free(workBuffers[i]);
            }
        }, nullptr, audioRendererThreadStack, sizeof(audioRendererThreadStack), nn::os::DefaultThreadPriority));

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&audioOutThread,
        [](void* args)
        {
            NN_UNUSED(args);
            nn::audio::AudioOutParameter audioOutParameter;
            nn::audio::InitializeAudioOutParameter(&audioOutParameter);
            nn::audio::AudioOut audioOut;

            nn::os::SystemEvent systemEventOut;

            NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(&audioOut, &systemEventOut, audioOutParameter));
            EXPECT_NE(systemEventOut.GetReadableHandle(), nn::os::InvalidNativeHandle);
            EXPECT_EQ(nn::audio::GetReleasedAudioOutBuffer(&audioOut), nullptr);

            nn::audio::AudioOutBuffer audioOutBuffer;
            nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_OutBuffer, sizeof(g_OutBuffer), sizeof(g_OutBuffer));
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioOut(&audioOut));
            nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer);
            EXPECT_EQ(nn::audio::GetReleasedAudioOutBuffer(&audioOut), nullptr);
            nn::audio::AudioOutBuffer* pReleased = nullptr;

            systemEventOut.Wait();

            pReleased = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
            EXPECT_EQ(pReleased, &audioOutBuffer);
            pReleased = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
            EXPECT_EQ(nn::audio::GetReleasedAudioOutBuffer(&audioOut), nullptr);

            if (nn::audio::GetAudioOutState(&audioOut) == nn::audio::AudioOutState_Started)
            {
                nn::audio::StopAudioOut(&audioOut);
            }
            nn::audio::CloseAudioOut(&audioOut);
        }, nullptr, audioOutThreadStack, sizeof(audioOutThreadStack), nn::os::DefaultThreadPriority));

    nn::os::StartThread(&audioRendererThread);
    nn::os::StartThread(&audioOutThread);
    nn::os::WaitThread(&audioRendererThread);
    nn::os::WaitThread(&audioOutThread);
    nn::os::DestroyThread(&audioRendererThread);
    nn::os::DestroyThread(&audioOutThread);
} // NOLINT(impl/function_size)
