﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>  // max_element, min_element
#include <cstdlib>  // free, malloc
#include <numeric>  // accumulate

#include <nnt.h>

#include <nn/nn_Log.h>
#include <nn/audio.h>
#include <nn/mem.h>
#include <nn/os.h>

TEST(AudioSystemEventPeriod, AudioRenderer)
{
    auto allocatorSize = 1 * 1024 * 1024;
    auto allocatorBuffer = std::malloc(allocatorSize);
    EXPECT_TRUE(allocatorBuffer != nullptr);
    nn::mem::StandardAllocator allocator;
    allocator.Initialize(allocatorBuffer, allocatorSize);

    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    EXPECT_TRUE(nn::audio::IsValidAudioRendererParameter(parameter));
    auto rendererBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    auto rendererBuffer = allocator.Allocate(rendererBufferSize, nn::os::MemoryPageSize);
    ASSERT_TRUE(rendererBuffer != nullptr);
    nn::audio::AudioRendererHandle handle;
    nn::os::SystemEvent systemEvent;
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&handle, &systemEvent, parameter, rendererBuffer, rendererBufferSize));
    auto configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
    auto configBuffer = allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
    ASSERT_TRUE(configBuffer != nullptr);
    nn::audio::AudioRendererConfig config;
    nn::audio::InitializeAudioRendererConfig(&config, parameter, configBuffer, configBufferSize);
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioRenderer(handle));

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    const auto Count = 1024;
    nn::TimeSpan periods[Count] = { 0 };
    systemEvent.Wait();
    auto begin = nn::os::GetSystemTick().ToTimeSpan();
    systemEvent.Wait();
    for (auto i = 0; i < Count; ++i)
    {
        auto end = nn::os::GetSystemTick().ToTimeSpan();
        periods[i] = end - begin;
        begin = end;
        systemEvent.Wait();
    }

    nn::audio::StopAudioRenderer(handle);
    nn::audio::CloseAudioRenderer(handle);
    nn::os::DestroySystemEvent(systemEvent.GetBase());
    allocator.Free(configBuffer);
    allocator.Free(rendererBuffer);

    auto max = std::max_element(periods, periods + Count)->GetMicroSeconds();
    auto min = std::min_element(periods, periods + Count)->GetMicroSeconds();
    auto avg = std::accumulate(periods, periods + Count, nn::TimeSpan(0)).GetMicroSeconds() / Count;

    NN_LOG("##teamcity[buildStatisticValue key='AudioRendererSystemEventPeriodMax(usec)' value='%lld']\n", max);
    NN_LOG("##teamcity[buildStatisticValue key='AudioRendererSystemEventPeriodMin(usec)' value='%lld']\n", min);
    NN_LOG("##teamcity[buildStatisticValue key='AudioRendererSystemEventPeriodAvg(usec)' value='%lld']\n", avg);

    allocator.Finalize();
    std::free(allocatorBuffer);
}
