﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>  // max_element, min_element
#include <cstdlib>  // free, malloc
#include <numeric>  // accumulate

#include <nnt.h>

#include <nn/nn_Log.h>
#include <nn/audio.h>
#include <nn/audio/audio_AudioInApi.private.h>
#include <nn/mem.h>
#include <nn/os.h>

TEST(AudioSystemEventPeriod, AudioIn)
{
    nn::audio::SetAudioInEnabled(true);

    auto allocatorSize = 1 * 1024 * 1024;
    auto allocatorBuffer = std::malloc(allocatorSize);
    EXPECT_TRUE(allocatorBuffer != nullptr);
    nn::mem::StandardAllocator allocator;
    allocator.Initialize(allocatorBuffer, allocatorSize);

    for (auto frame = 1; frame <= 3; ++frame)
    {
        nn::audio::AudioIn audioIn;
        nn::audio::AudioInParameter parameter;
        nn::audio::InitializeAudioInParameter(&parameter);
        nn::os::SystemEvent systemEvent;
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenDefaultAudioIn(&audioIn, &systemEvent, parameter));
        const auto ChannelCount = nn::audio::GetAudioInChannelCount(&audioIn);
        const auto SampleCount = nn::audio::GetAudioInSampleRate(&audioIn);
        const auto SampleFormat = nn::audio::GetAudioInSampleFormat(&audioIn);
        ASSERT_TRUE(SampleFormat == nn::audio::SampleFormat_PcmInt16);
        const auto FrameSampleCount = SampleCount / 200 * frame;
        const size_t DataSize = FrameSampleCount * ChannelCount * nn::audio::GetSampleByteSize(SampleFormat);
        const size_t BufferSize = nn::util::align_up(DataSize, nn::audio::AudioInBuffer::SizeGranularity);
        const auto BufferCount = 8;
        nn::audio::AudioInBuffer audioInBuffer[BufferCount];
        void* inBuffer[BufferCount];
        for (int i = 0; i < BufferCount; ++i)
        {
            inBuffer[i] = allocator.Allocate(BufferSize, nn::audio::AudioInBuffer::AddressAlignment);
            ASSERT_TRUE(inBuffer[i] != nullptr);
            nn::audio::SetAudioInBufferInfo(&audioInBuffer[i], inBuffer[i], BufferSize, DataSize);
            nn::audio::AppendAudioInBuffer(&audioIn, &audioInBuffer[i]);
        }
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioIn(&audioIn));

        nn::TimeSpan periods[1024];
        const auto Count = static_cast<int>(1024 / frame);
        systemEvent.Wait();
        auto begin = nn::os::GetSystemTick().ToTimeSpan();
        systemEvent.Wait();
        for (auto i = 0; i < Count; ++i)
        {
            auto end = nn::os::GetSystemTick().ToTimeSpan();
            periods[i] = end - begin;
            begin = end;
            auto pAudioInBuffer = nn::audio::GetReleasedAudioInBuffer(&audioIn);
            while (pAudioInBuffer)
            {
//                 auto pInBuffer = nn::audio::GetAudioInBufferDataPointer(pAudioInBuffer);
                auto inSize = nn::audio::GetAudioInBufferDataSize(pAudioInBuffer);
                ASSERT_TRUE(inSize == FrameSampleCount * ChannelCount * nn::audio::GetSampleByteSize(SampleFormat));
                nn::audio::AppendAudioInBuffer(&audioIn, pAudioInBuffer);
                pAudioInBuffer = nn::audio::GetReleasedAudioInBuffer(&audioIn);
            }
            systemEvent.Wait();
        }

        nn::audio::StopAudioIn(&audioIn);
        nn::audio::CloseAudioIn(&audioIn);
        nn::os::DestroySystemEvent(systemEvent.GetBase());
        for (auto i = 0; i < BufferCount; ++i)
        {
            allocator.Free(inBuffer[i]);
        }

        auto max = std::max_element(periods, periods + Count)->GetMicroSeconds();
        auto min = std::min_element(periods, periods + Count)->GetMicroSeconds();
        auto avg = std::accumulate(periods, periods + Count, nn::TimeSpan(0)).GetMicroSeconds() / Count;

        NN_LOG("##teamcity[buildStatisticValue key='AudioInSystemEventPeriod%02dMax(usec)' value='%lld']\n", 5 * frame, max);
        NN_LOG("##teamcity[buildStatisticValue key='AudioInSystemEventPeriod%02dMin(usec)' value='%lld']\n", 5 * frame, min);
        NN_LOG("##teamcity[buildStatisticValue key='AudioInSystemEventPeriod%02dAvg(usec)' value='%lld']\n", 5 * frame, avg);
    }

    allocator.Finalize();
    std::free(allocatorBuffer);
}
