﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <limits>
#include <random>
#include <utility>

#include "../../Programs/Eris/Sources/Libraries/audio/dsp/audio_Depop.h"

namespace {
void fillRandom(int32_t* buffer, int32_t sampleCount)
{
    std::mt19937 mt(0);
    for (auto i = 0; i < sampleCount; ++i)
    {
        const auto a = std::numeric_limits<uint16_t>::max();
        const auto b = std::numeric_limits<int16_t>::min();
        buffer[i] = (mt() % a) + b;
    }
}
}

TEST(DspDepop, CompareX1AndX2)
{
    // depop 開始サンプル値
    const int32_t startSamples[] = { 32767, 12345, 0, -12345, -32768 };

    // テストパラメータ
    const std::pair<int, int> parameters[] =
        {
            {240, 0x7b29}, // 48k
            {160, 0x78cb}  // 32k
        };

    // 波形比較誤差許容範囲
    // ApplyDepopMix1 と ApplyDepopMix2 の計算方法の違いによって、結果に誤差が生じる
    // どの範囲までを許容するかの許容量
    const int absError = 11;

    for (const auto startSample : startSamples)
    {
        for (const auto& param : parameters)
        {
            const int sampleCount = param.first;
            const int32_t factor = param.second;

            // prepare
            int32_t* output0 = new int32_t[sampleCount];
            int32_t* output1 = new int32_t[sampleCount];
            ASSERT_NE(output0, nullptr);
            ASSERT_NE(output1, nullptr);
            fillRandom(output0, sampleCount);
            memcpy(output1, output0, sizeof(int32_t) * sampleCount);
            auto endValue = output0[sampleCount - 1];

            // test
            auto lastValue1 = nn::audio::dsp::ApplyDepopMix1(output0, startSample, factor, sampleCount);
            auto lastValue2 = nn::audio::dsp::ApplyDepopMix2(output1, startSample, factor, sampleCount);
            if (std::abs(startSample) < std::numeric_limits<int16_t>::max())
            {
                // startSample が in16_t の範囲の場合、 depop による加算量は 0 に収束しているはず。
                EXPECT_EQ(lastValue1, 0);
                EXPECT_EQ(lastValue2, 0);

                // 0 収束しているので、バッファの終端のサンプルは値に変化がないはず
                EXPECT_EQ(endValue, output0[sampleCount - 1]);
                EXPECT_EQ(endValue, output1[sampleCount - 1]);
            }
            else
            {
                // 1 audio frame で終息しない場合のチェック
                EXPECT_NEAR(lastValue1, lastValue2, absError);
            }

            for (auto i = 0; i < sampleCount; ++i)
            {
                ASSERT_NEAR(output0[i], output1[i], absError) << "index: " << i;
            }

            delete[] output0;
            delete[] output1;
        }
    }
}

#if defined(__ARM_NEON__) || defined(__ARM_NEON)
namespace generic {

#include "../../Programs/Eris/Sources/Libraries/audio/dsp/detail/audio_Depop.generic.h"

}  // namespace negeric

namespace neon {

#include "../../Programs/Eris/Sources/Libraries/audio/dsp/detail/audio_Depop.neon.h"

}  // namespace neon

TEST(DspDepop, CompareGenericAndNeon)
{
    const int32_t startSamples[] = { 32767, 12345, 0, -12345, -32768 };
    const std::pair<int, int> parameters[] =
        {
            {240, 0x7b29}, // 48k
            {160, 0x78cb}  // 32k
        };

    for (const auto startSample : startSamples)
    {
        for (const auto& param : parameters)
        {
            const int sampleCount = param.first;
            const int32_t factor = param.second;

            // prepare
            int32_t* output0 = new int32_t[sampleCount];
            int32_t* output1 = new int32_t[sampleCount];
            ASSERT_NE(output0, nullptr);
            ASSERT_NE(output1, nullptr);
            fillRandom(output0, sampleCount);
            memcpy(output1, output0, sizeof(int32_t) * sampleCount);

            // test
            ASSERT_EQ(generic::nn::audio::dsp::detail::ApplyDepopMix2(output0, startSample, factor, sampleCount),
                      neon::nn::audio::dsp::detail::ApplyDepopMix2(output1, startSample, factor, sampleCount));
            for (auto i = 0; i < sampleCount; ++i)
            {
                ASSERT_EQ(output0[i], output1[i]) << "index: " << i;
            }

            delete[] output0;
            delete[] output1;
        }
    }
}

#endif  // defined(__ARM_NEON__) || defined(__ARM_NEON)
