﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <cstdlib>
#include <cmath>
#include <string>
#include <mutex>

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/nn_TimeSpan.h>
#include <nn/audio.h>

#include <nn/nn_Log.h>

#include <nn/audio/audio_AudioInApi.private.h>
#include <nn/audio/audio_DeviceApi-spec.nx.h>
#include <nn/audio/audio_FinalOutputRecorderApi.h>
#include <nn/audio/audio_FinalOutputRecorderTypes.h>

#include <nnt/audioUtil/testAudio_Util.h>
#include <nnt/audioUtil/testAudio_Constants.h>

#include <nn/fatal/fatal_Api.h>

namespace {
    const int MaxAudioOutSessions = nn::audio::AudioOutCountMax;

    //SIGLO-77050: nn::audio::AudioInCountMax = 4, but Ahub AudioIn max is 2.
    //TODO: Need a way to get AhubIn max and UacIn max
    const int MaxAudioInSessions = 2;
    const int MaxAudioRendererSessions = nn::audio::AudioRendererCountMax;
    const int MaxFinalOutputRecorderSessions = nn::audio::FinalOutputRecorderCountMax;

    const int FailureAudioOutSessions = MaxAudioOutSessions + 1;
    const int FailureAudioInSessions = MaxAudioInSessions + 1;
    const int FailureAudioRendererSessions = MaxAudioRendererSessions + 1;
    const int FailureFinalOutputRecorderSessions = MaxFinalOutputRecorderSessions + 1;

    const int RenderRate = 32000;
    const int RenderCount = (RenderRate / 200);

    NN_ALIGNAS(4096) char m_WorkBuffer[512 * 1024];
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char m_WaveBufferPoolMemory[512 * 1024];

    nn::mem::StandardAllocator m_Allocator;
    nn::mem::StandardAllocator m_WaveBufferAllocator;

    void CheckAudioInputOutputProperties(nn::audio::AudioIn* pAudioIn, nn::audio::AudioOut* pAudioOut)
    {
        NN_ABORT_UNLESS(
            nn::audio::GetAudioInChannelCount(pAudioIn) == nn::audio::GetAudioOutChannelCount(pAudioOut),
            "AudioIn and AudioOut are supposed to have same channel count in this sample."
        );
        NN_ABORT_UNLESS(
            nn::audio::GetAudioInSampleRate(pAudioIn) == nn::audio::GetAudioOutSampleRate(pAudioOut),
            "AudioIn and AudioOut are supposed to have same sample rate in this sample."
        );
        NN_ABORT_UNLESS(
            nn::audio::GetAudioInSampleFormat(pAudioIn) == nn::audio::GetAudioOutSampleFormat(pAudioOut),
            "AudioIn and AudioOut are supposed to have same sample format in this sample."
        );
    }
} // anonymous namespace

TEST(OpenAndCloseMaxSessions, Success)
{
    m_Allocator.Initialize(m_WorkBuffer, sizeof(m_WorkBuffer));
    m_WaveBufferAllocator.Initialize(m_WaveBufferPoolMemory, sizeof(m_WaveBufferPoolMemory));

    // Setup AudioRenderer
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = RenderRate;
    parameter.sampleCount = RenderCount;
    parameter.mixBufferCount = 2;
    parameter.voiceCount = 1;
    parameter.subMixCount = 0;
    parameter.sinkCount = 1;
    parameter.effectCount = 0;
    parameter.performanceFrameCount = 0;

    const int channelCount = 2;
    int8_t mainBus[channelCount];
    mainBus[nn::audio::ChannelMapping_FrontLeft] = 0;
    mainBus[nn::audio::ChannelMapping_FrontRight] = 1;

    NN_ABORT_UNLESS(
        nn::audio::IsValidAudioRendererParameter(parameter),
        "Invalid AudioRendererParameter specified."
    );

    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer[MaxAudioRendererSessions];
    nn::os::SystemEvent systemEvent[MaxAudioRendererSessions];
    nn::audio::AudioRendererHandle handle[MaxAudioRendererSessions];
    for (int i = 0; i < MaxAudioRendererSessions; i++)
    {
        workBuffer[i] = m_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
        NN_ABORT_UNLESS_NOT_NULL(workBuffer[i]);

        NN_ABORT_UNLESS(
            nn::audio::OpenAudioRenderer(&handle[i], &systemEvent[i], parameter, workBuffer[i], workBufferSize).IsSuccess(),
            "Failed to open AudioRenderer"
        );
    }

    // Setup FinalMix and routing
    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
    void* configBuffer = m_Allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(configBuffer);
    nn::audio::AudioRendererConfig config;
    nn::audio::InitializeAudioRendererConfig(&config, parameter, configBuffer, configBufferSize);

    nn::audio::FinalMixType finalMix;
    NN_ABORT_UNLESS(nn::audio::AcquireFinalMix(&config, &finalMix, 2));
    nn::audio::DeviceSinkType deviceSink;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::AddDeviceSink(&config, &deviceSink, &finalMix, mainBus, channelCount, "MainAudioOut"));

    for (int i = 0; i < MaxAudioRendererSessions; i++)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(handle[i], &config));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::StartAudioRenderer(handle[i]));
    }

    // Setup AudioIn and AudioOut
    nn::audio::SetAudioInEnabled(true);

    nn::audio::AudioInParameter audioInParameter;
    nn::audio::AudioOutParameter audioOutParameter;

    nn::audio::AudioIn audioIn[MaxAudioInSessions];
    nn::audio::AudioOut audioOut[MaxAudioOutSessions];
    nn::audio::AudioInInfo audioInInfos[4];
    nn::audio::ListAudioIns(&audioInInfos[0], 4);
    nn::os::SystemEvent audioInSystemEvent[MaxAudioInSessions];
    nn::os::SystemEvent audioOutSystemEvent[MaxAudioOutSessions];

    nn::audio::InitializeAudioInParameter(&audioInParameter);
    nn::audio::InitializeAudioOutParameter(&audioOutParameter);

    // Find a working session, then open multiple times
    int session;
    for (session = 0; session < 4; session++)
    {
        if (nn::audio::OpenAudioIn(&audioIn[0], &audioInSystemEvent[0], audioInInfos[session].name, audioInParameter).IsSuccess())
        {
            NN_ABORT_UNLESS(
                nn::audio::StartAudioIn(&audioIn[0]).IsSuccess(),
                "Failed to start recording."
            );
            break;
        }
    }
    for (int i = 1; i < MaxAudioInSessions; i++)
    {
        NN_ABORT_UNLESS(
            nn::audio::OpenAudioIn(&audioIn[i], &audioInSystemEvent[i], audioInInfos[session].name, audioInParameter).IsSuccess(),
            "Failed to open AudioIn."
        );
        NN_ABORT_UNLESS(
            nn::audio::StartAudioIn(&audioIn[i]).IsSuccess(),
            "Failed to start recording."
        );
    }
    for (int i = 0; i < MaxAudioOutSessions; i++)
    {
        NN_ABORT_UNLESS(
            nn::audio::OpenDefaultAudioOut(&audioOut[i], &audioOutSystemEvent[i], audioOutParameter).IsSuccess(),
            "Failed to open AudioOut."
        );
        NN_ABORT_UNLESS(
            nn::audio::StartAudioOut(&audioOut[i]).IsSuccess(),
            "Failed to start playback."
        );

        if (MaxAudioInSessions == MaxAudioOutSessions)
        {
            CheckAudioInputOutputProperties(&audioIn[i], &audioOut[i]);
        }
    }

    // Setup FinalOutputRecorder
    nn::audio::FinalOutputRecorder gameRecord[MaxFinalOutputRecorderSessions];
    nn::os::SystemEvent gameRecordBufferEvent[MaxFinalOutputRecorderSessions];
    nn::audio::FinalOutputRecorderParameter paramGameRecord{ 0 };
    nn::audio::InitializeFinalOutputRecorderParameter(&paramGameRecord);

    for (int i = 0; i < MaxFinalOutputRecorderSessions; i++)
    {
        if (nn::audio::OpenFinalOutputRecorder(&gameRecord[i], &gameRecordBufferEvent[i], paramGameRecord).IsFailure())
        {
            paramGameRecord.sampleRate = 0;
            NN_ABORT_UNLESS(
                nn::audio::OpenFinalOutputRecorder(&gameRecord[i], &gameRecordBufferEvent[i], paramGameRecord).IsSuccess(),
                "Failed to open GameRecord."
            );
        }
        NN_ABORT_UNLESS(
            nn::audio::StartFinalOutputRecorder(&gameRecord[i]).IsSuccess(),
            "Failed to start recording."
        );
    }

    // Wait for a few seconds
    NN_LOG("Sessions all opened \n");
    const nn::TimeSpan interval(nn::TimeSpan::FromSeconds(5));
    nn::os::SleepThread(interval);

    // Stop FinalOutputRecorder Sessions
    for (int i = 0; i < MaxFinalOutputRecorderSessions; i++)
    {
        nn::audio::StopFinalOutputRecorder(&gameRecord[i]);
        nn::audio::CloseFinalOutputRecorder(&gameRecord[i]);
        nn::os::DestroySystemEvent(gameRecordBufferEvent[i].GetBase());
    }

    // Stop AudioIn Sessions
    for (int i = 0; i < MaxAudioInSessions; i++)
    {
        nn::audio::StopAudioIn(&audioIn[i]);
        nn::audio::CloseAudioIn(&audioIn[i]);
        nn::os::DestroySystemEvent(audioInSystemEvent[i].GetBase());
    }

    // Stop AudioOut Sessions
    for (int i = 0; i < MaxAudioOutSessions; i++)
    {
        nn::audio::StopAudioOut(&audioOut[i]);
        nn::audio::CloseAudioOut(&audioOut[i]);
        nn::os::DestroySystemEvent(audioOutSystemEvent[i].GetBase());
    }

    // Close Renderer Sessions
    for (int i = 0; i < MaxAudioRendererSessions; i++)
    {
        nn::audio::StopAudioRenderer(handle[i]);
        nn::audio::CloseAudioRenderer(handle[i]);
        nn::os::DestroySystemEvent(systemEvent[i].GetBase());
        if (workBuffer[i])
        {
            m_Allocator.Free(workBuffer[i]);
            workBuffer[i] = nullptr;
        }
    }

    if (configBuffer)
    {
        m_Allocator.Free(configBuffer);
        configBuffer = nullptr;
    }

    NN_LOG("Sessions all closed \n");
    m_Allocator.Finalize();
    m_WaveBufferAllocator.Finalize();
} // NOLINT(readability/fn_size)

TEST(OpenAndCloseMaxRendererSessions, Failure)
{
    m_Allocator.Initialize(m_WorkBuffer, sizeof(m_WorkBuffer));
    m_WaveBufferAllocator.Initialize(m_WaveBufferPoolMemory, sizeof(m_WaveBufferPoolMemory));

    // Setup AudioRenderer
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = RenderRate;
    parameter.sampleCount = RenderCount;
    parameter.mixBufferCount = 2;
    parameter.voiceCount = 1;
    parameter.subMixCount = 0;
    parameter.sinkCount = 1;
    parameter.effectCount = 0;
    parameter.performanceFrameCount = 0;

    const int channelCount = 2;
    int8_t mainBus[channelCount];
    mainBus[nn::audio::ChannelMapping_FrontLeft] = 0;
    mainBus[nn::audio::ChannelMapping_FrontRight] = 1;

    NN_ABORT_UNLESS(
        nn::audio::IsValidAudioRendererParameter(parameter),
        "Invalid AudioRendererParameter specified."
    );

    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer[FailureAudioRendererSessions];
    nn::os::SystemEvent systemEvent[FailureAudioRendererSessions];
    nn::audio::AudioRendererHandle handle[FailureAudioRendererSessions];
    for (int i = 0; i < FailureAudioRendererSessions; i++)
    {
        workBuffer[i] = m_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
        NN_ABORT_UNLESS_NOT_NULL(workBuffer[i]);

        if (FailureAudioRendererSessions - 1 == i)
        {
            EXPECT_DEATH_IF_SUPPORTED(
                nn::audio::OpenAudioRenderer(&handle[i], &systemEvent[i], parameter, workBuffer[i], workBufferSize).IsSuccess(),
                "Failed to open AudioRenderer"
            );
        }
        else {
            NN_ABORT_UNLESS(
                nn::audio::OpenAudioRenderer(&handle[i], &systemEvent[i], parameter, workBuffer[i], workBufferSize).IsSuccess(),
                "Failed to open AudioRenderer"
            );
        }
    }

    // Setup FinalMix and routing
    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
    void* configBuffer = m_Allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(configBuffer);
    nn::audio::AudioRendererConfig config;
    nn::audio::InitializeAudioRendererConfig(&config, parameter, configBuffer, configBufferSize);

    nn::audio::FinalMixType finalMix;
    NN_ABORT_UNLESS(nn::audio::AcquireFinalMix(&config, &finalMix, 2));
    nn::audio::DeviceSinkType deviceSink;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::AddDeviceSink(&config, &deviceSink, &finalMix, mainBus, channelCount, "MainAudioOut"));

    for (int i = 0; i < MaxAudioRendererSessions; i++)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(handle[i], &config));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::StartAudioRenderer(handle[i]));
    }

    // Close Renderer Sessions
    for (int i = 0; i < MaxAudioRendererSessions; i++)
    {
        nn::audio::StopAudioRenderer(handle[i]);
        nn::audio::CloseAudioRenderer(handle[i]);
        nn::os::DestroySystemEvent(systemEvent[i].GetBase());
        if (workBuffer[i])
        {
            m_Allocator.Free(workBuffer[i]);
            workBuffer[i] = nullptr;
        }
    }

    if (configBuffer)
    {
        m_Allocator.Free(configBuffer);
        configBuffer = nullptr;
    }

    m_Allocator.Finalize();
    m_WaveBufferAllocator.Finalize();
}

TEST(OpenAndCloseMaxInOutSessions, Failure)
{
    m_Allocator.Initialize(m_WorkBuffer, sizeof(m_WorkBuffer));
    m_WaveBufferAllocator.Initialize(m_WaveBufferPoolMemory, sizeof(m_WaveBufferPoolMemory));

    // Setup AudioIn and AudioOut
    nn::audio::SetAudioInEnabled(true);

    nn::audio::AudioInParameter audioInParameter;
    nn::audio::AudioOutParameter audioOutParameter;

    nn::audio::AudioIn audioIn[FailureAudioInSessions];
    nn::audio::AudioOut audioOut[FailureAudioOutSessions];
    nn::audio::AudioInInfo audioInInfos[4];
    nn::audio::ListAudioIns(&audioInInfos[0], 4);
    nn::os::SystemEvent audioInSystemEvent[FailureAudioInSessions];
    nn::os::SystemEvent audioOutSystemEvent[FailureAudioOutSessions];

    nn::audio::InitializeAudioInParameter(&audioInParameter);
    nn::audio::InitializeAudioOutParameter(&audioOutParameter);

    // Find a working session, then open multiple times
    int session;
    for (session = 0; session < 4; session++)
    {
        if (nn::audio::OpenAudioIn(&audioIn[0], &audioInSystemEvent[0], audioInInfos[session].name, audioInParameter).IsSuccess())
        {
            NN_ABORT_UNLESS(
                nn::audio::StartAudioIn(&audioIn[0]).IsSuccess(),
                "Failed to start recording."
            );
            break;
        }
    }
    for (int i = 1; i < FailureAudioInSessions; i++)
    {
        if (FailureAudioInSessions - 1 == i)
        {
            EXPECT_DEATH_IF_SUPPORTED(
                nn::audio::OpenAudioIn(&audioIn[i], &audioInSystemEvent[i], audioInInfos[session].name, audioInParameter).IsSuccess(),
                "Failed to open AudioIn."
            );
            EXPECT_DEATH_IF_SUPPORTED(
                nn::audio::StartAudioIn(&audioIn[i]).IsSuccess(),
                "Failed to start recording."
            );
        }
        else {
            NN_ABORT_UNLESS(
                nn::audio::OpenAudioIn(&audioIn[i], &audioInSystemEvent[i], audioInInfos[session].name, audioInParameter).IsSuccess(),
                "Failed to open AudioIn."
            );
            NN_ABORT_UNLESS(
                nn::audio::StartAudioIn(&audioIn[i]).IsSuccess(),
                "Failed to start recording."
            );
        }
    }
    for (int i = 0; i < FailureAudioOutSessions; i++)
    {
        if (FailureAudioOutSessions - 1 == i)
        {
            EXPECT_DEATH_IF_SUPPORTED(
                nn::audio::OpenDefaultAudioOut(&audioOut[i], &audioOutSystemEvent[i], audioOutParameter).IsSuccess(),
                "Failed to open AudioOut."
            );
            EXPECT_DEATH_IF_SUPPORTED(
                nn::audio::StartAudioOut(&audioOut[i]).IsSuccess(),
                "Failed to start playback."
            );
        }
        else
        {
            NN_ABORT_UNLESS(
                nn::audio::OpenDefaultAudioOut(&audioOut[i], &audioOutSystemEvent[i], audioOutParameter).IsSuccess(),
                "Failed to open AudioOut."
            );
            NN_ABORT_UNLESS(
                nn::audio::StartAudioOut(&audioOut[i]).IsSuccess(),
                "Failed to start playback."
            );

            if (MaxAudioInSessions == MaxAudioOutSessions)
            {
                CheckAudioInputOutputProperties(&audioIn[i], &audioOut[i]);
            }
        }
    }

    // Stop AudioIn Sessions
    for (int i = 0; i < MaxAudioInSessions; i++)
    {
        nn::audio::StopAudioIn(&audioIn[i]);
        nn::audio::CloseAudioIn(&audioIn[i]);
        nn::os::DestroySystemEvent(audioInSystemEvent[i].GetBase());
    }

    // Stop AudioOut Sessions
    for (int i = 0; i < MaxAudioOutSessions; i++)
    {
        nn::audio::StopAudioOut(&audioOut[i]);
        nn::audio::CloseAudioOut(&audioOut[i]);
        nn::os::DestroySystemEvent(audioOutSystemEvent[i].GetBase());
    }

    m_Allocator.Finalize();
    m_WaveBufferAllocator.Finalize();
}

TEST(OpenAndCloseMaxRecorderSessions, Failure)
{
    m_Allocator.Initialize(m_WorkBuffer, sizeof(m_WorkBuffer));
    m_WaveBufferAllocator.Initialize(m_WaveBufferPoolMemory, sizeof(m_WaveBufferPoolMemory));

    // Setup FinalOutputRecorder
    nn::audio::FinalOutputRecorder gameRecord[FailureFinalOutputRecorderSessions];
    nn::os::SystemEvent gameRecordBufferEvent[FailureFinalOutputRecorderSessions];
    nn::audio::FinalOutputRecorderParameter paramGameRecord{ 0 };
    nn::audio::InitializeFinalOutputRecorderParameter(&paramGameRecord);

    for (int i = 0; i < FailureFinalOutputRecorderSessions; i++)
    {
        if (FailureFinalOutputRecorderSessions - 1 == i)
        {
            EXPECT_DEATH_IF_SUPPORTED(
                nn::audio::OpenFinalOutputRecorder(&gameRecord[i], &gameRecordBufferEvent[i], paramGameRecord).IsSuccess(),
                "Failed to open GameRecord."
            );
            EXPECT_DEATH_IF_SUPPORTED(
                nn::audio::StartFinalOutputRecorder(&gameRecord[i]).IsSuccess(),
                "Failed to start recording."
            );
        }
        else
        {
            if (nn::audio::OpenFinalOutputRecorder(&gameRecord[i], &gameRecordBufferEvent[i], paramGameRecord).IsFailure())
            {
                paramGameRecord.sampleRate = 0;
                NN_ABORT_UNLESS(
                    nn::audio::OpenFinalOutputRecorder(&gameRecord[i], &gameRecordBufferEvent[i], paramGameRecord).IsSuccess(),
                    "Failed to open GameRecord."
                );
            }
            NN_ABORT_UNLESS(
                nn::audio::StartFinalOutputRecorder(&gameRecord[i]).IsSuccess(),
                "Failed to start recording."
            );
        }
    }

    // Stop FinalOutputRecorder Sessions
    for (int i = 0; i < MaxFinalOutputRecorderSessions; i++)
    {
        nn::audio::StopFinalOutputRecorder(&gameRecord[i]);
        nn::audio::CloseFinalOutputRecorder(&gameRecord[i]);
        nn::os::DestroySystemEvent(gameRecordBufferEvent[i].GetBase());
    }

    m_Allocator.Finalize();
    m_WaveBufferAllocator.Finalize();
}
