﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/audio.h>
#include <nn/mem.h>

// SimpleAudioRenderer supports:
//   - 2ch FinalMix
//   - single 2ch SubMix
//   - single PcmInt16 2ch Voice
//
// Topology:
//   [Voice] -> [SubMix] -> [FinalMix] ---> [CircularBufferSink]
//                                      |-> [DeviceSink]
class SimpleAudioRenderer
{
public:
    SimpleAudioRenderer(void* buffer, std::size_t size, const nn::audio::AudioRendererParameter* pParameter = nullptr, int voiceChannelCount = 2) NN_NOEXCEPT
    {
        m_Allocator.Initialize(buffer, size);

        nn::audio::AudioRendererParameter parameter;
        nn::audio::InitializeAudioRendererParameter(&parameter);
        parameter.voiceCount = voiceChannelCount;
        parameter.mixBufferCount = ChannelCount + ChannelCount;  // FinalMix + SubMix
        parameter.sinkCount = 2;  // DeviceSink + CircularBufferSink
        parameter.subMixCount = 1;
        if (!pParameter)
        {
            pParameter = &parameter;
        }

        auto workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(*pParameter);
        m_WorkBuffer = m_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&m_Handle, &m_SystemEvent, *pParameter, m_WorkBuffer, workBufferSize));

        auto configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(*pParameter);
        m_ConfigBuffer = m_Allocator.Allocate(configBufferSize);
        nn::audio::InitializeAudioRendererConfig(&m_Config, *pParameter, m_ConfigBuffer, configBufferSize);

        const int8_t ChannelIndex[ChannelCount] = { 0, 1, };

        NN_ABORT_UNLESS(nn::audio::AcquireFinalMix(&m_Config, &m_FinalMix, ChannelCount));
        nn::audio::SetFinalMixVolume(&m_FinalMix, 1.0f);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::AddDeviceSink(&m_Config, &m_DeviceSink, &m_FinalMix, ChannelIndex, ChannelCount, "MainAudioOut"));

        NN_ABORT_UNLESS(nn::audio::AcquireSubMix(&m_Config, &m_SubMix, parameter.sampleRate, ChannelCount));
        nn::audio::SetSubMixDestination(&m_Config, &m_SubMix, &m_FinalMix);
        nn::audio::SetSubMixMixVolume(&m_SubMix, &m_FinalMix, 1.0f, 0, 0);
        nn::audio::SetSubMixMixVolume(&m_SubMix, &m_FinalMix, 1.0f, 1, 1);
        nn::audio::SetSubMixVolume(&m_SubMix, 1.0f);

        const int CircularBufferSinkFrameCount = 8;
        auto circularBufferSize = nn::audio::GetRequiredBufferSizeForCircularBufferSink(pParameter, ChannelCount, CircularBufferSinkFrameCount, nn::audio::SampleFormat_PcmInt16);
        m_CircularBufferSinkBuffer = m_Allocator.Allocate(nn::util::align_up(circularBufferSize, nn::audio::MemoryPoolType::SizeGranularity), nn::audio::MemoryPoolType::AddressAlignment);
        NN_ABORT_UNLESS_NOT_NULL(m_CircularBufferSinkBuffer);
        NN_ABORT_UNLESS(nn::audio::AcquireMemoryPool(&m_Config, &m_CircularBufferSinkMemoryPool, m_CircularBufferSinkBuffer, nn::util::align_up(circularBufferSize, nn::audio::MemoryPoolType::SizeGranularity)));
        NN_ABORT_UNLESS(nn::audio::RequestAttachMemoryPool(&m_CircularBufferSinkMemoryPool));

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::AddCircularBufferSink(&m_Config, &m_CircularBufferSink, &m_FinalMix, ChannelIndex, ChannelCount, m_CircularBufferSinkBuffer, circularBufferSize, nn::audio::SampleFormat_PcmInt16));

        NN_ABORT_UNLESS(nn::audio::AcquireVoiceSlot(&m_Config, &m_Voice, pParameter->sampleRate, voiceChannelCount, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
        nn::audio::SetVoiceDestination(&m_Config, &m_Voice, &m_SubMix);
        nn::audio::SetVoiceVolume(&m_Voice, 1.0f);

        for(int sourceChannel = 0; sourceChannel < nn::audio::GetVoiceChannelCount(&m_Voice); ++sourceChannel)
        {
            for(int destinationChannel = 0; destinationChannel < ChannelCount; ++destinationChannel)
            {
                nn::audio::SetVoiceMixVolume(&m_Voice, &m_SubMix, 1.0f, sourceChannel, destinationChannel);
            }
        }
        nn::audio::SetVoicePlayState(&m_Voice, nn::audio::VoiceType::PlayState_Play);

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::StartAudioRenderer(m_Handle));
    }

    ~SimpleAudioRenderer() NN_NOEXCEPT
    {
        nn::audio::StopAudioRenderer(m_Handle);
        while (nn::audio::WaveBuffer* p = const_cast<nn::audio::WaveBuffer*>(nn::audio::GetReleasedWaveBuffer(&m_Voice)))
        {
            m_Allocator.Free(reinterpret_cast<void*>(p));
        }
        nn::audio::CloseAudioRenderer(m_Handle);

        m_Allocator.Free(m_CircularBufferSinkBuffer);
        m_Allocator.Free(m_ConfigBuffer);
        m_Allocator.Free(m_WorkBuffer);

        m_Allocator.Finalize();
    }

    nn::audio::AudioRendererHandle GetHandle() const NN_NOEXCEPT
    {
        return m_Handle;
    }

    nn::audio::AudioRendererConfig& GetConfig() NN_NOEXCEPT
    {
        return m_Config;
    }

    void Wait() NN_NOEXCEPT
    {
        m_SystemEvent.Wait();
    }

    void Update() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(m_Handle, &m_Config));
        while (nn::audio::WaveBuffer* p = const_cast<nn::audio::WaveBuffer*>(nn::audio::GetReleasedWaveBuffer(&m_Voice)))
        {
            m_Allocator.Free(reinterpret_cast<void*>(p));
        }
    }

    std::size_t ReadFinalMixOut(void* buffer, std::size_t size) NN_NOEXCEPT
    {
        return nn::audio::ReadCircularBufferSink(&m_CircularBufferSink, buffer, size);
    }

    void AppendPcm16Waveform(const void* buffer, std::size_t size) NN_NOEXCEPT
    {
        nn::audio::WaveBuffer* pWaveBuffer = static_cast<nn::audio::WaveBuffer*>(m_Allocator.Allocate(sizeof(nn::audio::WaveBuffer)));
        pWaveBuffer->buffer = buffer;
        pWaveBuffer->size = size;
        pWaveBuffer->startSampleOffset = 0;
        pWaveBuffer->endSampleOffset = static_cast<int>(size / sizeof(int16_t)) / nn::audio::GetVoiceChannelCount(&m_Voice);
        pWaveBuffer->loop = 0;
        pWaveBuffer->isEndOfStream = false;

        NN_ABORT_UNLESS(nn::audio::AppendWaveBuffer(&m_Voice, pWaveBuffer));
    }

    void SetFinalMixVolume(float volume) NN_NOEXCEPT
    {
        nn::audio::SetFinalMixVolume(&m_FinalMix, volume);
    }

    void SetSubMixVolume(float volume) NN_NOEXCEPT
    {
        nn::audio::SetSubMixVolume(&m_SubMix, volume);
    }

    void SetVoiceVolume(float volume) NN_NOEXCEPT
    {
        nn::audio::SetVoiceVolume(&m_Voice, volume);
    }

    void SetVoiceMixVolume(float volume) NN_NOEXCEPT
    {
        for(int sourceChannel = 0; sourceChannel < nn::audio::GetVoiceChannelCount(&m_Voice); ++sourceChannel)
        {
            for(int destinationChannel = 0; destinationChannel < ChannelCount; ++destinationChannel)
            {
                nn::audio::SetVoiceMixVolume(&m_Voice, &m_SubMix, volume, sourceChannel, destinationChannel);
            }
        }
    }

private:
    static const int ChannelCount = 2;

    nn::mem::StandardAllocator m_Allocator;
    nn::os::SystemEvent m_SystemEvent;

    void* m_WorkBuffer;
    void* m_ConfigBuffer;
    void* m_CircularBufferSinkBuffer;
    nn::audio::MemoryPoolType m_CircularBufferSinkMemoryPool;
    nn::audio::AudioRendererHandle m_Handle;
    nn::audio::AudioRendererConfig m_Config;
    nn::audio::DeviceSinkType m_DeviceSink;
    nn::audio::CircularBufferSinkType m_CircularBufferSink;
    nn::audio::FinalMixType m_FinalMix;
    nn::audio::SubMixType m_SubMix;
    nn::audio::VoiceType m_Voice;
};
